package com.elitesland.scp.application.service.strategy;

import com.elitesland.scp.application.enums.AllocRuleType;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderComputeVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

@Slf4j
@Component
@RequiredArgsConstructor
public class PriorityEventService implements EventStrategy {

    /**
     * 根据仓库的可用数量，将需求量分配给每个门店。
     *
     * @param stores    门店对象的列表
     * @param warehouse 可用仓库数量
     */
    @Override
    public void handleInvStk(List<ScpDemandOrderComputeVO> stores, BigDecimal warehouse) {
        // 计算所有门店的需求总和
        BigDecimal sumDemandQty = stores.stream().map(row -> row.getAllocationDeQuantity()).reduce(BigDecimal::add).get();

        // 如果仓库的数量大于或等于总需求，那么每个门店都能得到其全部需求
        if (warehouse.compareTo(sumDemandQty) >= 0) {
            stores.forEach(store -> store.setPlanQuantity(store.getAllocationDeQuantity()));
            return;
        }

        // 计算需求没有完全满足的门店的数量
        long remindCount = stores.stream().filter(store -> store.getAllocationDeQuantity().compareTo(store.getPlanQuantity()) > 0).count();

        // 继续分配轮次，直到仓库为空或所有门店的需求都得到满足
        while (warehouse.compareTo(BigDecimal.ZERO) > 0 && remindCount > 0) {
            // 计算这一轮每个门店要分配的平均数量
            BigDecimal allocation = warehouse.divide(BigDecimal.valueOf(remindCount), 0, RoundingMode.DOWN);
            log.info("这一轮每个门店要分配的平均数量：{}", allocation);
            // 如果仓库的数量小于最小的剩余需求，那么将仓库剩余的数量分配给还有剩余需求的门店
            if (allocation.compareTo(BigDecimal.ONE) <= 0) {
                ScpDemandOrderComputeVO remindStore = stores.stream()
                        .filter(store -> store.getAllocationDeQuantity().compareTo(store.getPlanQuantity()) > 0)
                        .findFirst()
                        .orElse(null);
                if (remindStore != null) {
                    remindStore.setPlanQuantity(remindStore.getPlanQuantity().add(warehouse));
                }
                return;
            }

            // 标记这一轮的分配是否有变化
            boolean allocationChanged = false;

            // 给每个门店分配计算出的数量
            for (ScpDemandOrderComputeVO store : stores) {
                if (store.getPlanQuantity().compareTo(store.getAllocationDeQuantity()) < 0 && warehouse.compareTo(BigDecimal.ZERO) > 0) {
                    BigDecimal actualAllocation = allocation.min(store.getAllocationDeQuantity().subtract(store.getPlanQuantity()));
                    if (actualAllocation.compareTo(BigDecimal.ZERO) > 0) {
                        allocationChanged = true;
                        store.setPlanQuantity(store.getPlanQuantity().add(actualAllocation));
                        warehouse = warehouse.subtract(actualAllocation);

                        // 如果一个门店的需求已经完全满足，那么减少剩余需求的门店的数量
                        if (store.getAllocationDeQuantity().compareTo(store.getPlanQuantity()) == 0) {
                            remindCount--;
                        }
                    }
                }
            }
            // 如果这一轮的分配有变化，那么计算下一轮的最小需求
            if (!allocationChanged) {
                break;
            }
        }
    }

    @Override
    public String getAllocRuleType() {
        return AllocRuleType.PRIORITY.getCode();
    }

}
