package com.elitesland.scp.domain.service.alloc;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingStorePageParamVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingStoreParamVO;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingStoreRespVO;
import com.elitesland.scp.application.facade.vo.save.alloc.ScpAllocSettingStoreSaveVO;
import com.elitesland.scp.domain.convert.alloc.ScpAllocSettingStoreConvert;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingDO;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingStoreDO;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingRepo;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingStoreRepo;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingStoreRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import java.util.*;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpAllocSettingStoreDomainServiceImpl implements ScpAllocSettingStoreDomainService {
    private final EntityManager entityManager;
    private final ScpAllocSettingStoreRepo scpAllocSettingStoreRepo;
    private final ScpAllocSettingRepo scpAllocSettingRepo;
    private final ScpAllocSettingStoreRepoProc scpAllocSettingStoreRepoProc;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;

    @Override
    @SysCodeProc
    public PagingVO<ScpAllocSettingStoreRespVO> page(ScpAllocSettingStorePageParamVO paramVO) {
        long count = scpAllocSettingStoreRepoProc.countAllocSettingStore(paramVO);
        if (count > 0) {
            List<ScpAllocSettingStoreRespVO> storeRespVOList = scpAllocSettingStoreRepoProc.queryAllocSettingStore(paramVO);
            return new PagingVO<>(count, storeRespVOList);
        }
        return new PagingVO<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveAllocSettingStore(List<ScpAllocSettingStoreSaveVO> saveVOS, int batchSize) {
        this.batchInsert(ScpAllocSettingStoreConvert.INSTANCE.saveVosDOS(saveVOS), batchSize);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByMasId(Long masId) {
        scpAllocSettingStoreRepo.deleteByMasId(masId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpAllocSettingStoreRepo.deleteByIds(ids);
    }

    @Override
    public List<Long> findMasIdByStoreCode(List<String> storeCodes, Set<Long> masIds) {
        return scpAllocSettingStoreRepoProc.findMasIdByStoreCodeIn(storeCodes, masIds);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateAllocNumByParam(List<Long> masIds, String storeCode) {
        scpAllocSettingStoreRepo.updateAllocNumByParam(masIds, storeCode);
    }

    @Override
    public List<ScpAllocSettingStoreRespVO> findByParam(ScpAllocSettingStoreParamVO paramVO) {
        return scpAllocSettingStoreRepoProc.findByParam(paramVO);
    }

    @Override
    public Long updateAllocSettingStore(ScpAllocSettingStoreSaveVO saveVO) {
        Optional<ScpAllocSettingStoreDO> option = scpAllocSettingStoreRepo.findById(saveVO.getId());
        if (option.isEmpty()) {
            throw new BusinessException("订货强配门店ID：" + saveVO.getId() + "不存在");
        }
        ScpAllocSettingStoreDO scpAllocSettingDO = option.get();
        ScpAllocSettingStoreConvert.INSTANCE.copySaveParamToDo(saveVO, scpAllocSettingDO);
        return scpAllocSettingStoreRepo.save(scpAllocSettingDO).getId();
    }

    @Override
    @SysCodeProc
    public List<ScpAllocSettingStoreRespVO> findByMasId(Long masId) {
        ScpAllocSettingDO activityDO = scpAllocSettingRepo.findById(masId).orElseThrow(() -> new BusinessException("订货强配ID：" + masId + "不存在"));
        List<ScpAllocSettingStoreDO> settingStoreDOS = scpAllocSettingStoreRepo.findByMasId(masId);
        List<String> storeCodeList = settingStoreDOS.stream().map(ScpAllocSettingStoreDO::getStoreCode).toList();
        Map<String, Long> stringLongMap = scpDemandOrderRepoProc.countAllocatedOrderByStoreCodeList(storeCodeList, activityDO.getActivityCode(), activityDO.getValidFrom(), activityDO.getValidTo());
        if (CollUtil.isNotEmpty(settingStoreDOS)) {
            List<ScpAllocSettingStoreRespVO> result = new ArrayList<>();
            settingStoreDOS.forEach(settingStoreDO -> {
                ScpAllocSettingStoreRespVO scpAllocSettingStoreRespVO = ScpAllocSettingStoreConvert.INSTANCE.doToRespVO(settingStoreDO);
                Long allocNum = stringLongMap.get(settingStoreDO.getStoreCode());
                scpAllocSettingStoreRespVO.setAllocNum(allocNum == null ? 0 : allocNum.intValue());
                result.add(scpAllocSettingStoreRespVO);
            });
            return result;
        }
        return new ArrayList<>();
    }

    @Override
    public List<ScpAllocSettingStoreRespVO> findEnabledByParam(ScpAllocSettingStoreParamVO paramVO) {
        return scpAllocSettingStoreRepoProc.findEnabledByParam(paramVO);
    }

    public void batchInsert(List<ScpAllocSettingStoreDO> dataList, int batchSize) {
        int index = 0;
        for (ScpAllocSettingStoreDO data : dataList) {
            entityManager.persist(data);
            if (batchSize > 1) {
                // 开启批量
                index++;
                if (index % batchSize == 0) {
                    entityManager.flush();
                    entityManager.clear();
                }
            }
        }
        if (!dataList.isEmpty()) {
            entityManager.flush();
        }
    }
}
