package com.elitesland.scp.infr.repo.msg;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitesland.scp.domain.entity.msg.QScpMessageDO;
import com.elitesland.scp.domain.entity.msg.QScpMessageReceiverDO;
import com.elitesland.scp.domain.entity.msg.ScpMessageReceiverDO;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/28
 */
@Repository
public class ScpMessageReceiverRepoProc extends BaseRepoProc<ScpMessageReceiverDO> {
    private static final QScpMessageReceiverDO QDO = QScpMessageReceiverDO.scpMessageReceiverDO;
    private static final QScpMessageDO QDO_MSG = QScpMessageDO.scpMessageDO;

    public ScpMessageReceiverRepoProc() {
        super(QDO);
    }

    /**
     * 根据消息ID删除
     *
     * @param messageId
     */
    public void deleteByMessageId(long messageId) {
        super.deleteByValue(QDO.messageId, messageId);
    }

    public void deleteByMessageId(Collection<Long> messageIds) {
        super.deleteByValue(QDO.messageId, messageIds);
    }

    /**
     * 更新已读标识
     *
     * @param messageId
     * @param receiverType
     * @param receiver
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateViewed(long messageId, @NotBlank String receiverType, @NotBlank String receiver) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.viewed, true)
                .set(QDO.viewTime, LocalDateTime.now())
                .where(QDO.messageId.eq(messageId).and(QDO.receiverId.eq(receiver)).and(QDO.receiverType.eq(receiverType)).and(QDO.viewed.eq(false)))
                .execute();
    }

    /**
     * 删除消息接收者
     *
     * @param messageId
     * @param receiverType
     * @param receivers
     */
    public void deleteReceivers(long messageId, String receiverType, Collection<String> receivers) {
        super.delete(QDO.messageId.eq(messageId).and(QDO.receiverId.in(receivers)).and(QDO.receiverType.eq(receiverType)));
    }

    /**
     * 获取消息接收者
     *
     * @param messageId
     * @param receiverType
     * @return
     */
    public List<String> getReceivers(long messageId, @NotBlank String receiverType) {
        return super.jpaQueryFactory.select(QDO.receiverId)
                .from(QDO)
                .where(QDO.messageId.eq(messageId).and(QDO.receiverType.eq(receiverType)))
                .fetch();
    }

    public Map<Long, List<IdCodeNameParam>> getReceivers(Collection<Long> messageIds) {
        return super.jpaQueryFactory.select(QDO.messageId, QDO.receiverType, QDO.receiverId)
                .from(QDO)
                .where(QDO.messageId.in(messageIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.messageId),
                        Collectors.mapping(t -> new IdCodeNameParam(null, t.get(QDO.receiverId), t.get(QDO.receiverType)), Collectors.toList())));
    }

    /**
     * 统计未读的消息
     *
     * @param receiverType
     * @param receiver
     * @return
     */
    public Long countUnView(@NotBlank String receiverType, @NotBlank String receiver) {
        return jpaQueryFactory.select(QDO_MSG.count())
                .from(QDO_MSG)
                .where(QDO_MSG.id.in(
                        JPAExpressions.select(QDO.messageId)
                                .from(QDO)
                                .where(QDO.receiverId.eq(receiver).and(QDO.receiverType.eq(receiverType)).and(QDO.viewed.eq(false)))
                ).and(QDO_MSG.published.eq(true)
                        .and(QDO_MSG.publishTime.before(LocalDateTime.now())))
                                .and(QDO_MSG.invalidTime.isNull().or(QDO_MSG.invalidTime.after(LocalDateTime.now())))
                )
                .fetchOne();
    }

    /**
     * 统计未读的消息
     *
     * @param receiverType
     * @param receiver
     * @return
     */
    public Map<Long, Boolean> queryViewed(@NotEmpty Collection<Long> messageIds, @NotBlank String receiverType, @NotBlank String receiver) {
        return jpaQueryFactory.select(QDO.messageId, QDO.viewed)
                .from(QDO)
                .where(QDO.receiverId.eq(receiver).and(QDO.messageId.in(messageIds)).and(QDO.receiverType.eq(receiverType)))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.messageId), t -> t.get(QDO.viewed), (t1, t2) -> t1));
    }
}
