package com.elitesland.scp.application.service.stock;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.inv.provider.InvWhProvider;
import com.elitesland.scp.application.facade.vo.stock.ScpPredictStStockImportVO;
import com.elitesland.scp.application.facade.vo.stock.ScpSafetyTargetStockImportEntity;
import com.elitesland.scp.domain.entity.stock.ScpPredictStStockDO;
import com.elitesland.scp.domain.service.stock.ScpSafetyTargetStockDomainService;
import com.elitesland.scp.infr.repo.stock.ScpPredictStStockRepo;
import com.elitesland.scp.infr.repo.stock.ScpPredictStStockRepoProc;
import com.elitesland.scp.infr.repo.stock.ScpSafetyTargetStockRepo;
import com.elitesland.support.provider.item.dto.ItmItemSkuBusinessRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemSkuBusinessRpcDtoParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

/**
 * @author jeesie
 * @description:
 * @datetime 2025年 02月 08日 16:49
 * @version: 1.0
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class ScpPredictStStockImportServiceImpl implements DataImport<ScpPredictStStockImportVO> {

    private final InvWhProvider invWhProvider;
    private final ItmItemRpcService itemRpcService;
    private final OrgOuRpcService orgOuRpcService;

    private final ScpPredictStStockRepo scpPredictStStockRepo;

    @Override
    public String getTmplCode() {
        return "predict_st_stock_import";
    }

    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public List<String> executeImport(List<ScpPredictStStockImportVO> dataList, int startRowIndex){
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        Map<Integer, List<String>> errorMap = new HashMap<>();
        List<String> errorList = new ArrayList<>();
        HashMap<String, Integer> uniqueMap = new HashMap<>();
        for (int i = 0; i < dataList.size(); i++) {
            ScpPredictStStockImportVO importVO = dataList.get(i);
            importVO.setLine(i + 1);
            errorMap.put(importVO.getLine(), new ArrayList<>());
            if(importVO.getStIdStr() == null){
                errorMap.get(importVO.getLine()).add("列[预测批次ID]不能为空");
            }else{
                try {
                    importVO.setStId(Long.parseLong(importVO.getStIdStr()));
                } catch (Exception e) {
                    errorMap.get(importVO.getLine()).add("列[预测批次ID]格式不正确");
                }
            }
            if (!StringUtils.hasText(importVO.getOuCode())) {
                errorMap.get(importVO.getLine()).add("列[公司编码]不能为空");
            }
            if (!StringUtils.hasText(importVO.getWhCode())) {
                errorMap.get(importVO.getLine()).add("列[仓库编码]不能为空");
            }
            if (!StringUtils.hasText(importVO.getItemCode())) {
                errorMap.get(importVO.getLine()).add("列[商品编码]不能为空");
            }
            if(!StringUtils.hasText(importVO.getPlanUom())){
                errorMap.get(importVO.getLine()).add("列[计划单位]不能为空");
            }
            String uniqueKey = importVO.getStId().toString() +
                    importVO.getOuCode() + importVO.getWhCode() + importVO.getItemCode();
            Integer ex = uniqueMap.get(uniqueKey);
            if(ex != null){
                errorMap.get(importVO.getLine()).add("同批次，列[公司编码]-[仓库编码]-[商品编码] 已存在，不能重复,请检查");
                errorMap.get(ex).add("同批次，列[公司编码]-[仓库编码]-[商品编码] 已存在,不能重复，请检查");
            }else{
                uniqueMap.put(uniqueKey,i);
            }
            if (importVO.getSafetyQty() == null) {
                errorMap.get(importVO.getLine()).add("列[安全库存]不能为空");
            }
            if (importVO.getTargetQty() == null) {
                errorMap.get(importVO.getLine()).add("列[目标库存]不能为空");
            }
        }
        for (Integer line : errorMap.keySet()) {
            errorList.add(String.join(",", errorMap.get(line)));
        }
        if(Boolean.TRUE.equals(getErrorResult(errorList))){
            return errorList;
        }
        List<Long> stIds = dataList.stream().map(ScpPredictStStockImportVO::getStId)
                .distinct().collect(Collectors.toList());
        List<String> itemCodes = dataList.stream().map(ScpPredictStStockImportVO::getItemCode)
                .distinct().collect(Collectors.toList());
        List<String> ouCodes = dataList.stream().map(ScpPredictStStockImportVO::getOuCode)
                .distinct().collect(Collectors.toList());
        List<String> whCodes = dataList.stream().map(ScpPredictStStockImportVO::getWhCode)
                .distinct().collect(Collectors.toList());
        ItmItemSkuBusinessRpcDtoParam itmItemRpcDtoParam = new ItmItemSkuBusinessRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemSkuBusinessRpcDTO> itemInfos = itemRpcService.findItemSkuBusinessByParam(itmItemRpcDtoParam);
        Map<String, ItmItemSkuBusinessRpcDTO> itemMap = itemInfos.stream().collect(Collectors.toMap(ItmItemSkuBusinessRpcDTO::getItemCode, i -> i, (o, n) -> n));
        List<InvWhRpcSimpleDTO> invWhRpcSimpleDTOS = invWhProvider.findSimpleWhByWhCodes(whCodes).computeData();
        Map<String,InvWhRpcSimpleDTO> whMap = invWhRpcSimpleDTOS.stream()
                .collect(Collectors.toMap(i -> i.getOuId() + i.getWhCode(), i -> i, (o, n) -> n));
        Map<String, InvWhRpcSimpleDTO> whCodeMap = invWhRpcSimpleDTOS.stream().collect(Collectors.toMap(InvWhRpcSimpleDTO::getWhCode, i -> i,
                (o, n) -> n));
        List<OrgOuRpcSimpleDTO> simpleByOuCodes = orgOuRpcService.findSimpleByOuCodes(ouCodes);
        Map<String, OrgOuRpcSimpleDTO> ouMap = simpleByOuCodes.stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getOuCode, i -> i, (o, n) -> n));
        log.info("仓库字典:{}", JSONUtil.toJsonStr(whCodeMap));
        List<ScpPredictStStockDO> predictStStockDOS = scpPredictStStockRepo.findAllByStIdIn(stIds);
        Map<String, List<ScpPredictStStockDO>> stStockMap = predictStStockDOS.stream()
                .collect(Collectors.groupingBy(d -> d.getStId().toString() + d.getOuCode() + d.getWhCode()
                + d.getItemCode()));
        List<ScpPredictStStockDO> updates = new ArrayList<>();
        for(ScpPredictStStockImportVO importVO:dataList){
            String key = importVO.getStId().toString() + importVO.getOuCode() + importVO.getWhCode() + importVO.getItemCode();
            if(stStockMap.get(key) == null){
                errorMap.get(importVO.getLine()).add("数据不存在");
            }else{
                List<ScpPredictStStockDO> stockDOList = stStockMap.get(key);
                if(stockDOList.size() != 1){
                    errorMap.get(importVO.getLine()).add("数据存在多条");
                }else{
                    ScpPredictStStockDO scpPredictStStockDO = stockDOList.get(0);
                    if(!Objects.equals(scpPredictStStockDO.getOuCode(), importVO.getOuCode())){
                        errorMap.get(importVO.getLine()).add("列[公司编码]不一致");
                    } else if (!Objects.equals(scpPredictStStockDO.getWhCode(),importVO.getWhCode())) {
                        errorMap.get(importVO.getLine()).add("列[仓库编码]不一致");
                    } else if (!Objects.equals(scpPredictStStockDO.getItemCode(),importVO.getItemCode())) {
                        errorMap.get(importVO.getLine()).add("列[商品编码]不一致");
                    } else if (!Objects.equals(scpPredictStStockDO.getSafetyQty(),importVO.getSafetyQty())) {
                        errorMap.get(importVO.getLine()).add("列[安全库存]不一致");
                    }
                    else if (!Objects.equals(scpPredictStStockDO.getTargetQty(),importVO.getTargetQty())) {
                        errorMap.get(importVO.getLine()).add("列[目标库存]不一致");
                    } else if (!Objects.equals(scpPredictStStockDO.getPlanUom(),importVO.getPlanUom())) {
                        errorMap.get(importVO.getLine()).add("列[计划单位]不一致");
                    }
                    if(ouMap.get(importVO.getOuCode()) == null){
                        errorMap.get(importVO.getLine()).add("列[公司编码]不存在");
                    }else{
                        if(whMap.get(ouMap.get(importVO.getOuCode()).getId() + importVO.getWhCode()) == null){
                            errorMap.get(importVO.getLine()).add(String.format("列【公司编码】不属于 仓库 %s", importVO.getWhCode()));
                        }
                    }
                    if(itemMap.get(importVO.getItemCode()) == null){
                        errorMap.get(importVO.getLine()).add("列[商品编码]不存在");
                    }
                    scpPredictStStockDO.setPredTargetQty(importVO.getPredTargetQty());
                    scpPredictStStockDO.setPredSafetyQty(importVO.getPredSafetyQty());
                    scpPredictStStockDO.setModifyTime(LocalDateTime.now());
                    updates.add(scpPredictStStockDO);
                }
            }
        }
        if(!CollectionUtils.isEmpty(updates)){
            List<ScpPredictStStockDO> stStockDOS = updates.stream().filter(d -> d.getBusinessId() != null).collect(Collectors.toList());
            scpPredictStStockRepo.saveAll(stStockDOS);
        }
        for (Integer line : errorMap.keySet()) {
            if (errorMap.get(line).isEmpty()) {
                continue;
            }
            if (!Objects.equals(errorList.get(line - 1), null)) {
                errorList.set(line - 1, errorList.get(line - 1) + String.join(",", errorMap.get(line)));
            }
        }
        return errorList;
    }

    @Nullable
    private Boolean getErrorResult(List<String> errorList) {
        Boolean flag = false;
        errorList.replaceAll(new UnaryOperator<String>() {
            @Override
            public String apply(String s) {
                if (!StringUtils.hasText(s)) {
                    return null;
                }
                return s;
            }
        });
        for (String s : errorList) {
            if (StringUtils.hasText(s)) {
                flag = true;
                break;
            }
        }
        return flag;
    }
}
