package com.elitesland.scp.application.service.calendar;

import com.el.coordinator.core.common.utils.BeanCopyUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.calendar.*;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.service.scpsman.BaseScpmanAuthorityParam;
import com.elitesland.scp.domain.convert.calendar.ScpSuppDemandCalendarConvert;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.domain.service.calendar.ScpSuppDemandCalendarDomainService;
import com.elitesland.scp.infr.dto.calendar.ScpSuppDemandCalendarDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author jeesie
 * @description:
 * @datetime 2024年 04月 01日 15:01
 * @version: 1.0
 */
@RequiredArgsConstructor
@Service
@Slf4j
public class ScpSuppDemandCalendarServiceImpl implements ScpSuppDemandCalendarService {

    private final ScpSuppDemandCalendarDomainService scpSuppCalendarDomainService;

    private final ScpDemandAuthorityService scpDemandAuthorityService;

    @Override
    public PagingVO<ScpSuppDemandCalendarPageVO> searchPage(ScpSuppDemandCalendarPageParamVO param) {
        if(Boolean.TRUE.equals(param.getScpmanAuthority())){
            PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
            if(userAuthority.isEmpty()){
                return PagingVO.<ScpSuppDemandCalendarPageVO>builder()
                        .total(0L)
                        .records(new ArrayList<>())
                        .build();
            }
            extractedAuthorityParam(param, userAuthority);
        }
        PagingVO<ScpSuppDemandCalendarPageVO> paging = scpSuppCalendarDomainService.searchPage(param);
        if(CollectionUtils.isNotEmpty(paging.getRecords())){
            List<String> businessKey = paging.getRecords().stream()
                    .map(this::getBusinessString)
                    .collect(Collectors.toList());
            List<ScpSuppDemandCalendarDTO> dtos = scpSuppCalendarDomainService.findCalendarListByConcatStr(businessKey);
            if(CollectionUtils.isNotEmpty(dtos)){
                var listMap = dtos.stream().collect(Collectors.groupingBy(v -> getString(v.getSuppCode(), v.getType(), v.getStoreWhCode(), v.getYear(), v.getMonth())));
                paging.getRecords().forEach(d ->{
                    String concatKey = getString(d.getSuppCode(), d.getType(), d.getStoreWhCode(), d.getYear(), d.getMonth());
                    List<ScpSuppDemandCalendarDTO> calendarDTOS = listMap.get(concatKey);
                    if(CollectionUtils.isNotEmpty(calendarDTOS)){
                        d.setId(calendarDTOS.get(0).getId());
                        var details = calendarDTOS.stream().map(m -> {
                            ScpSuppDemandCalendarPageVO.CalendarDayDetail detail = ScpSuppDemandCalendarConvert.INSTANCE.dtoToPageVo(m);
                            detail.setDay(m.getDay());
                            detail.setWorkStatus(m.getWorkStatus());
                            return detail;
                        }).collect(Collectors.toList());
                        var sortList = details.stream()
                                .filter(v -> v.getDay() != null)
                                .sorted(getComparing())
                                .collect(Collectors.toList());
                        d.setDayDetails(sortList);
                    }
                });
            }
        }
        return paging;
    }

    private void extractedAuthorityParam(BaseScpmanAuthorityParam queryParam, PagingVO<ScpManAuthorityPageRespVO> userAuthority) {
        List<String> whCodes = userAuthority.stream()
                .filter(v -> 1 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        List<String> stores = userAuthority.stream()
                .filter(v -> 0 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        queryParam.setWhCodes(whCodes);
        queryParam.setStores(stores);
        queryParam.setScpmanAuthority(Boolean.TRUE);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveSuppCalendar(ScpSuppDemandCalendarSaveVO createParam) {
        var dayDetails = createParam.getDetails();
        List<ScpSuppDemandCalendarDTO> calendarDTOList = dayDetails.stream().map(d -> {
            ScpSuppDemandCalendarDTO demandCalendarDTO = ScpSuppDemandCalendarConvert.INSTANCE.saveVoToDto(d);
            demandCalendarDTO.setType(createParam.getType());
            demandCalendarDTO.setSuppCode(createParam.getSuppCode());
            demandCalendarDTO.setSuppId(createParam.getSuppId());
            demandCalendarDTO.setSuppName(createParam.getSuppName());
            demandCalendarDTO.setStoreWhCode(createParam.getStoreWhCode());
            demandCalendarDTO.setStoreWhId(createParam.getStoreWhId());
            demandCalendarDTO.setStoreWhName(createParam.getStoreWhName());
            demandCalendarDTO.setYear(createParam.getYear());
            demandCalendarDTO.setMonth(createParam.getMonth());
            return demandCalendarDTO;
        }).collect(Collectors.toList());
        List<ScpSuppDemandCalendarDTO> saveList = calendarDTOList.stream().filter(d -> d.getId() == null)
                .collect(Collectors.toList());
        List<ScpSuppDemandCalendarDTO> updateList = calendarDTOList.stream().filter(d -> d.getId() != null)
                .collect(Collectors.toList());
        if(CollectionUtils.isNotEmpty(saveList)){
            List<String> concatKey = saveList.stream()
                    .map(d -> d.getSuppCode() +d.getType() + d.getStoreWhCode()+d.getYear()
                            + d.getMonth()
                            + d.getDay()
                    ).collect(Collectors.toList());
            List<ScpSuppDemandCalendarDTO> existList = scpSuppCalendarDomainService.findCalendarByStoreDayConcatStr(concatKey);
            if(CollectionUtils.isNotEmpty(existList)) {
                List<String> saveKey = saveList.stream()
                        .map(d -> d.getSuppCode() +d.getType() + d.getStoreWhCode()+d.getYear()
                                + d.getMonth()
                                + d.getDay()
                        ).collect(Collectors.toList());
                List<String> existKeys = existList.stream()
                        .filter(d -> saveKey.contains(d.getSuppCode() +d.getType() + d.getStoreWhCode()+d.getYear() + d.getMonth() + d.getDay()))
                        .map(d -> d.getSuppCode() +d.getType() + d.getStoreWhCode()+d.getYear() + d.getMonth() + d.getDay())
                        .collect(Collectors.toList());
                if(CollectionUtils.isNotEmpty(existKeys)){
                    throw new BusinessException(ApiCode.FAIL, String.join(",", existKeys) + "配置已存在，请修改");
                }
            }
            scpSuppCalendarDomainService.createBatch(saveList);
        }
        if(CollectionUtils.isNotEmpty(updateList)){
            scpSuppCalendarDomainService.createBatch(updateList);
        }
    }

    @Override
    public ScpSuppDemandCalendarRespVO findSuppCalendarListByParam(ScpSuppDemandCalendarParamVO param) {
        ScpSuppDemandCalendarRespVO vo = new ScpSuppDemandCalendarRespVO();
        List<ScpSuppDemandCalendarDTO> calendarDOS = scpSuppCalendarDomainService.findSuppCalendarListByParam(param);
        if(CollectionUtils.isEmpty(calendarDOS)){
            return vo;
        }
        if(param.getMonth() != null && param.getYear() != null){
            List<ScpSuppDemandCalendarRespVO> calendarRespVOS = calendarDOS.stream()
                    .map(ScpSuppDemandCalendarConvert.INSTANCE::dtoToRespVo)
                    .collect(Collectors.toList());
            ScpSuppDemandCalendarRespVO calendarRespVO = calendarRespVOS.get(0);
            BeanCopyUtil.beanCopyWithIngore(calendarRespVO,vo,BeanCopyUtil.getNullAndIgnorePropertyNames(calendarRespVO));
            var vos = calendarRespVOS.stream().filter(d -> d.getDay() != null)
                    .sorted(getComparing2()).collect(Collectors.toList());
            vo.setDetails(vos);
            List<String> years = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            List<String> days = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getDay).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
            vo.setExistDaysOfMonth(days);

        }else if( param.getYear() != null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
        }else if( param.getYear() == null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpSuppDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
        }
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSuppCalendar(List<ScpSuppDemandCalendarParamVO> param) {
        verifyParam(param);
        List<String> concatKey = param.stream().map(d -> d.getSuppCode() +d.getType() + d.getStoreWhCode()+d.getYear()
                + d.getMonth()
                ).collect(Collectors.toList());
        List<ScpSuppDemandCalendarDTO> dtos = scpSuppCalendarDomainService.findCalendarListByConcatStr(concatKey);
        if(CollectionUtils.isNotEmpty( dtos)){
            List<Long> ids =  dtos.stream().map(ScpSuppDemandCalendarDTO::getId).distinct()
                    .collect(Collectors.toList());
            scpSuppCalendarDomainService.deleteBatch(ids);
        }
    }

    private void verifyParam(List<ScpSuppDemandCalendarParamVO> param) {
        long nullcount = param.stream().filter(d -> d.getYear() == null).count();
        if(nullcount > 0){
            throw new BusinessException(ApiCode.FAIL,"年不能为空");
        }
        long nullcount2 = param.stream().filter(d -> d.getMonth() == null).count();
        if(nullcount2 > 0){
            throw new BusinessException(ApiCode.FAIL,"月不能为空");
        }
    }

    @NotNull
    private String getString(String suppCode, String type,
                             String storeWhCode,
                             String year,
                             String month) {
        return suppCode + type + storeWhCode + year + month;
    }

    @NotNull
    private String getBusinessString(ScpSuppDemandCalendarPageVO d) {
        return getString(d.getSuppCode(), d.getType(), d.getStoreWhCode(), d.getYear(), d.getMonth());
    }

    private Comparator<ScpSuppDemandCalendarPageVO.CalendarDayDetail> getComparing() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }

    private Comparator<ScpSuppDemandCalendarRespVO> getComparing2() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }
}
