package com.elitesland.scp.mq;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.boot.provider.IdFactory;
import com.elitesland.scp.boh.StoreReceiveSendParam;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDDO;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDO;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.infr.dto.boh.DemandOrderDtl;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveDRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 门店收货单发货回写
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class StoreReceiveDeliveryListener implements MessageQueueListener<StoreReceiveSendParam> {

    private final StoreReceiveRepo storeReceiveRepo;
    private final StoreReceiveRepoProc storeReceiveRepoProc;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final StoreReceiveDRepo storeReceiveDRepo;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{StoreReceiveSendParam.DELIVERY_CHANNEL};
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void onConsume(@NotBlank String s, @NotNull StoreReceiveSendParam storeReceiveSendParam) {
        log.info("门店收货单-生成待收货：" + JSON.toJSONString(storeReceiveSendParam));
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(storeReceiveSendParam.getDocId());
        // 操作门店退货出库需要更新状态
        if (CollUtil.isNotEmpty(receiveDOS)) {
            StoreReceiveDO thisDO = receiveDOS.get(0);
            if (!(Objects.equals(thisDO.getStatus(), "CE") && Objects.equals(thisDO.getRtType(), "T"))) {
                log.info("门店收货单状态不正确，不处理：{}", thisDO.getStatus());
                return;
            }
            thisDO.setStatus("CF");
            thisDO.setRtnDate(LocalDateTime.now());
            List<DemandOrderDtl> orderDtl = storeReceiveRepoProc.findOrderDtl(thisDO.getOrderId());
            Map<Long, BigDecimal> priceMap = orderDtl.stream().filter(i -> i.getPrice() != null).collect(Collectors.toMap(i -> i.getItemId(), i -> i.getPrice(), (o, n) -> n));
            BigDecimal realAmt = storeReceiveSendParam.getItems().stream().map(i -> {
                if (i.getQty() == null || i.getQty().compareTo(BigDecimal.ZERO) <= 0) {
                    return BigDecimal.ZERO;
                }
                if (!priceMap.containsKey(i.getItemId())) {
                    return BigDecimal.ZERO;
                }
                return priceMap.get(i.getItemId()).multiply(i.getQty());
            }).reduce(BigDecimal.ZERO, BigDecimal::add);
            thisDO.setRealAmt(realAmt);
            thisDO.setTotalRealQty(storeReceiveSendParam.getItems().stream().map(i -> i.getQty()).reduce(BigDecimal.ZERO, BigDecimal::add));
            storeReceiveRepo.save(thisDO);
        } else {
            List<DemandOrderSimpleDTO> orderSimple = storeReceiveRepoProc.findOrderSimple(List.of(storeReceiveSendParam.getOrderId()));
            List<DemandOrderSimpleDTO> stOrderList =
                    orderSimple.stream().filter(i -> Objects.equals(i.getType(), "0")).collect(Collectors.toList());
            if (!stOrderList.isEmpty()) {
                List<Long> orderDidList =
                        storeReceiveSendParam.getItems().stream().map(i -> i.getOrderDid()).distinct().collect(Collectors.toList());
                List<DemandOrderDtl> orderDtl = storeReceiveRepoProc.findOrderDtl(orderDidList);
                Map<Long, BigDecimal> priceMap =
                        orderDtl.stream().filter(i -> i.getPrice() != null).collect(Collectors.toMap(i -> i.getId(),
                                i -> i.getPrice()));

                DemandOrderSimpleDTO demandOrderSimpleDTO = stOrderList.get(0);
                StoreReceiveDO receiveDO = new StoreReceiveDO();
                receiveDO.setDocCreateDate(storeReceiveSendParam.getCreateTime());
                receiveDO.setStatus("CE");
                receiveDO.setDocId(storeReceiveSendParam.getDocId());
                receiveDO.setOrderDate(demandOrderSimpleDTO.getCreateTime());
                receiveDO.setDocNo(storeReceiveSendParam.getDocNo());
                receiveDO.setOrderId(storeReceiveSendParam.getOrderId());
                receiveDO.setOrderNo(storeReceiveSendParam.getOrderNo());
                receiveDO.setStoreId(demandOrderSimpleDTO.getStoreId());
                receiveDO.setOrderSetId(demandOrderSimpleDTO.getOrderSetId());
                receiveDO.setDocType(storeReceiveSendParam.getDocType());
                receiveDO.setTotalQty(storeReceiveSendParam.getTotalOutQty());
                receiveDO.setDeliveryDate(storeReceiveSendParam.getDeliveryDate());
                BigDecimal totalAmt = BigDecimal.ZERO;
                for (StoreReceiveSendParam.OrderItem item : storeReceiveSendParam.getItems()) {
                    if (item.getQty().compareTo(BigDecimal.ZERO) <= 0) {
                        continue;
                    }
                    BigDecimal price = priceMap.get(item.getOrderDid());
                    if (price == null) {
                        continue;
                    }
                    totalAmt = totalAmt.add(price.multiply(item.getQty()));
                }
                receiveDO.setTotalAmt(totalAmt);
                receiveDO.setRtType("R");
                StoreReceiveDO saved = storeReceiveRepo.save(receiveDO);

                List<StoreReceiveDDO> receiveDDOS = new ArrayList<>();
                for (StoreReceiveSendParam.OrderItem item : storeReceiveSendParam.getItems()) {
                    StoreReceiveDDO storeReceiveDDO = new StoreReceiveDDO();
                    storeReceiveDDO.setDid(item.getDid());
                    storeReceiveDDO.setId(IdFactory.generateLong());
                    storeReceiveDDO.setItemId(item.getItemId());
                    storeReceiveDDO.setMasId(saved.getId());
                    storeReceiveDDO.setQty(item.getQty());
                    storeReceiveDDO.setPrice(priceMap.get(item.getOrderDid()));
                    receiveDDOS.add(storeReceiveDDO);
                }
                storeReceiveDRepo.saveAll(receiveDDOS);
            }

            if (storeReceiveSendParam.getDocType().equals("GR")) {
                return;
            }

            //回写发货数量
            log.info("调拨单更新订货单发货数量：{}", JSONUtil.toJsonStr(storeReceiveSendParam.getItems()));
            Map<Long, BigDecimal> orderDidQtyMap = processQty(storeReceiveSendParam.getItems());
            log.info("调拨单更新发货数量字典数据：{}", JSONUtil.toJsonStr(orderDidQtyMap));
            for (Map.Entry<Long, BigDecimal> item : orderDidQtyMap.entrySet()) {
                Long did = item.getKey();
                BigDecimal qty = item.getValue();
                scpDemandOrderDDomainService.updateQuantity(did, qty);
            }
            scpDemandOrderDomainService.updateDocStatusById(storeReceiveSendParam.getOrderId(), ScpConstant.DELIVERY);
        }

    }

    public static Map<Long, BigDecimal> processQty(List<StoreReceiveSendParam.OrderItem> orderItems) {
        return orderItems.stream()
                .collect(Collectors.groupingBy(StoreReceiveSendParam.OrderItem::getOrderDid,
                        Collectors.mapping(StoreReceiveSendParam.OrderItem::getQty,
                                Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));
    }
}
