package com.elitesland.scp.application.service.stock;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.inv.provider.InvWhProvider;
import com.elitesland.scp.application.facade.vo.stock.ScpSafetyTargetStockDownLoadParam;
import com.elitesland.scp.application.facade.vo.stock.ScpSafetyTargetStockImportEntity;
import com.elitesland.scp.application.facade.vo.stock.ScpSafetyTargetStockSaveVO;
import com.elitesland.scp.domain.entity.stock.ScpSafetyTargetStockDO;
import com.elitesland.scp.domain.service.stock.ScpSafetyTargetStockDomainService;
import com.elitesland.scp.infr.dto.stock.ScpSafetyTargetStockDTO;
import com.elitesland.scp.infr.repo.stock.ScpSafetyTargetStockRepo;
import com.elitesland.support.provider.item.dto.ItmItemSkuBusinessRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemSkuBusinessRpcDtoParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

/**
 * @author jeesie
 * @description:
 * @datetime 2025年 01月 20日 10:10
 * @version: 1.0
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class ScpSafetyTargetStockImportService implements DataImport<ScpSafetyTargetStockImportEntity> {

    private final InvWhProvider invWhProvider;
    private final ItmItemRpcService itemRpcService;
    private final OrgOuRpcService orgOuRpcService;

    private final ScpSafetyTargetStockDomainService safetyTargetStockDomainService;

    private final ScpSafetyTargetStockRepo scpSafetyTargetStockRepo;


    @Override
    public String getTmplCode() {
        return "safety_target_stock_import";
    }

    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public List<String> executeImport(List<ScpSafetyTargetStockImportEntity> dataList, int startRowIndex){
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        Map<Integer, List<String>> errorMap = new HashMap<>();
        List<String> errorList = new ArrayList<>();
        HashMap<String,Integer> uniqueMap = new HashMap();
        for (int i = 0; i < dataList.size(); i++) {
            ScpSafetyTargetStockImportEntity importEntity = dataList.get(i);
            importEntity.setLine(i + 1);
            errorMap.put(importEntity.getLine(), new ArrayList<>());
            if (!StringUtils.hasText(importEntity.getOuCode())) {
                errorMap.get(importEntity.getLine()).add("列[公司编码]不能为空");
            }
            if (!StringUtils.hasText(importEntity.getWhCode())) {
                errorMap.get(importEntity.getLine()).add("列[仓库编码]不能为空");
            }
            if (!StringUtils.hasText(importEntity.getItemCode())) {
                errorMap.get(importEntity.getLine()).add("列[商品编码]不能为空");
            }
            if(!StringUtils.hasText(importEntity.getPlanUom())){
                errorMap.get(importEntity.getLine()).add("列[计划单位]不能为空");
            }
            Integer ex = uniqueMap.get(importEntity.getOuCode() + importEntity.getWhCode() + importEntity.getItemCode());
            if(ex != null){
                errorMap.get(importEntity.getLine()).add("列[公司编码]-[仓库编码]-[商品编码] 已存在,请检查");
                errorMap.get(ex).add("列[公司编码]-[仓库编码]-[商品编码] 已存在,请检查");
            }else{
                uniqueMap.put(importEntity.getOuCode() + importEntity.getWhCode() + importEntity.getItemCode(),i);
            }
            if (importEntity.getSafetyQty() == null) {
                errorMap.get(importEntity.getLine()).add("列[安全库存]不能为空");
            }
            if (importEntity.getTargetQty() == null) {
                errorMap.get(importEntity.getLine()).add("列[目标库存]不能为空");
            }
        }

        for (Integer line : errorMap.keySet()) {
            errorList.add(String.join(",", errorMap.get(line)));
        }
        if(Boolean.TRUE.equals(getErrorResult(errorList))){
            return errorList;
        }
        List<String> itemCodes = dataList.stream().map(ScpSafetyTargetStockImportEntity::getItemCode)
                .distinct().collect(Collectors.toList());
        List<String> ouCodes = dataList.stream().map(ScpSafetyTargetStockImportEntity::getOuCode)
                .distinct().collect(Collectors.toList());
        List<String> whCodes = dataList.stream().map(ScpSafetyTargetStockImportEntity::getWhCode)
                .distinct().collect(Collectors.toList());
        ItmItemSkuBusinessRpcDtoParam itmItemRpcDtoParam = new ItmItemSkuBusinessRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemSkuBusinessRpcDTO> itemInfos = itemRpcService.findItemSkuBusinessByParam(itmItemRpcDtoParam);
        Map<String, ItmItemSkuBusinessRpcDTO> itemMap = itemInfos.stream().collect(Collectors.toMap(i -> i.getItemCode(), i -> i, (o, n) -> n));
        List<InvWhRpcSimpleDTO> invWhRpcSimpleDTOS = invWhProvider.findSimpleWhByWhCodes(whCodes).computeData();
        Map<String,InvWhRpcSimpleDTO> whMap = invWhRpcSimpleDTOS.stream()
                .collect(Collectors.toMap(i -> i.getOuId() + i.getWhCode(), i -> i, (o, n) -> n));
        Map<String, InvWhRpcSimpleDTO> whCodeMap = invWhRpcSimpleDTOS.stream().collect(Collectors.toMap(i -> i.getWhCode(), i -> i,
                (o, n) -> n));
        List<OrgOuRpcSimpleDTO> simpleByOuCodes = orgOuRpcService.findSimpleByOuCodes(ouCodes);
        Map<String, OrgOuRpcSimpleDTO> ouMap = simpleByOuCodes.stream().collect(Collectors.toMap(i -> i.getOuCode(), i -> i, (o, n) -> n));
        log.info("仓库字典:{}", JSONUtil.toJsonStr(whCodeMap));
        List<ScpSafetyTargetStockDTO> stockSaveVOS = new ArrayList<>();
        for (ScpSafetyTargetStockImportEntity importEntity : dataList){
            ScpSafetyTargetStockDTO stockSaveVO = new ScpSafetyTargetStockDTO();
            stockSaveVO.setPlanUom(importEntity.getPlanUom());
            stockSaveVO.setSafetyQty(importEntity.getSafetyQty());
            stockSaveVO.setTargetQty(importEntity.getTargetQty());
            if(!itemMap.containsKey(importEntity.getItemCode())){
                errorMap.get(importEntity.getLine()).add("列[商品编码]不存在");
            }else{
                ItmItemSkuBusinessRpcDTO itemSkuBusinessRpcDTO = itemMap.get(importEntity.getItemCode());
                stockSaveVO.setItemId(itemSkuBusinessRpcDTO.getItemId());
                stockSaveVO.setItemName(itemSkuBusinessRpcDTO.getItemName());
                stockSaveVO.setItemType2(itemSkuBusinessRpcDTO.getItemType2());
                stockSaveVO.setBrand(itemSkuBusinessRpcDTO.getBrand());
                stockSaveVO.setItemCateCode(itemSkuBusinessRpcDTO.getItemCateCode());
                stockSaveVO.setUom(itemSkuBusinessRpcDTO.getUom());
            }
            if(!ouMap.containsKey(importEntity.getOuCode())){
                errorMap.get(importEntity.getLine()).add("列[公司编码]不存在");
            }else{
                OrgOuRpcSimpleDTO orgOuRpcSimpleDTO = ouMap.get(importEntity.getOuCode());
                stockSaveVO.setOuId(orgOuRpcSimpleDTO.getId());
                if(!whMap.containsKey(stockSaveVO.getOuId() + importEntity.getWhCode())){
                    errorMap.get(importEntity.getLine()).add(String.format("列【公司编码】不属于 仓库 %s", importEntity.getWhCode()));
                }else{
                    stockSaveVO.setOuName(orgOuRpcSimpleDTO.getOuName());
                    stockSaveVO.setOuCode(importEntity.getOuCode());
                    stockSaveVO.setWhId(whMap .get(stockSaveVO.getOuId() + importEntity.getWhCode()).getWhId());
                    stockSaveVO.setWhName(whMap .get(stockSaveVO.getOuId() + importEntity.getWhCode()).getWhName());
                    stockSaveVO.setWhCode(importEntity.getWhCode());
                    stockSaveVO.setBusinessId(stockSaveVO.getOuId().toString() + stockSaveVO.getWhId().toString() + stockSaveVO.getItemId().toString());
                }
            }
            stockSaveVOS.add(stockSaveVO);
        }
        if(!CollectionUtils.isEmpty(stockSaveVOS)){
            List<ScpSafetyTargetStockDTO> stockDTOS = stockSaveVOS.stream().filter(d -> d.getBusinessId() != null).distinct().collect(Collectors.toList());
            List<String> businessIds = stockDTOS.stream().map(ScpSafetyTargetStockDTO::getBusinessId).collect(Collectors.toList());
            List<ScpSafetyTargetStockDO> targetStockDOS = scpSafetyTargetStockRepo.findAllByBusinessIdIn(businessIds);
            if(!CollectionUtils.isEmpty(targetStockDOS)){
                List<Long> ids = targetStockDOS.stream().map(BaseModel::getId).collect(Collectors.toList());
                safetyTargetStockDomainService.delete(ids);
            }
            safetyTargetStockDomainService.saveAll(stockDTOS);
        }
        for (Integer line : errorMap.keySet()) {
            if (errorMap.get(line).isEmpty()) {
                continue;
            }
            if (!Objects.equals(errorList.get(line - 1), null)) {
                errorList.set(line - 1, errorList.get(line - 1) + String.join(",", errorMap.get(line)));
            }
        }
        return errorList;
    }

    @Nullable
    private Boolean getErrorResult(List<String> errorList) {
        Boolean flag = false;
        errorList.replaceAll(new UnaryOperator<String>() {
            @Override
            public String apply(String s) {
                if (!StringUtils.hasText(s)) {
                    return null;
                }
                return s;
            }
        });
        for (String s : errorList) {
            if (StringUtils.hasText(s)) {
                flag = true;
                break;
            }
        }
        return flag;
    }


    @Override
    public Integer stepSize() {
        return 200000;
    }

    @Override
    public Set<Integer> sheetNoList() {
        return Set.of(1);
    }
}
