package com.elitesland.scp.application.service.stock;

import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.invwh.InvWhRpcDtoParam;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.inv.dto.invwh.InvWhSimpleRpcDtoParam;
import com.elitesland.inv.provider.InvWhProvider;
import com.elitesland.scp.application.facade.vo.stock.*;
import com.elitesland.scp.domain.convert.stock.ScpSafetyTargetStockConvert;
import com.elitesland.scp.domain.entity.stock.ScpSafetyTargetStockDO;
import com.elitesland.scp.domain.service.stock.ScpSafetyTargetStockDomainService;
import com.elitesland.scp.infr.dto.stock.ScpSafetyTargetStockDTO;
import com.elitesland.scp.infr.repo.stock.ScpSafetyTargetStockRepo;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemSimpleRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBusinessRpcDtoParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author jeesie
 * @description:
 * @datetime 2025年 01月 16日 11:24
 * @version: 1.0
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ScpSafetyTargetStockServiceImpl implements ScpSafetyTargetStockService {

    private final ScpSafetyTargetStockDomainService safetyTargetStockDomainService;

    private final ScpSafetyTargetStockRepo scpSafetyTargetStockRepo;

    private final InvWhProvider invWhProvider;

    private final ItmItemRpcService itemRpcService;

    private final OrgOuRpcService orgOuRpcService;

    private final UdcProvider udcProvider;



    @Override
    public PagingVO<ScpSafetyTargetStockPageVO> searchPage(ScpSafetyTargetStockPageParam param) {
        PagingVO<ScpSafetyTargetStockPageVO> page = safetyTargetStockDomainService.searchPage(param);
        Map<String, String> uomMap = udcProvider.getValueMapByUdcCode("yst-supp", "UOM");
        if(CollectionUtils.isNotEmpty(page.getRecords())){
            List<Long> itemIds = page.getRecords().stream().map(ScpSafetyTargetStockPageVO::getItemId)
                    .filter(Objects::nonNull)
                    .distinct().collect(Collectors.toList());
            List<ItmItemSimpleRpcDTO> itemSimpleRpcDTOS = itemRpcService.findSimpleItem(itemIds);
            Map<Long, ItmItemSimpleRpcDTO> itmMap = itemSimpleRpcDTOS.stream()
                    .collect(Collectors.toMap(ItmItemSimpleRpcDTO::getId, v -> v, (e1, e2) -> e1));
            page.getRecords().forEach(d -> {
                if(itmMap.containsKey(d.getItemId())){
                    d.setItemName(itmMap.get(d.getItemId()).getItemName());
                    d.setItemCode(itmMap.get(d.getItemId()).getItemCode());
                    d.setBrandName(itmMap.get(d.getItemId()).getBrandName());
                    d.setBrand(itmMap.get(d.getItemId()).getBrand());
                    d.setItemType2(itmMap.get(d.getItemId()).getItemType2());
                }
                d.setPlanUomName(uomMap.get(d.getPlanUom()));
            });
        }
        return page;
    }

    @Override
    @Transactional
    public String saveSafetyTargetStock(ScpSafetyTargetStockSaveVO createParam) {
        ScpSafetyTargetStockDTO saveVoToDto = ScpSafetyTargetStockConvert.INSTANCE.saveVoToDto(createParam);
        String businessKey = createParam.computeBusinessKey();
        if(createParam.getId() == null){
            if(safetyTargetStockDomainService.findByBusinessKey(businessKey) != null){
                throw new BusinessException(ApiCode.FAIL,"数据已存在");
            }
        }else{
            ScpSafetyTargetStockDTO safetyTargetStockDTO = safetyTargetStockDomainService.findById(createParam.getId());
            if(safetyTargetStockDTO == null){
                throw new BusinessException(ApiCode.FAIL,"数据不存在");
            }
            List<ScpSafetyTargetStockDTO> safetyTargetStockDTOS = safetyTargetStockDomainService.findByBusinessKey(businessKey);
            long count = safetyTargetStockDTOS.stream().filter(d -> Objects.equals(d.getId(), createParam.getId())).count();
            if(count != 1){
                throw new BusinessException(ApiCode.FAIL,"同一公司、仓库、商品 数据已存在，不能重复");
            }
            ScpSafetyTargetStockConvert.INSTANCE.copySaveParamToDto(createParam,safetyTargetStockDTO);
        }
        saveVoToDto.setBusinessId(saveVoToDto.getOuId().toString()+saveVoToDto.getWhId().toString()+saveVoToDto.getItemId().toString());
        safetyTargetStockDomainService.save(saveVoToDto);
        return saveVoToDto.getId().toString();
    }

    @Override
    @Transactional
    public void delete(List<Long> ids) {
        safetyTargetStockDomainService.delete(ids);
    }

    @Override
    @Transactional
    public void batchSaveSafetyTargetStock(List<ScpSafetyTargetStockSaveVO> createParams) {
        if(CollectionUtils.isEmpty(createParams)){
            return;
        }
        List<String> businessKeys = createParams.stream().map(ScpSafetyTargetStockSaveVO::computeBusinessKey)
                .collect(Collectors.toList());
        List<String> businessKeys2 = createParams.stream().map(ScpSafetyTargetStockSaveVO::computeBusinessKey)
                .distinct()
                .collect(Collectors.toList());
        if(businessKeys2.size() != businessKeys.size()){
            throw new BusinessException(ApiCode.FAIL,"参数校验失败，相同公司、仓库、商品配置不能重复，请检查");
        }
        List<ScpSafetyTargetStockDO> targetStockDOS = scpSafetyTargetStockRepo.findAllByBusinessIdIn(businessKeys);
        List<Long> existIds = new ArrayList<>();
        Map<Long, List<ScpSafetyTargetStockDO>> existMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(targetStockDOS)){
            existMap = targetStockDOS.stream().collect(Collectors.groupingBy(d -> d.getWhId() + d.getOuId() + d.getItemId()));

        }
        List<ScpSafetyTargetStockSaveVO> saveVOList = createParams.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
        List<ScpSafetyTargetStockSaveVO> saveVOList2 = createParams.stream().filter(d -> d.getId() == null).collect(Collectors.toList());
        List<ScpSafetyTargetStockDTO> dtoList = new ArrayList<>();
        Map<Long, List<ScpSafetyTargetStockDO>> finalExistMap = existMap;
        if(CollectionUtils.isNotEmpty(saveVOList)){
            List<Long> ids = saveVOList.stream().map(ScpSafetyTargetStockSaveVO::getId).collect(Collectors.toList());
            List<ScpSafetyTargetStockDTO> safetyTargetStockDTOS = safetyTargetStockDomainService.findIdBatch(ids);
            Map<Long, ScpSafetyTargetStockDTO> targetStockDTOMap = safetyTargetStockDTOS.stream()
                    .collect(Collectors.toMap(ScpSafetyTargetStockDTO::getId, d -> d));
            List<ScpSafetyTargetStockDTO> stockDTOS = saveVOList.stream().map(d -> {
                d.setBusinessId(d.getBusinessId());
                List<ScpSafetyTargetStockDO> stockDOList = finalExistMap.get(d.getWhId() + d.getOuId() + d.getItemId());
                if (CollectionUtils.isNotEmpty(stockDOList)) {
                    List<ScpSafetyTargetStockDO> existList = stockDOList.stream().filter(e -> !Objects.equals(e.getId(), d.getId())).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(existList)) {
                        existIds.addAll(existList.stream().map(ScpSafetyTargetStockDO::getId).collect(Collectors.toList()));
                    }
                }
                ScpSafetyTargetStockDTO targetStockDTO = targetStockDTOMap.get(d.getId());
                ScpSafetyTargetStockConvert.INSTANCE.copySaveParamToDto(d, targetStockDTO);
                targetStockDTO.setBusinessId(targetStockDTO.getOuId().toString()+targetStockDTO.getWhId().toString()+targetStockDTO.getItemId().toString());
                return targetStockDTO;
            }).collect(Collectors.toList());
            dtoList.addAll(stockDTOS);
        }
        if(CollectionUtils.isNotEmpty(saveVOList2)){
            List<ScpSafetyTargetStockDTO> stockDTOS = saveVOList2.stream().map(d -> {
                d.setBusinessId(d.getBusinessId());
                List<ScpSafetyTargetStockDO> stockDOList = finalExistMap.get(d.getWhId() + d.getOuId() + d.getItemId());
                if (CollectionUtils.isNotEmpty(stockDOList)) {
                    existIds.addAll(stockDOList.stream().map(ScpSafetyTargetStockDO::getId).collect(Collectors.toList()));
                }
                return ScpSafetyTargetStockConvert.INSTANCE.saveVoToDto(d);
            }).collect(Collectors.toList());
            dtoList.addAll(stockDTOS);
        }
        if(CollectionUtils.isNotEmpty(existIds)){
            scpSafetyTargetStockRepo.deleteAllById(existIds);
        }
        if(CollectionUtils.isNotEmpty(dtoList)){
            safetyTargetStockDomainService.saveAll(dtoList);
        }
    }

    @Override
    public List<ScpSafetyTargetStockDownloadVO> downloadTmplFile(ScpSafetyTargetStockDownLoadParam param) {
        List<ScpSafetyTargetStockDownloadVO> downloadVOS = new ArrayList<>();
        InvWhSimpleRpcDtoParam invWhRpcDtoParam = new InvWhSimpleRpcDtoParam();
        invWhRpcDtoParam.setOuIds(param.getOuIds());
        List<InvWhRpcSimpleDTO> invWhRpcSimpleDTOS = invWhProvider.findNonStoreWhRpcDTO(invWhRpcDtoParam).computeData();
        if(CollectionUtils.isEmpty(invWhRpcSimpleDTOS)){
            return downloadVOS;
        }
        List<OrgOuRpcSimpleDTO> simpleOuDto1 = orgOuRpcService.findSimpleOuDto(param.getOuIds());
        if(CollectionUtils.isEmpty(simpleOuDto1)){
            throw new BusinessException(ApiCode.FAIL,"未查询到公司信息，请检查");
        }
        Map<Long, OrgOuRpcSimpleDTO> ouMap1 = simpleOuDto1.stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getId, v -> v, (e1, e2) -> e1));
        List<String> ouCodes = invWhRpcSimpleDTOS.stream().map(InvWhRpcSimpleDTO::getOuCode).distinct().collect(Collectors.toList());
        List<Long> ouIds = invWhRpcSimpleDTOS.stream().map(InvWhRpcSimpleDTO::getOuId).distinct().collect(Collectors.toList());
        List<String> filterOuNames = param.getOuIds().stream()
                .filter(d -> !ouIds.contains(d))
                .filter(d -> ouMap1.get(d) != null)
                .map(d -> ouMap1.get(d).getOuName())
                .collect(Collectors.toList());
        if(CollectionUtils.isNotEmpty(filterOuNames)){
            throw new BusinessException(ApiCode.FAIL,"参数校验失败，公司："+filterOuNames+"不存在非门店仓类型仓库，请配置");
        }
        List<OrgOuRpcSimpleDTO> simpleOuDto = orgOuRpcService.findSimpleOuDto(ouIds);
        Map<String, OrgOuRpcSimpleDTO> ouMap = simpleOuDto.stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getOuCode, v -> v, (e1, e2) -> e1));
        ItmItemBusinessRpcDtoParam businessRpcDtoParam = new ItmItemBusinessRpcDtoParam();
        businessRpcDtoParam.setBuCodes(ouCodes);
        businessRpcDtoParam.setMrpType("PD");
        List<ItmItemBusinessRpcDTO> businessRpcDTOList = itemRpcService.findItmItemBusinessByParam(businessRpcDtoParam);
        if(CollectionUtils.isEmpty(businessRpcDTOList)){
            log.info("没有符合运行MRP类型商品数据，请先创建商品");
            return downloadVOS;
        }
        Map<String, String> uomMap = udcProvider.getValueMapByUdcCode("yst-supp", "UOM");
        Map<String, List<ItmItemBusinessRpcDTO>> businessItemMap = businessRpcDTOList.stream().collect(Collectors.groupingBy(ItmItemBusinessRpcDTO::getBuCode));
        invWhRpcSimpleDTOS.forEach(d ->{
            List<ItmItemBusinessRpcDTO> itemBusinessRpcDTOS = businessItemMap.get(d.getOuCode());
            if(itemBusinessRpcDTOS != null && !itemBusinessRpcDTOS.isEmpty()){
                itemBusinessRpcDTOS.forEach(i ->{
                    ScpSafetyTargetStockDownloadVO safetyTargetStockDownloadVO = new ScpSafetyTargetStockDownloadVO();
                    safetyTargetStockDownloadVO.setWhCode(d.getWhCode());
                    safetyTargetStockDownloadVO.setWhName(d.getWhName());
                    safetyTargetStockDownloadVO.setOuCode(d.getOuCode());
                    safetyTargetStockDownloadVO.setOuName(ouMap.get(d.getOuCode()).getOuName());
                    safetyTargetStockDownloadVO.setItemCode(i.getItemCode());
                    safetyTargetStockDownloadVO.setItemName(i.getItemName());
                    safetyTargetStockDownloadVO.setPlanUom(i.getUom4());
                    safetyTargetStockDownloadVO.setPlanUomName(uomMap.get(i.getUom4()));
                    downloadVOS.add(safetyTargetStockDownloadVO);
                });
            }
        });
        return downloadVOS;
    }
}
