package com.elitesland.scp.rmi;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.support.provider.org.dto.*;
import com.elitesland.support.provider.org.param.OrgStoreBasePageRpcParam;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.param.OrgStoreRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class RmiOrgStoreRpcService {

    private final OrgStoreRpcService orgStoreRpcService;

    /**
     * 查询门店信息
     *
     * @param storeCode
     * @return
     */
    public OrgStoreDetailRpcDTO findOrgStoreByStoreCode(String storeCode) {
        log.info("查询门店信息入参：storeCode={}", storeCode);
        try {
            OrgStoreDetailRpcDTO rpcDTO = orgStoreRpcService.getByCode(storeCode);
            log.info("门店查询结果:{}", JSONUtil.toJsonStr(rpcDTO));
            return rpcDTO;
        } catch (Exception e) {
            log.error("查询门店信息:{}", e.getMessage(), e);
            throw new BusinessException(ApiCode.FAIL, "查询门店信息接口异常:" + e);
        }
    }

    /**
     * 查询门店信息集合
     *
     * @param storeCodes
     * @return
     */
    public List<OrgStoreDetailRpcDTO> queryByStoreCodes(List<String> storeCodes) {
        log.info("查询门店信息集合：storeCodes={}", storeCodes);
        try {
            return orgStoreRpcService.queryByStoreCodes(storeCodes);
        } catch (Exception e) {
            log.error("查询门店信息集合:{}", e);
            throw new BusinessException(ApiCode.FAIL, "查询门店信息集合接口异常:" + e);
        }
    }

    public List<OrgStoreWhDTO> listWhByCode(List<String> storeCodes) {
        log.info("查询门店仓库信息入参：storeCodes={}", storeCodes);
        ApiResult<List<OrgStoreWhDTO>> listApiResult = orgStoreRpcService.listWhByCode(storeCodes);
        if (!listApiResult.isSuccess()) {
            throw new BusinessException("查询门店仓库信息接口异常:" + listApiResult.getMsg());
        }
        return listApiResult.getData();
    }

    /**
     * 查询门店列表
     *
     * @param param
     * @return
     */
    public List<OrgStoreRpcDTO> appSearch(OrgStoreRpcParam param) {
        log.info("查询门店列表入参：storeCode={}", param);
        try {
            List<OrgStoreRpcDTO> orgStoreRpcDTOS = orgStoreRpcService.appSearch(param);
            log.info("查询门店列表结果:{}", JSONUtil.toJsonStr(orgStoreRpcDTOS));
            return orgStoreRpcDTOS;
        } catch (Exception e) {
            log.error("查询门店列表信息:{}", e.getMessage(), e);
            throw new BusinessException(ApiCode.FAIL, "查询门店列表接口异常:" + e);
        }
    }

    /**
     * 查询门店所有编码
     *
     *
     * @return
     */
    public List<String> queryAllStoreCodes() {
        try {
            List<String> listApiResult = orgStoreRpcService.queryAllStoreCodes().getData();
            return listApiResult;
        } catch (Exception e) {
            log.error("查询门店列表信息:{}", e.getMessage(), e);
            throw new BusinessException(ApiCode.FAIL, "查询门店列表接口异常:" + e);
        }
    }

    /**
     * 查询门店列表分页
     *
     * @param param
     * @return
     */
    public PagingVO<OrgStoreRpcDTO> rpcStoreQuery(OrgStoreRpcParam param) {
        log.info("查询门店列表入参：storeCode={}", param);
        try {
            PagingVO<OrgStoreRpcDTO> orgStoreRpcDTOPagingVO = orgStoreRpcService.rpcStoreQuery(param);
            log.info("查询门店列表结果:{}", JSONUtil.toJsonStr(orgStoreRpcDTOPagingVO));
            return orgStoreRpcDTOPagingVO;
        } catch (Exception e) {
            log.error("查询门店列表信息:{}", e.getMessage(), e);
            throw new BusinessException(ApiCode.FAIL, "查询门店列表接口异常:" + e);
        }
    }

    /**
     * 查询门店信息
     *
     * @param storeCode
     * @return
     */
    public OrgStoreDetailRpcDTO getByCode(String storeCode) {
        log.info("查询门店信息入参：storeCode={}", storeCode);
        try {
            OrgStoreDetailRpcDTO byCode = orgStoreRpcService.getByCode(storeCode);
            log.info("查询门店信息结果:{}", JSONUtil.toJsonStr(byCode));
            return byCode;
        } catch (Exception e) {
            log.error("查询门店信息信息:{}", e.getMessage(), e);
            throw new BusinessException(ApiCode.FAIL, "查询门店信息接口异常:" + e);
        }
    }

    public Map<String, OrgStoreBaseRpcDTO> getOrgStoreBaseMap(List<String> storeCodes) {
        log.info("查询门店信息入参getOrgStoreMap：storeCode={}", storeCodes);
        OrgStoreBaseRpcParam orgStoreBaseRpcParam = new OrgStoreBaseRpcParam();
        orgStoreBaseRpcParam.setStoreCodeList(storeCodes);
        ApiResult<List<OrgStoreBaseRpcDTO>> rpcDTO = orgStoreRpcService.findOrgStoreBaseByParam(orgStoreBaseRpcParam);
        log.info("门店查询结果:{}", JSONUtil.toJsonStr(rpcDTO));
        if (!rpcDTO.isSuccess()) {
            throw new BusinessException(ApiCode.FAIL, "查询门店信息接口异常:" + rpcDTO.getMsg());
        }
        List<OrgStoreBaseRpcDTO> data = rpcDTO.getData();
        if (CollUtil.isEmpty(data)) {
            return new HashMap<>();
        }
        return data.stream().collect(Collectors.toMap(OrgStoreBaseRpcDTO::getStoreCode, Function.identity()));
    }

    public OrgStoreBaseRpcDTO getOrgStoreBase(String storeCode) {
        log.info("查询门店信息入参getOrgStoreBase：storeCode={}", storeCode);
        OrgStoreBaseRpcParam orgStoreBaseRpcParam = new OrgStoreBaseRpcParam();
        orgStoreBaseRpcParam.setStoreCodeList(CollUtil.toList(storeCode));
        ApiResult<List<OrgStoreBaseRpcDTO>> rpcDTO = orgStoreRpcService.findOrgStoreBaseByParam(orgStoreBaseRpcParam);
        log.info("门店查询结果:{}", JSONUtil.toJsonStr(rpcDTO));
        if (!rpcDTO.isSuccess()) {
            throw new BusinessException(ApiCode.FAIL, "查询门店信息接口异常:" + rpcDTO.getMsg());
        }
        List<OrgStoreBaseRpcDTO> data = rpcDTO.getData();
        if (CollUtil.isEmpty(data)) {
            return null;
        }
        return data.get(0);
    }

    public List<OrgStoreBaseRpcDTO> findOrgStoreBaseByParam(List<String> storeCodes) {
        log.info("查询门店基础信息入参：baseRpcParam={}", JSONUtil.toJsonStr(storeCodes));

        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setStoreCodeList(storeCodes);
        ApiResult<List<OrgStoreBaseRpcDTO>> rpcDTO = orgStoreRpcService.findOrgStoreBaseByParam(baseRpcParam);
        log.info("查询门店基础信息结果:{}", JSONUtil.toJsonStr(rpcDTO));
        if (!rpcDTO.isSuccess()) {
            throw new BusinessException(ApiCode.FAIL, "查询门店基础信息接口异常:" + rpcDTO.getMsg());
        }
        return rpcDTO.getData();

    }

    public List<OrgStoreBaseRpcDTO> findOrgStoreBaseByBuIds(List<Long> buIds) {
        log.info("查询门店基础信息入参：baseRpcParam={}", JSONUtil.toJsonStr(buIds));
        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setBuIdList(buIds);
        ApiResult<List<OrgStoreBaseRpcDTO>> rpcDTO = orgStoreRpcService.findOrgStoreBaseByParam(baseRpcParam);
        log.info("查询门店基础信息结果:{}", JSONUtil.toJsonStr(rpcDTO));
        if (!rpcDTO.isSuccess()) {
            throw new BusinessException(ApiCode.FAIL, "查询门店基础信息接口异常:" + rpcDTO.getMsg());
        }
        return rpcDTO.getData();

    }


    public List<OrgStoreBaseRpcDTO> findOrgStoreBaseByRegion(List<String> regionList) {
        log.info("按区域查询门店基础信息入参：region={}", JSONUtil.toJsonStr(regionList));
        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setRegionList(regionList);
        ApiResult<List<OrgStoreBaseRpcDTO>> rpcDTO = orgStoreRpcService.findOrgStoreBaseByParam(baseRpcParam);
        log.info("按区域查询门店基础信息结果:{}", JSONUtil.toJsonStr(rpcDTO));
        if (!rpcDTO.isSuccess()) {
            throw new BusinessException(ApiCode.FAIL, "查询门店基础信息接口异常:" + rpcDTO.getMsg());
        }
        return rpcDTO.getData();

    }

    /**
     * 查询门店联系人信息
     *
     * @param storeCode
     * @return
     */
    public List<OrgContactRpcDTO> findContactByStoreCode(String storeCode) {
        List<OrgContactRpcDTO> contactRpcDTOS = orgStoreRpcService.findContactByStoreCode(storeCode);
        log.info("查询门店联系人信息结果:{}", JSONUtil.toJsonStr(contactRpcDTOS));
        return contactRpcDTOS;
    }

    public Long countBaseStoreByParam(OrgStoreBasePageRpcParam param) {
        ApiResult<Long> result = orgStoreRpcService.countBaseStoreByParam(param);
        return result.getData();
    }

    public PagingVO<OrgStoreBaseRpcDTO> queryBaseStore(OrgStoreBasePageRpcParam param) {
        return orgStoreRpcService.pageBaseStore(param);
    }

}
