package com.elitesland.scp.application.service.order;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCode;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.udc.UdcProvider;
import com.elitesland.scp.application.facade.vo.param.order.ScpOrderTemplatePageParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpOrderTemplatePageRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpOrderTemplateRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpOrderTemplateImportVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpOrderTemplateSaveVO;
import com.elitesland.scp.domain.convert.order.ScpOrderTemplateConvert;
import com.elitesland.scp.domain.entity.order.ScpOrderTemplateDO;
import com.elitesland.scp.infr.repo.order.ScpOrderTemplateRepo;
import com.elitesland.scp.infr.repo.order.ScpOrderTemplateRepoProc;
import com.elitesland.scp.rmi.RmiOrgRegionRpcService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.service.OrgRegionRpcService;
import io.swagger.annotations.ApiModelProperty;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpOrderTemplateServiceImpl implements ScpOrderTemplateService{
    private final ScpOrderTemplateRepo scpOrderTemplateRepo;
    private final ScpOrderTemplateRepoProc scpOrderTemplateRepoProc;
    private final UdcProvider udcProvider;
    private final RmiOrgStoreRpcService rmiOrgStoreService;
    private final RmiOrgRegionRpcService rmiOrgRegionRpcService;
    private final OrgRegionRpcService orgRegionRpcService;



    @Override
    public List<Long> addOrUpdate(List<ScpOrderTemplateSaveVO> scpOrderTemplateSaveVOss){
        // 1. 校验输入参数
        if (scpOrderTemplateSaveVOss == null || scpOrderTemplateSaveVOss.isEmpty()) {
            throw new IllegalArgumentException("模板列表不能为空");
        }

        // 2.重复数据
        ArrayList<ScpOrderTemplateDO> all = new ArrayList<>(ScpOrderTemplateConvert.INSTANCE.saveVoToDO(scpOrderTemplateSaveVOss));
        List<ScpOrderTemplateDO> existList = scpOrderTemplateRepo.findByIsActive(1);
        all.addAll(existList);
        this.validateDuplicates(all);
        // 2. 校验时间冲突
        this.validateDateConflicts(scpOrderTemplateSaveVOss);


        Map<String, List<ScpOrderTemplateDO>> existMap = existList.stream().collect(Collectors.groupingBy(this::buildUniqueKey2));
        ArrayList<ScpOrderTemplateDO> scpOrderTemplateDOS = new ArrayList<>();
        ScpOrderTemplateConvert.INSTANCE.saveVoToDO(scpOrderTemplateSaveVOss).forEach(saveDo -> {
            scpOrderTemplateDOS.addAll(handleTimeSplit(saveDo, existMap.get(this.buildUniqueKey2(saveDo))));

        });
        return this.saveSplitAndNewTemplates(scpOrderTemplateDOS);

    }

    /**
     * 处理时间区间分割逻辑
     */
    private List<ScpOrderTemplateDO> handleTimeSplit(ScpOrderTemplateDO newVo, List<ScpOrderTemplateDO> historyTemplates) {

        List<ScpOrderTemplateDO> splitTemplates = new ArrayList<>();

        splitTemplates.add(newVo);

        LocalDate newStart = newVo.getStartDate();
        LocalDate newEnd = newVo.getEndDate();

        // 如果开始日期和结束日期相同，不需要分割
        if (newStart.isEqual(newEnd) || CollectionUtils.isEmpty(historyTemplates)) {
            return splitTemplates;
        }

        for (ScpOrderTemplateDO history : historyTemplates) {
            LocalDate historyStart = history.getStartDate();
            LocalDate historyEnd = history.getEndDate();
            //大区间
            if (newStart.isBefore(historyStart) && newEnd.isAfter(historyEnd)) {
                scpOrderTemplateRepo.deleteById(history.getId());
                continue;
            }

            // 检查历史数据与新数据是否有重叠
            if (historyEnd.isBefore(newStart) || historyStart.isAfter(newEnd)) {
                // 无重叠，不处理
                continue;
            }
            scpOrderTemplateRepo.deleteById(history.getId());
            // 有重叠，需要分割历史数据
            // 1. 处理历史数据中早于新数据开始日期的部分
            if (historyStart.isBefore(newStart)) {
                ScpOrderTemplateDO beforePart = new ScpOrderTemplateDO();
                BeanUtils.copyProperties(history, beforePart);
                history.setId(null);
                beforePart.setEndDate(newStart.minusDays(1));
                splitTemplates.add(beforePart);
            }

            // 2. 处理历史数据中晚于新数据结束日期的部分
            if (historyEnd.isAfter(newEnd)) {
                ScpOrderTemplateDO afterPart = new ScpOrderTemplateDO();
                BeanUtils.copyProperties(history, afterPart);
                history.setId(null);
                afterPart.setStartDate(newEnd.plusDays(1));
                // 如果分割后的区间有效（开始日期 <= 结束日期）才保存
                if (!afterPart.getStartDate().isAfter(afterPart.getEndDate())) {
                    splitTemplates.add(afterPart);
                }
            }


            // 原历史数据将被删除或更新，不加入splitTemplates
        }

        return splitTemplates;
    }

    /**
     * 保存分割后的历史数据和新数据
     */
    private List<Long> saveSplitAndNewTemplates(List<ScpOrderTemplateDO> splitTemplates) {

        return scpOrderTemplateRepo.saveAll(splitTemplates.stream().collect(Collectors.toList())).stream().map(ScpOrderTemplateDO::getId).collect(Collectors.toList());
    }


    /**
     * 校验模板列表中的时间是否存在冲突
     */
    private void validateDateConflicts(List<ScpOrderTemplateSaveVO> templates) {
        // 先检查单个模板的时间有效性
        for (ScpOrderTemplateSaveVO template : templates) {
            LocalDate startDate = template.getStartDate();
            LocalDate endDate = template.getEndDate();

            // 检查时间是否为空
            if (startDate == null || endDate == null) {
                throw new BusinessException("开始时间或结束时间不能为空");
            }

            // 检查开始时间是否晚于结束时间
            if (startDate.isAfter(endDate)) {
                throw new BusinessException(
                        "开始时间(" + startDate + ")晚于结束时间(" + endDate + ")");
            }
        }

        // 检查模板之间的时间是否存在交集
        for (int i = 0; i < templates.size(); i++) {
            for (int j = i + 1; j < templates.size(); j++) {
                ScpOrderTemplateSaveVO template1 = templates.get(i);
                ScpOrderTemplateSaveVO template2 = templates.get(j);

                LocalDate s1 = template1.getStartDate();
                LocalDate e1 = template1.getEndDate();
                LocalDate s2 = template2.getStartDate();
                LocalDate e2 = template2.getEndDate();

                // 检查两个时间区间是否存在交集
                boolean hasConflict = !(e1.isBefore(s2) || e2.isBefore(s1));
                if (hasConflict) {
                    throw new BusinessException(
                            "时间存在冲突，" +
                                    "时间范围分别为: " + s1 + "至" + e1 + " 和 " + s2 + "至" + e2
                    );
                }
            }
        }
    }

    private void validateDuplicates(List<ScpOrderTemplateDO> templates) {
        // 用于检查列表内部重复的Set
        Set<String> duplicateCheckSet = new HashSet<>();

        for (ScpOrderTemplateDO template : templates) {
            if (StringUtils.isNotBlank(template.getStoreCode())  && StringUtils.isNotBlank(template.getRegionCode())) {
                throw new BusinessException(
                        "门店区域不能同时添加" +
                                ", 应用门店=" + template.getStoreCode() +
                                ", 区域=" + template.getRegionCode()

                );
            }
            // 构建唯一标识键
            String uniqueKey = buildUniqueKey(template);

            // 检查列表内部是否有重复
            if (duplicateCheckSet.contains(uniqueKey)) {
                throw new BusinessException(
                        "存在重复的模板配置：类型=" + template.getType() +
                                ", 单据类型=" + template.getDocType() +
                                ", 应用门店=" + template.getStoreCode() +
                                ", 区域=" + template.getRegionCode() +
                                ", 生效日期=" + template.getStartDate() +
                                ", 失效日期=" + template.getEndDate()
                );
            }
            duplicateCheckSet.add(uniqueKey);

            // 检查与数据库中已有记录是否重复
        }
    }


    /**
     * 构建唯一标识键
     */
    private String buildUniqueKey(ScpOrderTemplateDO template) {
        // 使用分隔符避免不同字段值拼接后产生歧义
        return String.join("|",
                Objects.toString(template.getType(), ""),
                Objects.toString(template.getDocType(), ""),
                Objects.toString(template.getStoreCode(), ""),
                Objects.toString(template.getRegionCode(), ""),
                Objects.toString(template.getStartDate(), ""),
                Objects.toString(template.getEndDate(), "")
        );
    }

    private String buildUniqueKey2(ScpOrderTemplateDO template) {
        // 使用分隔符避免不同字段值拼接后产生歧义
        return String.join("|",
                Objects.toString(template.getType(), ""),
                Objects.toString(template.getDocType(), ""),
                Objects.toString(template.getStoreCode(), ""),
                Objects.toString(template.getRegionCode(), "")
        );
    }



    @Override
    public void delete(Long id){
        scpOrderTemplateRepo.deleteById(id);
    }

    @Override
    public ScpOrderTemplateRespVO query(Long id){
        var s = scpOrderTemplateRepo.findById(id);
        if(s.isPresent()){
            ScpOrderTemplateRespVO scpOrderTemplateRespVO = new ScpOrderTemplateRespVO();
            BeanUtils.copyProperties(s.get(), scpOrderTemplateRespVO);
            scpOrderTemplateRespVO.update(udcProvider);
            return scpOrderTemplateRespVO;
        }
        return null;
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpOrderTemplatePageRespVO> page(ScpOrderTemplatePageParamVO scpOrderTemplatePageParamVO){

        var s = scpOrderTemplateRepoProc.page(scpOrderTemplatePageParamVO);
        Set<String> regionCodes = s.getRecords().stream().map(ScpOrderTemplatePageRespVO::getRegionCode).filter(Objects::nonNull).collect(Collectors.toSet());
        Map<String, String> regionMap =new HashMap<>();
        if (CollectionUtils.isNotEmpty(regionCodes)){
             regionMap = orgRegionRpcService.getNameByCode(regionCodes).computeData();
        }
        Map<String, String> finalRegionMap = regionMap;
        s.getRecords().forEach(e->{
//            e.update(udcProvider);
            e.setRegionName(finalRegionMap.get(e.getRegionCode()));
        });
        return s;
    }

    @Override
    @Transactional
    public List<String> executeImport(List<ScpOrderTemplateImportVO> dataList, int startRowIndex) {
        List<String> errorList = new ArrayList<>();

        // 提取公共 formatter 为静态 final 常量更高效
        final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss");

        Map<String, String> demandSetType = udcProvider.getValueMapByUdcCode("yst-suplan", "DEMAND_SET_TYPE");
        Map<String, String> obDocType = udcProvider.getValueMapByUdcCode("yst-suplan", "OB_DOC_TYPE");

        // 构造反向映射表
        Map<String, String> obDocTypeMap = obDocType.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        Map<String, String> demandSetTypeMap = demandSetType.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

        // 获取门店及区域信息
        Set<String> storeCodes = dataList.stream()
                .map(ScpOrderTemplateImportVO::getStoreCode)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        Set<String> regionCodes = dataList.stream()
                .map(ScpOrderTemplateImportVO::getRegionCode)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        Map<String, OrgStoreDetailRpcDTO> storeDetailRpcDTOMap = new HashMap<>();
        Map<String, String> rpcDtoByRegionCodes = new HashMap<>();

        if (!storeCodes.isEmpty()) {
            storeDetailRpcDTOMap = rmiOrgStoreService.queryByStoreCodes(new ArrayList<>(storeCodes))
                    .stream()
                    .collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode, Function.identity()));
        }

        if (!regionCodes.isEmpty()) {
            rpcDtoByRegionCodes = rmiOrgRegionRpcService.findRpcDtoByRegionCodes(regionCodes);
        }

        List<ScpOrderTemplateSaveVO> scpOrderTemplateSaveVOS = new ArrayList<>();
        int rowIndex;

        for (int i = 0; i < dataList.size(); i++) {

            rowIndex = startRowIndex + i;

            try {
                ScpOrderTemplateImportVO importVO = dataList.get(i);
                validateRow(importVO, rowIndex, storeDetailRpcDTOMap, rpcDtoByRegionCodes, errorList);

                ScpOrderTemplateSaveVO saveVO = convertToSaveVO(importVO, obDocTypeMap, demandSetTypeMap, DATE_FORMATTER, TIME_FORMATTER, storeDetailRpcDTOMap);
                scpOrderTemplateSaveVOS.add(saveVO);
            }catch (Exception e){
                errorList.add("第" + rowIndex + "行导入失败：" + e.getMessage());
            }
        }

        if (errorList.isEmpty()) {
            this.addOrUpdate(scpOrderTemplateSaveVOS);
        }

        return errorList;
    }

    private void validateRow(ScpOrderTemplateImportVO importVO,
                             int rowIndex,
                             Map<String, OrgStoreDetailRpcDTO> storeDetailRpcDTOMap,
                             Map<String, String> rpcDtoByRegionCodes,
                             List<String> errorList) {

        boolean hasStoreCode = StringUtils.isNotBlank(importVO.getStoreCode());
        boolean hasRegionCode = StringUtils.isNotBlank(importVO.getRegionCode());

        if (!hasStoreCode && !hasRegionCode) {
            throw new RuntimeException("门店代码,区域不能同时为空");
        }

        if (hasStoreCode && hasRegionCode) {
            throw new RuntimeException("门店代码,区域不能同时填写");

        }

        if (hasStoreCode && !storeDetailRpcDTOMap.containsKey(importVO.getStoreCode())) {
            throw new RuntimeException("门店编码:" + importVO.getStoreCode() + "不存在");

        }

        if (hasRegionCode && !rpcDtoByRegionCodes.containsKey(importVO.getRegionCode())) {
            throw new RuntimeException("区域编码:" + importVO.getRegionCode() + "不存在");

        }
    }

    private ScpOrderTemplateSaveVO convertToSaveVO(
            ScpOrderTemplateImportVO importVO,
            Map<String, String> obDocTypeMap,
            Map<String, String> demandSetTypeMap,
            DateTimeFormatter dateFormatter,
            DateTimeFormatter timeFormatter,
            Map<String, OrgStoreDetailRpcDTO> storeDetailRpcDTOMap) {

        ScpOrderTemplateSaveVO saveVO = new ScpOrderTemplateSaveVO();
        BeanUtils.copyProperties(importVO, saveVO);

        saveVO.setDocType(obDocTypeMap.get(importVO.getDocTypeName()));
        saveVO.setType(demandSetTypeMap.get(importVO.getType()));

        try {
            saveVO.setStartDate(LocalDate.parse(importVO.getStartDate(), dateFormatter));
            saveVO.setEndDate(LocalDate.parse(importVO.getEndDate(), dateFormatter));
            saveVO.setDeadlineTime(LocalTime.parse(importVO.getDeadlineTime(), timeFormatter));
        } catch (Exception e) {
            throw new RuntimeException("日期时间格式错误，请检查输入数据", e);
        }

        saveVO.setStoreName(Optional.ofNullable(storeDetailRpcDTOMap.get(importVO.getStoreCode()))
                .map(OrgStoreDetailRpcDTO::getStoreName)
                .orElse(null));

        return saveVO;
    }


}
