package com.elitesland.scp.utils;

import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.vo.SysSettingVO;
import com.elitesland.scp.rmi.RmiSysSettingService;
import com.elitesland.support.provider.org.dto.OrgStoreWhDTO;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import jodd.util.StringUtil;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author yangzhibin
 * @date 2023/8/30
 */
public class SysUtils {

    private static final String PRICE_DECIMAL_PLACE = "PRICE_DECIMAL_PLACE";
    private static final String AMOUNT_DECIMAL_PLACE = "AMOUNT_DECIMAL_PLACE";
    private static final String QUANTITY_DECIMAL_PLACE = "QUANTITY_DECIMAL_PLACE";
    private static final String SCP_ORDER_SETTING = "SCP_ORDER_SETTING";
    private static final String SCP_ORDER_ITEM_SETTING = "SCP_ORDER_ITEM_SETTING";
    private static final String RETURN_ORDER_COMPENSATION_ITEM_SETTING ="RETURN_ORDER_COMPENSATION_ITEM_SETTING";
    private static final String SCP_ORDER_ITEM_IMG_URL_SETTING = "SCP_ORDER_ITEM_IMG_URL_SETTING";
    private static final String SCP_COMPENSATION_ITEM_IMG_URL_SETTING = "SCP_COMPENSATION_ITEM_IMG_URL_SETTING";
    private static final String DEFAULT_COMPENSATION_RECEIPT_WH = "DEFAULT_COMPENSATION_RECEIPT_WH";
    private static final String SCP_ORDER_EXPIRE_SETTING = "SCP_ORDER_EXPIRE_SETTING";
    private static final String INV_OIDOC = "INV_OIDOC";
    private static final String STORE_UNPAID_ORDER_CLOSE_TIME_SET = "STORE_UNPAID_ORDER_CLOSE_TIME_SET";

    private static final String SCP_ORDER_SET_AUTO_CREATE_NUM = "SCP_ORDER_SET_AUTO_CREATE_NUM";

    private static Cache<String, String> orderSettingCache = Caffeine.newBuilder().maximumSize(5).expireAfterWrite(8, TimeUnit.HOURS).build();

    private static OrgStoreRpcService orgStoreRpcService;

    private static final Cache<String, List<OrgStoreWhDTO>> storeWhCache = Caffeine.newBuilder()
            .maximumSize(200)
            .expireAfterAccess(8L, TimeUnit.HOURS)
            .build();

    private static Cache<String, Integer> placeCache = Caffeine.newBuilder()
            .maximumSize(3)
            .expireAfterAccess(8L, TimeUnit.HOURS)
            .build();

    /**
     * 调用系统域，获取单价小数位数
     */
    public static Integer getPricePlace() {
        return placeCache.get(PRICE_DECIMAL_PLACE, key -> {
            SysSettingVO priceDecimalPlace = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(PRICE_DECIMAL_PLACE);
            return (priceDecimalPlace == null || StrUtil.isBlank(priceDecimalPlace.getSettingVal())) ? 8 : Integer.parseInt(priceDecimalPlace.getSettingVal().trim());
        });
    }

    /**
     * 调用系统域，获取金额小数位数
     */
    public static Integer getAmtPlace() {
        return placeCache.get(AMOUNT_DECIMAL_PLACE, key -> {
            SysSettingVO amountDecimalPlace = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(AMOUNT_DECIMAL_PLACE);
            return (amountDecimalPlace == null || StrUtil.isBlank(amountDecimalPlace.getSettingVal())) ? 8 : Integer.parseInt(amountDecimalPlace.getSettingVal().trim());
        });
    }

    /**
     * 调用系统域，获取数量小数位数
     */
    public static Integer getQtyPlace() {
        return placeCache.get(QUANTITY_DECIMAL_PLACE, key -> {
            SysSettingVO qtyDecimalPlace = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(QUANTITY_DECIMAL_PLACE);
            return (qtyDecimalPlace == null || StrUtil.isBlank(qtyDecimalPlace.getSettingVal())) ? 8 : Integer.parseInt(qtyDecimalPlace.getSettingVal().trim());
        });
    }

    /**
     * 调用系统域，获取赔付默认仓库
     */
    public static String getDefaultInStore() {
        SysSettingVO defaultInStoreCode = null;
        try {
            defaultInStoreCode = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(DEFAULT_COMPENSATION_RECEIPT_WH);
            return defaultInStoreCode.getSettingVal();
        } catch (Exception e) {
            throw new BusinessException("调用系统域报错：获取赔付默认仓库报错");
        }
    }

    public static BigDecimal processPriceScale(BigDecimal number) {
        if (number == null) {
            return null;
        }
        return number.setScale(getPricePlace(), RoundingMode.HALF_UP);
    }

    public static BigDecimal processAmtScale(BigDecimal number) {
        if (number == null) {
            return null;
        }
        return number.setScale(getAmtPlace(), RoundingMode.HALF_UP);
    }

    public static BigDecimal processQtyScale(BigDecimal number) {
        if (number == null) {
            return null;
        }
        return number.setScale(getQtyPlace(), RoundingMode.HALF_UP);
    }

    public static BigDecimal processStoreQtyScale(BigDecimal number) {
        if (number == null) {
            return null;
        }
        return number.setScale(0, RoundingMode.HALF_UP);
    }

    public static Map<String, Integer> refreshSetting() {
        placeCache.invalidateAll();
        getPricePlace();
        getAmtPlace();
        getQtyPlace();
        getOrderSetting();
        getOrderItemSetting();
        return placeCache.asMap();
    }

    /**
     * 获取订货订单设置
     *
     * @return
     */
    public static String getOrderSetting() {
        return orderSettingCache.get(SCP_ORDER_SETTING, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(SCP_ORDER_SETTING);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "1" : orderSetting.getSettingVal().trim();
        });
    }

    /**
     * 获取订货单运费商品行设置
     *
     * @return
     */
    public static String getOrderItemSetting() {
        return orderSettingCache.get(SCP_ORDER_ITEM_SETTING, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(SCP_ORDER_ITEM_SETTING);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "1" : orderSetting.getSettingVal().trim();
        });
    }

    /**
     * 获取订货单运费商品行图片地址设置
     *
     * @return
     */
    public static String getOrderItemImgUrlSetting() {
        return orderSettingCache.get(SCP_ORDER_ITEM_IMG_URL_SETTING, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(SCP_ORDER_ITEM_IMG_URL_SETTING);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "" : orderSetting.getSettingVal().trim();
        });
    }

    /**
     * 获取赔付商品行图片地址设置
     *
     * @return
     */
    public static String getCompImgUrlSetting() {
        return orderSettingCache.get(SCP_COMPENSATION_ITEM_IMG_URL_SETTING, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(SCP_COMPENSATION_ITEM_IMG_URL_SETTING);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "" : orderSetting.getSettingVal().trim();
        });
    }

    /**
     * 获取赔付商品行设置
     *
     * @return
     */
    public static String getCompItemSetting() {
        return orderSettingCache.get(RETURN_ORDER_COMPENSATION_ITEM_SETTING, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(RETURN_ORDER_COMPENSATION_ITEM_SETTING);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "1" : orderSetting.getSettingVal().trim();
        });
    }

    /**
     * 调用系统域，未支付订单自动关闭时间
     */
    public static String getDefaultCloseTime() {
        SysSettingVO defaultInStoreCode = null;
        try {
            defaultInStoreCode = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(STORE_UNPAID_ORDER_CLOSE_TIME_SET);
            return defaultInStoreCode.getSettingVal();
        } catch (Exception e) {
            throw new BusinessException("调用系统域报错：获取未支付订单自动关闭时间报错");
        }
    }


    /**
     * 生成redis key
     *
     * @param prefix
     * @param itemKey
     * @return
     */
    public static String generateRedisKey(String prefix, String itemKey) {
        return prefix + ":" + itemKey;
    }

    /**
     *
     * @param prefix
     * @param combineItemCode
     * @param itemCode
     * @return
     */
    public static String generateRedisKey(String prefix, String combineItemCode, String itemCode) {
        if (StringUtil.isBlank(combineItemCode)) {
            return prefix + ":" + itemCode;
        }
        return prefix + ":" + combineItemCode + ":" + itemCode;
    }

    /**
     * 添加强配的购物车商品
     *
     * @param prefix
     * @param combineItemCode
     * @param itemCode
     * @param activityCode
     * @return
     */
    public static String generateRedisKey(String prefix, String combineItemCode, String itemCode, String activityCode) {
        if (StringUtil.isBlank(activityCode)) {
            return generateRedisKey(prefix, combineItemCode, itemCode);
        }
        if (StringUtil.isBlank(combineItemCode)) {
            return prefix + ":" + itemCode + ":" + activityCode + ":ALLOC";
        }
        return prefix + ":" + combineItemCode + ":" + itemCode + ":" + activityCode + ":ALLOC";
    }


    /**
     * 获取价格key
     * @param combineItemCode
     * @param itemCode
     * @return
     */
    public static String getItemKey(String combineItemCode, String itemCode) {
        if (StringUtil.isBlank(combineItemCode)) {
            return itemCode;
        }
        return combineItemCode + ":" + itemCode;
    }

    public static String getItemKey(String combineItemCode, String itemCode, String activityCode) {
        if (StringUtil.isBlank(activityCode)) {
            return getItemKey(combineItemCode, itemCode);
        }
        if (StringUtil.isBlank(combineItemCode)) {
            return itemCode + ":" + activityCode + ":ALLOC";
        }
        return combineItemCode + ":" + itemCode + ":" + activityCode + ":ALLOC";
    }

    public static Map<String, List<OrgStoreWhDTO>> getStoreWhListCache(List<String> storeCodes) {
        if (orgStoreRpcService == null) {
            orgStoreRpcService = SpringContextHolder.getBean(OrgStoreRpcService.class);
        }
        List<String> notCacheCodes = new ArrayList<>();
        Map<String, List<OrgStoreWhDTO>> storeWhMap = new HashMap<>();
        for (String store : storeCodes) {
            List<OrgStoreWhDTO> storeWhDTOS = storeWhCache.getIfPresent(store);
            if (CollectionUtils.isEmpty(storeWhDTOS)) {
                notCacheCodes.add(store);
            } else {
                storeWhMap.put(store, storeWhDTOS);
            }
        }
        if (!CollectionUtils.isEmpty(notCacheCodes)) {
            List<OrgStoreWhDTO> storeWhDTOS = orgStoreRpcService.listWhByCode(storeCodes).computeData();
            Map<String, List<OrgStoreWhDTO>> map = storeWhDTOS.stream().collect(Collectors.groupingBy(OrgStoreWhDTO::getStoreCode));
            storeWhMap.putAll(map);
            storeWhCache.putAll(map);
        }
        return storeWhMap;
    }

    public static String getInvOidoc() {
        return orderSettingCache.get(INV_OIDOC, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(INV_OIDOC);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "1" : orderSetting.getSettingVal().trim();
        });
    }

    public static Integer getAutoOrderSetNumSetting() {
        String s = orderSettingCache.get(SCP_ORDER_SET_AUTO_CREATE_NUM, key -> {
            SysSettingVO orderSetting = SpringUtil.getBean(RmiSysSettingService.class).findSysSettingByNo(SCP_ORDER_SET_AUTO_CREATE_NUM);
            return (orderSetting == null || StrUtil.isBlank(orderSetting.getSettingVal())) ? "1" : orderSetting.getSettingVal().trim();
        });
        return Integer.parseInt(s);
    }
}
