package com.elitesland.scp.application.service.calendar;

import com.el.coordinator.core.common.utils.BeanCopyUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.calendar.*;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreDemandCalendarParam;
import com.elitesland.scp.application.facade.vo.param.setting.ScpOrderSettingParamVO;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreDemandCalendarRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.supalloc.ScpSupplyAllocationPageVO;
import com.elitesland.scp.application.service.scpsman.BaseScpmanAuthorityParam;
import com.elitesland.scp.application.service.setting.ScpOrderSettingService;
import com.elitesland.scp.domain.convert.calendar.ScpStoreDemandCalendarConvert;
import com.elitesland.scp.domain.entity.calendar.ScpStoreDemandCalendarDO;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.domain.service.calendar.ScpStoreDemandCalendarDomainService;
import com.elitesland.scp.infr.dto.calendar.ScpStoreDemandCalendarDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
* @description:  
* @author: jeesie.jiang
* @create: 2024-03-28
* @Version 1.0
**/
@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreDemandCalendarServiceImpl  implements ScpStoreDemandCalendarService {

    private final ScpStoreDemandCalendarDomainService storeDemandCalendarDomainService;

    private final ScpDemandAuthorityService scpDemandAuthorityService;

    private final ScpOrderSettingService scpOrderSettingService;

    @Override
    @SysCodeProc
    public PagingVO<ScpStoreDemandCalendarPageVO> searchPage(ScpStoreDemandCalendarPageParamVO param) {
        if(Boolean.TRUE.equals(param.getScpmanAuthority())){
            PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
            if(userAuthority.isEmpty()){
                return PagingVO.<ScpStoreDemandCalendarPageVO>builder()
                        .total(0L)
                        .records(new ArrayList<>())
                        .build();
            }
            extractedAuthorityParam(param, userAuthority);
        }
        PagingVO<ScpStoreDemandCalendarPageVO> page = storeDemandCalendarDomainService.searchPage(param);
        if(CollectionUtils.isNotEmpty(page.getRecords())){
            List<String> strings = page.getRecords().stream().map(d -> d.getStoreCode() + d.getItemType2() +d.getYear() + d.getMonth()).collect(Collectors.toList());
            List<ScpStoreDemandCalendarDO> calendardos = storeDemandCalendarDomainService.findByConcatYearAndMonthKey(strings);
            Map<String, List<ScpStoreDemandCalendarDO>> listMap = calendardos.stream().collect(Collectors.groupingBy(v -> v.getStoreCode()  + v.getItemType2() + v.getYear() + v.getMonth()));
            page.getRecords().forEach(d -> {
                List<ScpStoreDemandCalendarDO> calendarDOList = listMap.get(d.getStoreCode() + d.getItemType2() + d.getYear() + d.getMonth());
                d.setId(calendarDOList.get(0).getId());
                if(CollectionUtils.isNotEmpty(calendarDOList)){
                    var details = calendarDOList.stream().map(m -> {
                        ScpStoreDemandCalendarPageVO.CalendarDayDetail detail = ScpStoreDemandCalendarConvert.INSTANCE.doToPageVo(m);
                        detail.setDay(m.getDay());
                        detail.setWorkStatus(m.getWorkStatus());
                        return detail;
                    }).collect(Collectors.toList());
                    var sortList = details.stream()
                            .filter(v -> v.getDay() != null)
                            .sorted(getComparing())
                            .collect(Collectors.toList());
                    d.setDayDetails(sortList);
                }
            });
        }
        return page;
    }

    private void extractedAuthorityParam(BaseScpmanAuthorityParam queryParam, PagingVO<ScpManAuthorityPageRespVO> userAuthority) {
        List<String> whCodes = userAuthority.stream()
                .filter(v -> 1 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        List<String> stores = userAuthority.stream()
                .filter(v -> 0 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        queryParam.setWhCodes(whCodes);
        queryParam.setStores(stores);
        queryParam.setScpmanAuthority(Boolean.TRUE);
    }

    private Comparator<ScpStoreDemandCalendarPageVO.CalendarDayDetail> getComparing() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveStoreCalendar(ScpStoreDemandCalendarSaveVO createParam) {
        if (CollectionUtils.isEmpty(createParam.getItemType2s())){
            throw new BusinessException(ApiCode.FAIL,"请至少选择一种商品类型");
        }

        if(createParam.getItemType2s().contains("all")) {
            ScpOrderSettingParamVO scpOrderSettingParamVO = new ScpOrderSettingParamVO();
            List<ScpOrderSettingRespVO> scpOrderSettingRespVOS = scpOrderSettingService.queryOrderSettingList(scpOrderSettingParamVO);
            createParam.setItemType2s(scpOrderSettingRespVOS.stream().map(ScpOrderSettingRespVO::getDocType).collect(Collectors.toList()));
        }

          var dayDetails = createParam.getDayDetails();
        List<ScpStoreDemandCalendarDO> calendarDOList=new ArrayList<>();
        createParam.getItemType2s().forEach(itemType2 -> {
            calendarDOList.addAll(dayDetails.stream().map(d -> {
                ScpStoreDemandCalendarDO demandCalendarDO = ScpStoreDemandCalendarConvert.INSTANCE.saveVoToDo(d);
                demandCalendarDO.setDay(d.getDay());
                demandCalendarDO.setWorkStatus(d.getWorkStatus());
                demandCalendarDO.setStoreCode(createParam.getStoreCode());
                demandCalendarDO.setStoreId(createParam.getStoreId());
                demandCalendarDO.setStoreName(createParam.getStoreName());
                demandCalendarDO.setYear(createParam.getYear());
                demandCalendarDO.setMonth(createParam.getMonth());
                demandCalendarDO.setType(createParam.getType());
                demandCalendarDO.setItemType2(itemType2);
                return demandCalendarDO;
            }).collect(Collectors.toList()));
        });

        List<String> concatKey = calendarDOList.stream()
                .map(d -> d.getStoreCode() + d.getItemType2() + d.getYear()
                        + d.getMonth()
                        + d.getDay()
                )
                .collect(Collectors.toList());
        List<ScpStoreDemandCalendarDO> existList = storeDemandCalendarDomainService.findByStoreDayCalendar(concatKey);
        List<ScpStoreDemandCalendarDO> saveList = calendarDOList.stream().filter(d -> d.getId() == null)
                .collect(Collectors.toList());
        List<ScpStoreDemandCalendarDO> updateList = calendarDOList.stream().filter(d -> d.getId() != null)
                .collect(Collectors.toList());
        if(CollectionUtils.isNotEmpty(saveList)){
            List<String> saveKey = saveList.stream()
                    .map(d -> d.getStoreCode() + d.getItemType2() + d.getYear()
                            + d.getMonth()
                            + d.getDay()
                    ).collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(existList)) {
                List<String> existKeys = existList.stream().filter(d -> saveKey.contains(d.getStoreCode() +d .getItemType2() +  d.getYear()
                        + d.getMonth()
                        + d.getDay())).map(d -> d.getStoreCode() + "_"  + d.getItemType2() + "_" + d.getYear()
                        + "_" + d.getMonth()
                        ).distinct().collect(Collectors.toList());
                if(CollectionUtils.isNotEmpty(existKeys)){
                    throw new BusinessException(ApiCode.FAIL, String.join(",", existKeys) + "配置已存在，请修改");
                }
            }
            storeDemandCalendarDomainService.createBatch(saveList);
        }
        if(CollectionUtils.isNotEmpty(updateList)){
            storeDemandCalendarDomainService.createBatch(updateList);
        }
    }


    @Override
    @SysCodeProc
    public List<ScpStoreDemandCalendarRespVO> listCalendarVOs(ScpStoreDemandCalendarParam scpStoreDemandCalendarParam) {
        return ScpStoreDemandCalendarConvert.INSTANCE.dtoToRespVO(storeDemandCalendarDomainService.listCalendarDTOs(scpStoreDemandCalendarParam));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteStoreCalendar(List<ScpStoreDemandCalendarParamVO> deleteParams) {
        long nullcount = deleteParams.stream().filter(d -> d.getYear() == null).count();
        if(nullcount > 0){
            throw new BusinessException(ApiCode.FAIL,"年不能为空");
        }
        long nullcount2 = deleteParams.stream().filter(d -> d.getMonth() == null).count();
        if(nullcount2 > 0){
            throw new BusinessException(ApiCode.FAIL,"月不能为空");
        }
        List<String> concatKey = deleteParams.stream().map(d -> d.getStoreCode() + d.getItemType2() + d.getYear() + d.getMonth()).collect(Collectors.toList());
        List<ScpStoreDemandCalendarDO> demandCalendarDOS = storeDemandCalendarDomainService.findByConcatYearAndMonthKey(concatKey);
        if(CollectionUtils.isNotEmpty(demandCalendarDOS)){
            List<Long> ids = demandCalendarDOS.stream().map(ScpStoreDemandCalendarDO::getId).distinct()
                    .collect(Collectors.toList());
            storeDemandCalendarDomainService.deleteBatch(ids);
        }
    }

    @Override
    @SysCodeProc
    public ScpStoreDemandCalendarRespVO findStoreCalendarDetail(ScpStoreDemandCalendarParamVO param) {
        ScpStoreDemandCalendarRespVO vo = new ScpStoreDemandCalendarRespVO();
        List<ScpStoreDemandCalendarDTO> calendarDOS = storeDemandCalendarDomainService.findStoreCalendarByParam(param);
        if(CollectionUtils.isEmpty(calendarDOS)){
            return vo;
        }
        if(param.getMonth() != null && param.getYear() != null){
            List<ScpStoreDemandCalendarRespVO> calendarRespVOS = calendarDOS.stream()
                    .map(ScpStoreDemandCalendarConvert.INSTANCE::dtoToRespVo)
                    .collect(Collectors.toList());
            ScpStoreDemandCalendarRespVO calendarRespVO = calendarRespVOS.get(0);
            BeanCopyUtil.beanCopyWithIngore(calendarRespVO,vo,BeanCopyUtil.getNullAndIgnorePropertyNames(calendarRespVO));
            var vos = calendarRespVOS.stream().filter(d -> d.getDay() != null)
                    .sorted(getComparing2()).collect(Collectors.toList());
            vo.setDayDetails(vos);
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            List<String> days = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getDay).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
            vo.setExistDaysOfMonth(days);

        }else if( param.getYear() != null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
        }else if( param.getYear() == null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
        }
        return vo;
    }

    @Override
    public Boolean isExistStoreByDateAndWorkStatus(ScpCalendarStoreQueryParamVO paramVO) {
        return CollectionUtils.isNotEmpty(storeDemandCalendarDomainService.listStoreByQueryParam(paramVO));
    }

    @Override
    public List<ScpCalendarStoreRespVO> listCalendarStoreByParam(ScpCalendarStoreQueryParamVO paramVO) {
        return storeDemandCalendarDomainService.listStoreByQueryParam(paramVO);
    }

    private Comparator<ScpStoreDemandCalendarRespVO> getComparing2() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }
}