package com.elitesland.scp.application.service.alloc;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.PageUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingPageParamVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingStatusParamVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingStoreParamVO;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingItemRespVO;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingRespVO;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingStoreRespVO;
import com.elitesland.scp.application.facade.vo.save.alloc.*;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.alloc.ScpAllocSettingConvert;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingDO;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingRedisDO;
import com.elitesland.scp.domain.service.alloc.*;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingRepo;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.rmi.RmiSysUserRpcService;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.dto.SpuImageDTO;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBasePageRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpAllocSettingServiceImpl implements ScpAllocSettingService {
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private final ScpAllocSettingDomainService scpAllocSettingDomainService;
    private final ScpAllocSettingItemDomainService scpAllocSettingItemDomainService;
    private final ScpAllocSettingRegionDomainService scpAllocSettingRegionDomainService;
    private final ScpAllocSettingBrandDomainService scpAllocSettingBrandDomainService;
    private final ScpAllocSettingStoreDomainService scpAllocSettingStoreDomainService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RedisTemplate redisClient;
    private final ScpAllocSettingRepo scpAllocSettingRepo;
    private final RmiItemService rmiItemService;

    @Override
    public PagingVO<ScpAllocSettingPageRespVO> page(ScpAllocSettingPageParamVO paramVO) {
        Set<Long> masIds = new HashSet<>();
        if (StrUtil.isNotBlank(paramVO.getRegion())) {
            List<Long> regionMasIds = scpAllocSettingRegionDomainService.findMasIdByRegionCode(paramVO.getRegion());
            if (CollUtil.isEmpty(regionMasIds)) {
                return new PagingVO<>();
            }
            masIds.addAll(regionMasIds);
        }
        if (StrUtil.isNotBlank(paramVO.getBrand())) {
            List<Long> brandMasIds = scpAllocSettingBrandDomainService.findMasIdByBrand(paramVO.getBrand());
            if (CollUtil.isEmpty(brandMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, brandMasIds);
        }
        if (paramVO.getStoreCode() != null) {
            List<Long> storeMasIds = scpAllocSettingStoreDomainService.findMasIdByStoreCode(paramVO.getStoreCode());
            if (CollUtil.isEmpty(storeMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, storeMasIds);
        }
        if (paramVO.getItemId() != null) {
            List<Long> itemMasIds = scpAllocSettingItemDomainService.findMasIdByItemId(paramVO.getItemId());
            if (CollUtil.isEmpty(itemMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, itemMasIds);
        }
        paramVO.setIds(new ArrayList<>(masIds));
        return scpAllocSettingDomainService.queryAllocSettingList(paramVO);
    }

    @Override
    @SysCodeProc
    public ScpAllocSettingRespVO findAllocSettingById(Long id) {
        Optional<ScpAllocSettingRespVO> scpAllocSettingRespVO = scpAllocSettingDomainService.findAllocSettingById(id).map(ScpAllocSettingConvert.INSTANCE::dtoToRespVO);
        if (scpAllocSettingRespVO.isPresent()) {
            ScpAllocSettingRespVO allocSettingRespVO = scpAllocSettingRespVO.get();
            allocSettingRespVO.setAllStoreFlag(Boolean.FALSE);
            allocSettingRespVO.setItemRespVOList(scpAllocSettingItemDomainService.findByMasId(id));
            allocSettingRespVO.setBrandRespVOList(scpAllocSettingBrandDomainService.findByMasId(id));
            allocSettingRespVO.setRegionRespVOList(scpAllocSettingRegionDomainService.findByMasId(id));
            allocSettingRespVO.setStoreRespVOList(scpAllocSettingStoreDomainService.findByMasId(id));
            if (StrUtil.isNotBlank(allocSettingRespVO.getDocType())) {
                allocSettingRespVO.setDocTypes(StrUtil.split(allocSettingRespVO.getDocType(), ','));
            }
            return allocSettingRespVO;
        }
        return new ScpAllocSettingRespVO();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveAllocSetting(ScpAllocSettingSaveVO saveVO) {
        //校验参数
        checkParam(saveVO);
        if (saveVO.getId() == null) {
            if (CollUtil.isEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
                throw new BusinessException("门店不能为空");
            }
            String orderNo = rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.FA, new ArrayList<>());
            saveVO.setActivityCode(orderNo);
        } else {
            ScpAllocSettingStoreParamVO scpAllocSettingStoreParamVO = new ScpAllocSettingStoreParamVO();
            scpAllocSettingStoreParamVO.setMasIds(List.of(saveVO.getId()));
            scpAllocSettingStoreParamVO.setUsed(Boolean.TRUE);
            List<ScpAllocSettingStoreRespVO> byParam = scpAllocSettingStoreDomainService.findByParam(scpAllocSettingStoreParamVO);
            if (CollUtil.isNotEmpty(byParam)) {
                throw new BusinessException("活动已被引用，不允许编辑");
            }
        }
        //保存表头
        Long masId = scpAllocSettingDomainService.saveAllocSetting(saveVO);
        scpAllocSettingItemDomainService.deleteByMasId(masId);
        scpAllocSettingBrandDomainService.deleteByMasId(masId);
        scpAllocSettingRegionDomainService.deleteByMasId(masId);
        //保存活动区域
        saveVO.getRegionList().forEach(row -> row.setMasId(masId));
        scpAllocSettingRegionDomainService.batchSaveAllocSettingRegion(saveVO.getRegionList());
        //保存强配商品
        //查询商品信息
        List<String> itemCodes = saveVO.getItemList().stream().map(ScpAllocSettingItemSaveVO::getItemCode).distinct().collect(Collectors.toList());
        ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
        Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));
        saveVO.getItemList().forEach(row -> {
            row.setMasId(masId);
            if (itemRpcDTOMap.containsKey(row.getItemCode())) {
                row.setItemId(itemRpcDTOMap.get(row.getItemCode()).getId());
            }
        });
        scpAllocSettingItemDomainService.batchSaveAllocSettingItem(saveVO.getItemList());
        //保存品牌
        saveVO.getBrandList().forEach(row -> row.setMasId(masId));
        scpAllocSettingBrandDomainService.batchSaveAllocSettingBrand(saveVO.getBrandList());
        //保存门店
        scpAllocSettingStoreDomainService.deleteByMasId(masId);
        if (CollUtil.isNotEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
            saveVO.getStoreList().forEach(row -> {
                row.setMasId(masId);
                row.setId(null);
            });
            scpAllocSettingStoreDomainService.batchSaveAllocSettingStore(saveVO.getStoreList());
            refreshRedisValue(saveVO, masId, saveVO.getStoreList(), itemRpcDTOMap);
        }
        if (CollUtil.isEmpty(saveVO.getStoreList()) && saveVO.getAllStoreFlag()) {
            saveDefaultStore(saveVO, masId, itemRpcDTOMap);
        }
        String key = ScpConstant.ALLOC_SETTING + masId;
        if (LocalDateTime.now().compareTo(saveVO.getValidTo()) <= 0) {
            long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), saveVO.getValidTo());
            redisClient.opsForValue().set(key, saveVO.getValidFrom(), between, TimeUnit.MINUTES);
        }
        scpAllocSettingDomainService.enableAllocSetting(List.of(masId), Boolean.TRUE);
        return masId;
    }

    /**
     * 刷新强配活动缓存
     *
     * @param saveVO
     * @param masId
     */
    private void refreshRedisValue(ScpAllocSettingSaveVO saveVO, Long masId, List<ScpAllocSettingStoreSaveVO> storeList, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        Map<String, String> redisMap = new HashMap<>();
        List<ScpAllocSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
        for (ScpAllocSettingItemSaveVO item : saveVO.getItemList()) {
            ScpAllocSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpAllocSettingRedisDO.ActivityDO.InnerItemDO();
            innerItemDO.setItemCode(item.getItemCode());
            innerItemDO.setMasId(masId);
            innerItemDO.setMinNum(item.getMinNum());
            innerItemDO.setItemType2(item.getItemType2());
            if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                if (CollUtil.isNotEmpty(spuAttchmentList)) {
                    Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                    innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                }
                innerItemDO.setItemId(itmItemRpcDTO.getId());
                innerItemDO.setItemName(itmItemRpcDTO.getItemName());
            }
            innerItemDOS.add(innerItemDO);
        }
        for (ScpAllocSettingStoreSaveVO store : storeList) {
            ScpAllocSettingRedisDO scpAllocSettingRedisDO = new ScpAllocSettingRedisDO();
            Object obj = redisClient.opsForValue().get(ScpConstant.ALLOC_SETTING + store.getStoreCode());
            if (obj == null) {
                ScpAllocSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                scpAllocSettingRedisDO.setActivityList(List.of(activityDO1));
            } else {
                String str = (String) obj;
                var setting = JSON.parseObject(str, ScpAllocSettingRedisDO.class);
                List<ScpAllocSettingRedisDO.ActivityDO> activityList = setting.getActivityList();
                Boolean flag = Boolean.FALSE;
                for (ScpAllocSettingRedisDO.ActivityDO inActivityDO : activityList) {
                    if (inActivityDO.getActivityId().equals(masId)) {
                        inActivityDO.setActivityId(masId);
                        inActivityDO.setActivityCode(saveVO.getActivityCode());
                        inActivityDO.setDocTypes(saveVO.getDocTypes());
                        inActivityDO.setMaxNum(store.getMaxNum());
                        inActivityDO.setAllocNum(0);
                        inActivityDO.setValidTo(saveVO.getValidTo());
                        inActivityDO.setValidFrom(saveVO.getValidFrom());
                        inActivityDO.setInnerItemDOS(innerItemDOS);
                        flag = Boolean.TRUE;
                    }
                }
                if (!flag) {
                    ScpAllocSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                    activityList.add(activityDO1);
                }
                scpAllocSettingRedisDO.setActivityList(activityList);
            }
            String key = ScpConstant.ALLOC_SETTING + store.getStoreCode();
            redisMap.put(key, JSONUtil.toJsonStr(scpAllocSettingRedisDO));
        }
        redisClient.opsForValue().multiSet(redisMap);
    }

    // 提取重复代码
    private ScpAllocSettingRedisDO.ActivityDO createActivityDO(Long masId, ScpAllocSettingSaveVO saveVO, ScpAllocSettingStoreSaveVO store, List<ScpAllocSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS) {
        ScpAllocSettingRedisDO.ActivityDO activityDO = new ScpAllocSettingRedisDO.ActivityDO();
        activityDO.setInnerItemDOS(innerItemDOS);
        activityDO.setActivityId(masId);
        activityDO.setActivityCode(saveVO.getActivityCode());
        activityDO.setDocTypes(saveVO.getDocTypes());
        activityDO.setMaxNum(store.getMaxNum());
        activityDO.setAllocNum(0);
        activityDO.setValidFrom(saveVO.getValidFrom());
        activityDO.setValidTo(saveVO.getValidTo());
        return activityDO;
    }

    /**
     * 保存默认的门店配置
     *
     * @param saveVO 强配设置保存对象，包含区域、品牌等信息
     * @param masId  主配置ID
     *               <p>
     *               此方法主要用于保存SCP分配设置中的默认门店配置它首先校验最大强配次数是否为空，
     *               然后根据提供的区域和品牌列表查询符合条件的门店，并将这些门店信息保存到数据库中
     */
    private void saveDefaultStore(ScpAllocSettingSaveVO saveVO, Long masId, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        if (saveVO.getMaxAllocNum() == null) {
            throw new BusinessException("最大强配次数不能为空");
        }
        List<String> regionList = saveVO.getRegionList().stream().map(ScpAllocSettingRegionSaveVO::getRegion).distinct().collect(Collectors.toList());
        List<String> brandList = saveVO.getBrandList().stream().map(ScpAllocSettingBrandSaveVO::getBrand).distinct().collect(Collectors.toList());
        OrgStoreBasePageRpcParam orgStoreBasePageRpcParam = new OrgStoreBasePageRpcParam();
        orgStoreBasePageRpcParam.setRegionList(regionList);
        orgStoreBasePageRpcParam.setBrandCodeList(brandList);
        if (CollUtil.isEmpty(regionList) || CollUtil.isEmpty(brandList)) {
            log.info("saveDefaultStore品牌或者区域为空");
            return;
        }
        Long count = rmiOrgStoreRpcService.countBaseStoreByParam(orgStoreBasePageRpcParam);
        if (count == null || count == 0) {
            log.info("saveDefaultStore查询符合条件门店数据为空");
            return;
        }
        Integer totalPage = PageUtil.totalPage(count, 1000);
        for (int i = 1; i <= totalPage; i++) {
            orgStoreBasePageRpcParam.setSize(1000);
            orgStoreBasePageRpcParam.setCurrent(i);
            PagingVO<OrgStoreBaseRpcDTO> orgStoreBaseRpcDTOPagingVO = rmiOrgStoreRpcService.queryBaseStore(orgStoreBasePageRpcParam);
            List<ScpAllocSettingStoreSaveVO> storeSaveVOS = orgStoreBaseRpcDTOPagingVO.getRecords().stream().map(row -> {
                ScpAllocSettingStoreSaveVO scpAllocSettingStoreSaveVO = new ScpAllocSettingStoreSaveVO();
                scpAllocSettingStoreSaveVO.setStoreId(row.getId());
                scpAllocSettingStoreSaveVO.setStoreCode(row.getStoreCode());
                scpAllocSettingStoreSaveVO.setStoreName(row.getStoreName());
                scpAllocSettingStoreSaveVO.setStoreType2(row.getStoreType2());
                scpAllocSettingStoreSaveVO.setMasId(masId);
                scpAllocSettingStoreSaveVO.setMaxNum(saveVO.getMaxAllocNum());
                return scpAllocSettingStoreSaveVO;
            }).collect(Collectors.toList());
            scpAllocSettingStoreDomainService.batchSaveAllocSettingStore(storeSaveVOS, 1000);
            refreshRedisValue(saveVO, masId, storeSaveVOS, itemRpcDTOMap);
        }
    }

    /**
     * 校验参数
     *
     * @param saveVO
     */
    private static void checkParam(ScpAllocSettingSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getBrandList())) {
            throw new BusinessException("品牌不能为空");
        }
        if (CollUtil.isEmpty(saveVO.getRegionList())) {
            throw new BusinessException("区域不能为空");
        }
        if (CollUtil.isEmpty(saveVO.getItemList())) {
            throw new BusinessException("强配商品不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpAllocSettingDomainService.deleteByIds(ids);
        List<String> keys = ids.stream().map(row -> ScpConstant.ALLOC_SETTING + row).collect(Collectors.toList());
        redisClient.delete(keys);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeAllocSettingStatus(ScpAllocSettingStatusParamVO paramVO) {
        if (Boolean.TRUE.equals(paramVO.getStatus())) {
            //校验门店是否存在其他活动中
            ScpAllocSettingStoreParamVO scpAllocSettingStoreParamVO = new ScpAllocSettingStoreParamVO();
            scpAllocSettingStoreParamVO.setMasIds(paramVO.getIds());
            List<ScpAllocSettingStoreRespVO> storeRespVOList = scpAllocSettingStoreDomainService.findByParam(scpAllocSettingStoreParamVO);
            if (CollUtil.isEmpty(storeRespVOList)) {
                throw new BusinessException("门店信息未维护");
            }
            List<ScpAllocSettingDO> settingDOS = scpAllocSettingRepo.findByIdIn(paramVO.getIds());
            for (ScpAllocSettingDO settingDO : settingDOS) {
                String key = ScpConstant.ALLOC_SETTING + settingDO.getId();
                if (LocalDateTime.now().compareTo(settingDO.getValidTo()) <= 0) {
                    long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), settingDO.getValidTo());
                    redisClient.opsForValue().set(key, settingDO.getValidFrom(), between, TimeUnit.MINUTES);
                } else {
                    continue;
                }
                List<ScpAllocSettingItemRespVO> itemList = scpAllocSettingItemDomainService.findByMasId(settingDO.getId());
                //查询商品信息
                List<String> itemCodes = itemList.stream().map(ScpAllocSettingItemRespVO::getItemCode).distinct().collect(Collectors.toList());
                ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
                itmItemRpcDtoParam.setItemCodes(itemCodes);
                List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
                Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));

                List<ScpAllocSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
                for (ScpAllocSettingItemRespVO item : itemList) {
                    ScpAllocSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpAllocSettingRedisDO.ActivityDO.InnerItemDO();
                    innerItemDO.setItemCode(item.getItemCode());
                    innerItemDO.setMasId(settingDO.getId());
                    innerItemDO.setMinNum(item.getMinNum());
                    innerItemDO.setItemType2(item.getItemType2());
                    if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                        ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                        List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                        if (CollUtil.isNotEmpty(spuAttchmentList)) {
                            Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                            innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                        }
                        innerItemDO.setItemId(itmItemRpcDTO.getId());
                        innerItemDO.setItemName(itmItemRpcDTO.getItemName());
                    }
                    innerItemDOS.add(innerItemDO);
                }
                //写入缓存
                Map<String, String> redisMap = new HashMap<>();
                for (ScpAllocSettingStoreRespVO store : storeRespVOList) {
                    ScpAllocSettingRedisDO scpAllocSettingRedisDO = new ScpAllocSettingRedisDO();
                    Object obj = redisClient.opsForValue().get(ScpConstant.ALLOC_SETTING + store.getStoreCode());
                    if (obj == null) {
                        ScpAllocSettingRedisDO.ActivityDO activityDO1 = new ScpAllocSettingRedisDO.ActivityDO();
                        activityDO1.setActivityId(settingDO.getId());
                        activityDO1.setActivityCode(settingDO.getActivityCode());
                        activityDO1.setDocTypes(StrUtil.split(settingDO.getDocType(), ","));
                        activityDO1.setMaxNum(store.getMaxNum());
                        activityDO1.setAllocNum(0);
                        activityDO1.setInnerItemDOS(innerItemDOS);
                        scpAllocSettingRedisDO.setActivityList(List.of(activityDO1));
                        String storeKey = ScpConstant.ALLOC_SETTING + store.getStoreCode();
                        redisMap.put(storeKey, JSONUtil.toJsonStr(scpAllocSettingRedisDO));
                    }
                }
                if (CollUtil.isNotEmpty(redisMap)) {
                    redisClient.opsForValue().multiSet(redisMap);
                }
            }

        } else {
            List<String> keys = paramVO.getIds().stream().map(row -> ScpConstant.ALLOC_SETTING + row).collect(Collectors.toList());
            redisClient.delete(keys);
        }
        scpAllocSettingDomainService.enableAllocSetting(paramVO.getIds(), paramVO.getStatus());
    }

    private void intersectSets(Set<Long> set1, List<Long> list2) {
        if (set1.isEmpty()) {
            set1.addAll(list2);
        }
        set1.retainAll(list2);
    }

}
