package com.elitesland.scp.rmi;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.utils.ExceptionUtil;
import com.elitesland.support.provider.item.dto.*;
import com.elitesland.support.provider.item.param.*;
import com.elitesland.support.provider.item.service.ItmBrandRpcService;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/3/27 11:48
 * @Description:
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class RmiItemService {

    @Autowired
    private ItmItemRpcService itmItemRpcService;
    @Autowired
    private ItmBrandRpcService itmBrandRpcService;

    public List<ItmItemBusinessRpcDTO> findItmItemBusinessByBUCodeAndBrand(String ouCode, String brandCode) {
        ItmItemBusinessRpcDtoParam param = new ItmItemBusinessRpcDtoParam();
        if (StrUtil.isNotBlank(ouCode)) {
            param.setBuCodes(Collections.singletonList(ouCode));
        }
        if (StrUtil.isNotBlank(brandCode)) {
            param.setBrandList(Collections.singletonList(brandCode));
        }
        return selectItmItemBusinessByParam(param);
    }

    public List<ItmItemBusinessRpcDTO> findItmItemBusinessByBUCodeAndBrandAndItemList(String ouCode,
                                                                                      String brandCode,
                                                                                      List<String> itemCodeList) {
        log.info("根据品牌组织模板商品获取商品权限数据,参数:{},{},{}", ouCode, brandCode, JSONUtil.toJsonStr(itemCodeList));
        ItmItemBusinessRpcDtoParam param = new ItmItemBusinessRpcDtoParam();
        if (StrUtil.isNotBlank(ouCode)) {
            param.setBuCodes(Collections.singletonList(ouCode));
        }
        if (StrUtil.isNotBlank(brandCode)) {
            param.setBrandList(Collections.singletonList(brandCode));
        }
        if (CollUtil.isNotEmpty(itemCodeList)) {
            param.setItemCodes(itemCodeList);
        }
        return selectItmItemBusinessByParam(param);
    }

    public List<ItmItemBusinessRpcDTO> selectItmItemBusinessByParam(ItmItemBusinessRpcDtoParam param) {
        try {
            log.info("新经营目录列表入参,{}", JSON.toJSONString(param));
            List<ItmItemBusinessRpcDTO> dtos = itmItemRpcService.selectItmItemBusinessByParam(param);
            log.info("新经营目录列表查询结果:{}", JSONUtil.toJsonStr(dtos));
            return dtos;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("新经营目录列表服务异常:" + e.getMessage(), e);
            throw new BusinessException("新经营目录列表服务异常", e);
        }

    }

    public PagingVO<ItmItemBusinessRpcPagingDTO> selectItemBusinessRpcDtoByParam(ItmItemBusinessRpcPagingParam param) {
        try {
            log.info("新分页经营目录列表入参,{}", JSON.toJSONString(param));
            PagingVO<ItmItemBusinessRpcPagingDTO> dtos = itmItemRpcService.selectItemBusinessRpcDtoByParam(param);
            log.info("新分页经营目录列表查询结果:{}", JSONUtil.toJsonStr(dtos));
            return dtos;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("新分页查询经营目录列表服务异常:" + e.getMessage(), e);
            throw new BusinessException("新分页查询经营目录列表服务异常", e);
        }

    }

    public List<ItmItemBusinessRpcDTO> findItmItemBusinessByParam(ItmItemBusinessRpcDtoParam param) {
        try {
            log.info("经营目录列表入参,{}", JSON.toJSONString(param));
            List<ItmItemBusinessRpcDTO> dtos = itmItemRpcService.findItmItemBusinessByParam(param);
            log.info("经营目录列表查询结果:{}", JSONUtil.toJsonStr(dtos));
            return dtos;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("经营目录列表服务异常:" + e.getMessage());
            throw new BusinessException("经营目录列表服务异常", e);
        }

    }

    public PagingVO<ItmItemBusinessRpcPagingDTO> findItemBusinessRpcDtoByParam(ItmItemBusinessRpcPagingParam param) {
        try {
            log.info("分页经营目录列表入参,{}", JSON.toJSONString(param));
            PagingVO<ItmItemBusinessRpcPagingDTO> dtos = itmItemRpcService.findItemBusinessRpcDtoByParam(param);
            log.info("分页经营目录列表查询结果:{}", JSONUtil.toJsonStr(dtos));
            return dtos;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("分页查询经营目录列表服务异常:" + e.getMessage(), e);
            throw new BusinessException("分页查询经营目录列表服务异常", e);
        }

    }


    public List<ItmItemBaseRpcDTO> findItemBaseRpcDtoByParam(ItmItemBaseRpcParam param) {
        try {
            log.info("获取商品基础信息列表入参,{}", JSON.toJSONString(param));
            List<ItmItemBaseRpcDTO> itemBaseRpcDtoByParam = itmItemRpcService.findItemBaseRpcDtoByParam(param);
            log.info("获取商品基础信息列表查询结果:{}", JSONUtil.toJsonStr(itemBaseRpcDtoByParam));
            return itemBaseRpcDtoByParam;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("获取商品基础信息列表服务异常:" + e.getMessage());
            throw new BusinessException("获取商品基础信息列表服务异常", e);
        }
    }

    public List<ItmItemUomConvRpcDTO> findItemUomConvDtoByParam(ItmItemUomConvRpcDtoParam param) {
        try {
            log.info("获取商品单位转化率入参,{}", JSON.toJSONString(param));
            List<ItmItemUomConvRpcDTO> result = itmItemRpcService.findItemUomConvDtoByParam(param);
            log.info("获取商品单位转化率查询结果:{}", JSONUtil.toJsonStr(result));
            return result;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("获取商品单位转化率服务异常:" + e.getMessage());
            throw new BusinessException("获取商品单位转化率服务异常", e);
        }
    }

    /**
     * 查询商品图片
     *
     * @param itemIds ->supID
     * @return
     */
    public List<ItmItemAttachmentProviderDTO> findSkuImgByItemIds(List<Long> itemIds) {
        try {
            log.info("商品图片查询入参,{}", JSON.toJSONString(itemIds));
            List<ItmItemAttachmentProviderDTO> skuImgByItemIds = itmItemRpcService.findSkuImgByItemIds(itemIds);
            log.info("商品图片查询结果:{}", JSONUtil.toJsonStr(skuImgByItemIds));
            return skuImgByItemIds;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("商品图片查询服务异常:" + e.getMessage());
            throw new BusinessException("商品图片查询服务异常", e);
        }
    }

    /**
     * 查询商品基本信息-小程序列表，详情使用
     *
     * @param param
     * @return
     */
    public List<ItmItemScpBaseRpcDTO> findItemScpBaseRpcDtoByParam(ItmItemScpBaseRpcParam param) {
        try {
            log.info("小程序-商品基本信息查询入参,{}", JSON.toJSONString(param));
            List<ItmItemScpBaseRpcDTO> itmItemScpBaseRpcDTO = itmItemRpcService.findItemScpBaseRpcDtoByParam(param);
            log.info("小程序--商品基本信息查询结果:{}", JSONUtil.toJsonStr(itmItemScpBaseRpcDTO));
            return itmItemScpBaseRpcDTO;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("小程序-商品基本信息查询服务异常:" + e.getMessage(), e);
            throw new BusinessException("小程序-商品基本信息查询服务异常", e);
        }
    }

    /**
     * 查询商品信息
     *
     * @param param
     * @return
     */
    public List<ItmItemRpcDTO> findItemRpcDtoByParam(ItmItemRpcDtoParam param) {
        try {
            log.info("商品信息查询入参,{}", JSON.toJSONString(param));
            List<ItmItemRpcDTO> itemRpcDtoByParam = itmItemRpcService.findItemRpcDtoByParam(param);
            log.info("商品信息查询结果:{}", JSONUtil.toJsonStr(itemRpcDtoByParam));
            return itemRpcDtoByParam;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("商品信息查询服务异常:" + e.getMessage());
            throw new BusinessException("商品信息查询服务异常", e);
        }
    }

    /**
     * 根据itemids查询商品简要信息
     *
     * @param itemIds
     * @return
     */
    public List<ItmItemLotRpcDTO> findLotRpcDTOS(List<Long> itemIds) {
        try {
            log.info("商品简要信息查询入参,{}", JSON.toJSONString(itemIds));
            List<ItmItemLotRpcDTO> lotRpcDTOS = itmItemRpcService.findLotRpcDTOS(itemIds);
            log.info("商品简要信息查询结果:{}", JSONUtil.toJsonStr(lotRpcDTOS));
            return lotRpcDTOS;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("商品信息查询服务异常:" + e.getMessage());
            throw new BusinessException("商品信息查询服务异常", e);
        }
    }

    /**
     * 根据itemids查询商品简要信息
     *
     * @param itemIds
     * @return
     */
    public Map<Long, ItmItemIcDTO> findSimpleItemMap(List<Long> itemIds) {
        try {
            log.info("商品简要信息查询入参,{}", JSON.toJSONString(itemIds));
            List<ItmItemIcDTO> simpleItem = itmItemRpcService.findIcItem(itemIds);
            log.info("商品简要信息查询结果:{}", JSONUtil.toJsonStr(simpleItem));
            return simpleItem.stream().collect(Collectors.toMap(ItmItemIcDTO::getId, Function.identity()));
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("商品信息查询服务异常:" + e.getMessage());
            throw new BusinessException("商品信息查询服务异常", e);
        }
    }

    public Map<Long, ItmItemScpBaseRpcDTO> findScpItemMap(List<Long> itemIds) {
        ItmItemScpBaseRpcParam itmItemScpBaseRpcParam = new ItmItemScpBaseRpcParam();
        itmItemScpBaseRpcParam.setItemIds(itemIds);
        List<ItmItemScpBaseRpcDTO> itmItemScpBaseList = itmItemRpcService.findItemScpBaseRpcDtoByParam(itmItemScpBaseRpcParam);
        if (CollUtil.isEmpty(itmItemScpBaseList)) {
            return new HashMap<>();
        }
        return itmItemScpBaseList.stream().collect(Collectors.toMap(ItmItemScpBaseRpcDTO::getId, Function.identity()));
    }

    public Map<Long, ItmItemSimpleRpcDTO> findBaseItemMap(List<Long> itemIds) {
        List<ItmItemSimpleRpcDTO> itmItemScpBaseList = itmItemRpcService.findSimpleItem(itemIds);
        if (CollUtil.isEmpty(itmItemScpBaseList)) {
            return new HashMap<>();
        }
        return itmItemScpBaseList.stream().collect(Collectors.toMap(ItmItemSimpleRpcDTO::getId, Function.identity()));
    }

    public Map<Long, List<String>> findRootBrandByCateId(Long cateId) {
        try {
            log.info("商品品牌查询入参,{}", JSON.toJSONString(cateId));
            Map<Long, List<String>> itemBrand = itmItemRpcService.findRootBrandByCateId(cateId);
            log.info("商品品牌查询结果:{}", JSONUtil.toJsonStr(itemBrand));
            return itemBrand;
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("商品品牌查询服务异常:" + e.getMessage());
            throw new BusinessException("商品品牌查询服务异常", e);
        }
    }

    public String getFirstCateByCodes(List<String> codes) {
        try {
            return itmItemRpcService.findFirstCateByCodes(codes);
        } catch (Exception e) {
            ExceptionUtil.handleRootException(e);
            log.error("获取首选分类异常:" + e.getMessage(), e);
            throw new BusinessException("获取首选分类服务异常", e);
        }
    }

    public Map<String, List<ItmItemSpuBusinessRpcDTO>> findMoqByParam(List<String> spuCodes, String buCode) {
        if (StrUtil.isBlank(buCode)) {
            log.info("查询商品最小起订量，buCode为空");
            return new HashMap<>();
        }
        ItmItemBusinessRpcDtoParam businessRpcDtoParam = new ItmItemBusinessRpcDtoParam();
        businessRpcDtoParam.setSpuCodes(spuCodes);
        businessRpcDtoParam.setBusinessBuCode(buCode);
        List<ItmItemSpuBusinessRpcDTO> itemBusinessMoqList = itmItemRpcService.findItemBusinessMoq(businessRpcDtoParam);
        if (CollUtil.isNotEmpty(itemBusinessMoqList)) {
            return itemBusinessMoqList.stream().collect(Collectors.groupingBy(ItmItemSpuBusinessRpcDTO::getSpuCode));
        }
        return new HashMap<>();
    }

    public Map<String, ItmBrandRpcDTO> getBrandMap(List<String> brandCodes) {
        if (CollUtil.isEmpty(brandCodes)) {
            return new HashMap<>();
        }
        ItmBrandRpcDtoParam itmBrandRpcDtoParam = new ItmBrandRpcDtoParam();
        itmBrandRpcDtoParam.setBrandCodes(brandCodes);
        List<ItmBrandRpcDTO> brandDtoByParam = itmBrandRpcService.findBrandDtoByParam(itmBrandRpcDtoParam);
        if (CollUtil.isNotEmpty(brandDtoByParam)) {
            return brandDtoByParam.stream().collect(Collectors.toMap(ItmBrandRpcDTO::getBrandCode, Function.identity()));
        }
        return new HashMap<>();
    }

}
