package com.elitesland.scp.application.service.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.threadpool.ThreadPoolAutoConfiguration;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcDTO;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcParam;
import com.elitesland.inv.enums.InvDeter2TypeEnum;
import com.elitesland.scp.application.facade.vo.param.app.AppItemActivityItemPriceParamVO;
import com.elitesland.scp.application.facade.vo.resp.app.AppInvStkRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.service.supalloc.ScpSupplyAllocationService;
import com.elitesland.scp.application.service.whnet.ScpWhNetRelationService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.cart.ScpStoreCartDO;
import com.elitesland.scp.dto.supalloc.ScpSupplyAllocationRpcDTO;
import com.elitesland.scp.dto.whnet.ScpWhNetRelationRpcDTO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiPriceRpcService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemSimpleRpcDTO;
import com.elitesland.support.provider.pri.service.dto.PriPriceRpcDTO;
import com.elitesland.support.provider.pri.service.param.ItmBomPriceRpcDtoParam;
import com.elitesland.support.provider.pri.service.param.ItmPriPriceRpcDtoParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class CommonPriceServiceImpl implements CommonPriceService {
    @Autowired
    @Qualifier(ThreadPoolAutoConfiguration.BEAN_NAME)
    private TaskExecutor taskExecutor;
    private final RmiPriceRpcService rmiPriceRpcService;
    private final RmiItemService rmiItemService;
    private final ScpSupplyAllocationService scpSupplyAllocationService;
    private final ScpWhNetRelationService scpWhNetRelationService;
    private final RmiInvStkRpcService rmiInvStkRpcService;

    @Override
    public Map<String, List<PriPriceRpcDTO>> getPriceAndStockDetails(List<AppItemActivityItemPriceParamVO> paramVOList, String storeCode, ScpOrderSettingRespVO orderSetting, String type,
                                                                     AtomicReference<Map<String, BigDecimal>> stockMap) {
        List<PriPriceRpcDTO> priceList = CollUtil.newCopyOnWriteArrayList(new ArrayList<>());
        List<ItmPriPriceRpcDtoParam> priceParamList = new ArrayList<>();

        // 查询子件价格参数列表
        List<ItmBomPriceRpcDtoParam> bomPriceParams = new ArrayList<>();

        //获取仓网供应关系
        List<ScpStoreCartDO> dataList = paramVOList.stream().map(row -> {
            ScpStoreCartDO storeCartDO = new ScpStoreCartDO();
            storeCartDO.setItemCode(row.getItemCode());
            storeCartDO.setItemCateCode(row.getItemCateCode());
            return storeCartDO;
        }).collect(Collectors.toList());
        Map<String, List<ScpWhNetRelationRpcDTO>> whNetMap = scpWhNetRelationService.findWhNetByParam(storeCode, type, dataList);
        //获取供应商份额
        List<String> itemCodes = paramVOList.stream().map(AppItemActivityItemPriceParamVO::getItemCode).distinct().collect(Collectors.toList());
        Map<String, List<ScpSupplyAllocationRpcDTO>> supplyAllocationMap = scpSupplyAllocationService.findSupplyAllocationByParam(storeCode, type, itemCodes);
        List<String> priorities = Arrays.asList(orderSetting.getFirstPriority(), orderSetting.getSecPriority());
        List<AppInvStkRespVO> suppPriorityList = new ArrayList<>();
        List<AppInvStkRespVO> whPriorityList = new ArrayList<>();
        // 优先级查询价格
        for (AppItemActivityItemPriceParamVO activity : paramVOList) {
            AppInvStkRespVO stkParam = new AppInvStkRespVO();
            for (String priority : priorities) {
                ItmPriPriceRpcDtoParam priceParam = null;
                if (ScpUdcEnum.ORDER_PRIORITY_SUPALLOC.getValueCode().equals(priority)) {
                    // 供应商份额配置销售价格
                    priceParam = buildSuppPriceParamList(supplyAllocationMap, activity, priceList, stkParam);
                    if (stkParam.getItemId() != null) {
                        suppPriorityList.add(stkParam);
                    }
                } else if (ScpUdcEnum.ORDER_PRIORITY_WHNET.getValueCode().equals(priority)) {
                    // 仓网关系配置销售价格
                    priceParam = buildWhPriceParamList(whNetMap, activity, priceList, stkParam);
                    if (stkParam.getItemId() != null) {
                        whPriorityList.add(stkParam);
                    }
                }
                if (priceParam != null && priceParam.getItemCode() == null) {
                    break;
                }
                if (priceParam != null && priceParam.getItemCode() != null) {
                    priceParamList.add(priceParam);
                    break;
                }
            }
            if(Boolean.TRUE.equals(activity.getCombineItemFlag())){
                //组套子件商品 价格参数组装
                ItmBomPriceRpcDtoParam bomPriceParam = new ItmBomPriceRpcDtoParam();
                bomPriceParam.setItemCode(activity.getItemCode());
                bomPriceParam.setCombineItemCode(activity.getCombineItemCode());
                bomPriceParam.setUom(activity.getUom());
                bomPriceParams.add(bomPriceParam);
            }
        }
        if (CollUtil.isEmpty(priceParamList) && CollUtil.isNotEmpty(priceList)) {
            // 构建库存查询参数
            queryInvStock(suppPriorityList, stockMap, whPriorityList);
            return priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
        }
        if (CollUtil.isEmpty(priceParamList) && CollUtil.isEmpty(priceList)) {
            return new HashMap<>();
        }
        if (stockMap == null) {
            return getItemPrice(priceParamList, priceList);
        }

        // 查询组套商品子件价格
        Map<String, List<PriPriceRpcDTO>> bomPriceMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(bomPriceParams)){
            List<PriPriceRpcDTO> bomPriceList = rmiPriceRpcService.findBomAndMainItemPriceByParam(bomPriceParams);
            if (CollectionUtils.isNotEmpty(bomPriceList)) {
                try {
                    bomPriceMap = bomPriceList.stream().collect(Collectors.groupingBy(c -> SysUtils.getItemKey(c.getCombineItemCode(), c.getItemCode())));
                } catch (Exception e) {
                    log.info("查询子件和主件价格报错：", e);
                    throw new BusinessException("查询子件和主件价格报错：" + e);
                }
            }
        }

        CompletableFuture<Map<String, List<PriPriceRpcDTO>>> future1 = CompletableFuture.supplyAsync(() -> getItemPrice(priceParamList, priceList), taskExecutor);
        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> queryInvStock(suppPriorityList, stockMap, whPriorityList), taskExecutor);

        CompletableFuture.allOf(future1, future2).join();
        try {
            bomPriceMap.putAll(future1.get());
            return bomPriceMap;
        } catch (Exception e) {
            log.info("查询价格报错：{}", e);
            throw new BusinessException("查询价格报错：" + e.toString());
        }
    }

    private void queryInvStock(List<AppInvStkRespVO> suppPriorityList, AtomicReference<Map<String, BigDecimal>> stockMap,
                               List<AppInvStkRespVO> whPriorityList) {
        if (stockMap == null) {
            return;
        }
        // 构建库存查询参数
        for (AppInvStkRespVO stk : suppPriorityList) {
            stockMap.updateAndGet(currentMap -> {
                currentMap.put(stk.getItemCode(), new BigDecimal("99999"));
                return currentMap;
            });
        }
        InvWhItemTotalStkRpcParam stockParam = buildStockParam(whPriorityList);
        getWhStockMap(stockParam, stockMap, whPriorityList);
    }

    private InvWhItemTotalStkRpcParam buildStockParam(List<AppInvStkRespVO> whPriorityList) {
        if (CollUtil.isNotEmpty(whPriorityList)) {
            List<Long> itemIds = whPriorityList.stream().map(AppInvStkRespVO::getItemId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<Long> whIds = whPriorityList.stream().map(AppInvStkRespVO::getWhId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            InvWhItemTotalStkRpcParam invWhItemTotalStkRpcParam = new InvWhItemTotalStkRpcParam();
            invWhItemTotalStkRpcParam.setItemIds(itemIds);
            invWhItemTotalStkRpcParam.setWhIds(whIds);
            invWhItemTotalStkRpcParam.setExcludeDeter2Types(List.of(InvDeter2TypeEnum.TRANS.getType()));
            return invWhItemTotalStkRpcParam;
        }
        return null;
    }

    private void getWhStockMap(InvWhItemTotalStkRpcParam stkRpcParam, AtomicReference<Map<String, BigDecimal>> stockMap, List<AppInvStkRespVO> whPriorityList) {
        if (stkRpcParam == null) {
            return;
        }

        Map<String, List<Long>> itemCodeAndWhId = whPriorityList.stream().collect(Collectors.groupingBy(AppInvStkRespVO::getItemCode, Collectors.mapping(AppInvStkRespVO::getWhId, Collectors.toList())));
        List<InvWhItemTotalStkRpcDTO> invWhItemTotalStkRpcDTOS = rmiInvStkRpcService.queryInvWhItemTotalStk(stkRpcParam);
        if (CollUtil.isNotEmpty(invWhItemTotalStkRpcDTOS)) {
            for (InvWhItemTotalStkRpcDTO dto : invWhItemTotalStkRpcDTOS) {
                List<Long> longs = itemCodeAndWhId.get(dto.getItemCode());
                if (longs.contains(dto.getWhId())) {
                    stockMap.updateAndGet(currentMap -> {
                        currentMap.put(dto.getItemCode(), dto.getAvalQty2());
                        return currentMap;
                    });
                }
            }
        }
    }

    private Map<String, List<PriPriceRpcDTO>> getItemPrice(List<ItmPriPriceRpcDtoParam> priceParamList, List<PriPriceRpcDTO> priceList) {
        List<ItmPriPriceRpcDtoParam> itmPriPriceParamList = BeanUtils.copyToList(priceParamList, ItmPriPriceRpcDtoParam.class);
        List<PriPriceRpcDTO> priceByParam = rmiPriceRpcService.findPriceByParam(itmPriPriceParamList);
        // 未查询到价格,查询基本单位配置的价格
        if (CollUtil.isNotEmpty(priceByParam)) {
            priceList.addAll(priceByParam);
            //查询基本单位价格
            Map<String, List<PriPriceRpcDTO>> inPriceMap =
                    priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
            priceParamList.removeAll(priceParamList.stream().filter(row -> inPriceMap.containsKey(row.getItemCode())).collect(Collectors.toList()));
            getBasePrice(priceParamList, priceList);
        } else {
            //查询基本单位价格
            getBasePrice(priceParamList, priceList);
        }
        Map<String, List<PriPriceRpcDTO>> priceMap =
                priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
        log.info("批量查询商品价格信息:{}", JSONUtil.toJsonStr(priceMap));
        return priceMap;
    }

    private void getBasePrice(List<ItmPriPriceRpcDtoParam> priceParamList, List<PriPriceRpcDTO> priceList) {
        if (CollUtil.isEmpty(priceParamList)) {
            return;
        }
        //查询商品
        List<Long> itemIds = priceParamList.stream().map(ItmPriPriceRpcDtoParam::getItemId).distinct().collect(Collectors.toList());
        Map<Long, ItmItemSimpleRpcDTO> simpleItemMap = rmiItemService.findBaseItemMap(itemIds);

        List<ItmPriPriceRpcDtoParam> baseUomList = priceParamList.stream().map(row -> {
            ItmPriPriceRpcDtoParam param = BeanUtils.copyProperties(row, ItmPriPriceRpcDtoParam.class);
            ItmItemSimpleRpcDTO simpleItem = simpleItemMap.get(row.getItemId());
            param.setUom(simpleItem.getUom());
            return param;
        }).collect(Collectors.toList());
        List<PriPriceRpcDTO> basePriceList = rmiPriceRpcService.findPriceByParam(baseUomList);
        log.info("查询基本单位价格:{}", JSONUtil.toJsonStr(basePriceList));
        if (CollUtil.isNotEmpty(basePriceList)) {
            List<PriPriceRpcDTO> resultPriceList = basePriceList.stream().map(row -> {
                PriPriceRpcDTO priPriceRpcDTO = BeanUtils.copyProperties(row, PriPriceRpcDTO.class);
                ItmItemSimpleRpcDTO simpleItem = simpleItemMap.get(row.getItemId());
                if (simpleItem.getUomRatio2() == null) {
                    return null;
                }
                if (simpleItem.getDecimalPlaces() != null) {
                    priPriceRpcDTO.setPrice(row.getPrice().multiply(simpleItem.getUomRatio2()).setScale(simpleItem.getDecimalPlaces(), BigDecimal.ROUND_HALF_UP));
                } else {
                    priPriceRpcDTO.setPrice(row.getPrice().multiply(simpleItem.getUomRatio2()));
                }
                return priPriceRpcDTO;
            }).filter(Objects::nonNull).collect(Collectors.toList());
            priceList.addAll(resultPriceList);
        }
    }

    /**
     * 供应商份额销售价格
     *
     * @param suppAllocMap
     * @param item
     * @param priceList
     * @return
     */
    private ItmPriPriceRpcDtoParam buildSuppPriceParamList(Map<String, List<ScpSupplyAllocationRpcDTO>> suppAllocMap, AppItemActivityItemPriceParamVO item, List<PriPriceRpcDTO> priceList,
                                                           AppInvStkRespVO stkParam) {
        // 结算价格
        List<ScpSupplyAllocationRpcDTO> suppList = suppAllocMap.get(item.getItemCode());
        // 未查询到供应商配额,属于没有配置价格
        if (CollUtil.isEmpty(suppList)) {
            return null;
        }

        //库存查询参数
        stkParam.setItemId(item.getItemId());
        stkParam.setItemCode(item.getItemCode());

        ScpSupplyAllocationRpcDTO supp = suppList.get(0);
        // 供应商未配置销售OU,属于零价格
        if (StrUtil.isBlank(supp.getSaleOuCode())) {
            PriPriceRpcDTO priPriceRpcDTO = new PriPriceRpcDTO();
            priPriceRpcDTO.setItemId(item.getItemId());
            priPriceRpcDTO.setItemCode(item.getItemCode());
            priPriceRpcDTO.setPrice(BigDecimal.ZERO);
            priPriceRpcDTO.setNetPrice(BigDecimal.ZERO);
            priceList.add(priPriceRpcDTO);
            return new ItmPriPriceRpcDtoParam();
        }
        ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
        priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setItemId(item.getItemId());
        priceParam.setCustCode(supp.getSaleCustCode());
        priceParam.setItemCode(item.getItemCode());
        priceParam.setUom(item.getUom());
        priceParam.setCurrCode(ScpConstant.CNY);
        priceParam.setOuCode(supp.getSaleOuCode());
        return priceParam;
    }

    /**
     * 仓网关系销售价格
     *
     * @param whNetMap
     * @param item
     * @param priceList
     * @return
     */
    private ItmPriPriceRpcDtoParam buildWhPriceParamList(Map<String, List<ScpWhNetRelationRpcDTO>> whNetMap, AppItemActivityItemPriceParamVO item,
                                                         List<PriPriceRpcDTO> priceList, AppInvStkRespVO stkParam) {
        List<ScpWhNetRelationRpcDTO> whNetList = whNetMap.get(item.getItemCode());
        // 未查询到仓网关系,属于没有配置价格
        if (CollUtil.isEmpty(whNetList)) {
            return null;
        }
        ScpWhNetRelationRpcDTO supp = whNetList.get(0);
        //库存查询参数
        stkParam.setItemId(item.getItemId());
        stkParam.setItemCode(item.getItemCode());
        stkParam.setWhId(supp.getSupplyWhId());
        // 仓网关系未配置销售OU,属于零价格
        if (StrUtil.isBlank(supp.getOuCode())) {
            PriPriceRpcDTO priPriceRpcDTO = new PriPriceRpcDTO();
            priPriceRpcDTO.setItemId(item.getItemId());
            priPriceRpcDTO.setItemCode(item.getItemCode());
            priPriceRpcDTO.setPrice(BigDecimal.ZERO);
            priPriceRpcDTO.setNetPrice(BigDecimal.ZERO);
            priceList.add(priPriceRpcDTO);
            return new ItmPriPriceRpcDtoParam();
        }
        ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
        priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setItemId(item.getItemId());
        priceParam.setCustCode(supp.getCustCode());
        priceParam.setItemCode(item.getItemCode());
        priceParam.setUom(item.getUom());
        priceParam.setCurrCode(ScpConstant.CNY);
        priceParam.setOuCode(supp.getOuCode());
        return priceParam;
    }
}
