package com.elitesland.scp.application.service.mrp;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.threadpool.ThreadPoolAutoConfiguration;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.invIo.InvIoCalcParamRpcDTO;
import com.elitesland.inv.dto.invIo.InvIoCalcRpcDTO;
import com.elitesland.pur.dto.po.PurPoDSaveDTO;
import com.elitesland.pur.dto.po.PurPoRouteParamDTO;
import com.elitesland.pur.dto.po.PurPoRouteRpcDTO;
import com.elitesland.pur.dto.po.PurPoSaveDTO;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcDTO;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcParam;
import com.elitesland.pur.provider.PurPoProvider;
import com.elitesland.pur.provider.PurSuppProvider;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpMrpPageParam;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDRespVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDetailRespVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpRespVO;
import com.elitesland.scp.application.facade.vo.stock.ScpPredictStStockRespVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.mrp.ScpMrpDDO;
import com.elitesland.scp.domain.entity.mrp.ScpMrpDO;
import com.elitesland.scp.domain.entity.mrp.ScpMrpDPlanDO;
import com.elitesland.scp.domain.entity.stock.ScpPredictStStockCalcDO;
import com.elitesland.scp.domain.entity.supalloc.ScpSupplyAllocationDO;
import com.elitesland.scp.domain.service.mrp.ScpMrpDDomainService;
import com.elitesland.scp.domain.service.mrp.ScpMrpDPlanDomainService;
import com.elitesland.scp.domain.service.mrp.ScpMrpDomainService;
import com.elitesland.scp.domain.service.supalloc.ScpSupplyAllocationDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.repo.stock.ScpPredictStStockCalcRepo;
import com.elitesland.scp.infr.repo.stock.ScpPredictStStockRepoProc;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgOuService;
import com.elitesland.scp.rmi.RmiSysUserRpcService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.scp.utils.TransactionCommitHandler;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemUomConvRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import com.elitesland.support.provider.item.param.ItmItemBusinessRpcDtoParam;
import com.elitesland.support.provider.item.param.ItmItemUomConvRpcDtoParam;
import com.elitesland.support.provider.item.param.ItmItemUomConvRpcOtherParam;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpMrpServiceImpl implements ScpMrpService {

    private final ScpMrpDomainService scpMrpDomainService;

    private final ScpMrpDDomainService scpMrpDDomainService;

    private final ScpMrpDPlanDomainService scpMrpDPlanDomainService;

    private final PurPoProvider purPoProvider;

    private final RmiOrgOuService rmiOrgOuService;

    private final ScpPredictStStockCalcRepo scpPredictStStockCalcRepo;

    private final ScpPredictStStockRepoProc scpPredictStStockRepoProc;

    private final RmiInvStkRpcService rmiInvStkRpcService;

    private final RmiItemService rmiItemService;

    private final RmiSysUserRpcService rmiSysUserRpcService;

    private final TransactionCommitHandler transactionCommitHandler;

    private final ScpSupplyAllocationDomainService scpSupplyAllocationDomainService;

    private final TransactionTemplate transactionTemplate;

    private final UdcProvider udcProvider;

    private final PurSuppProvider purSuppProvider;

    @Autowired
    @Qualifier(ThreadPoolAutoConfiguration.BEAN_NAME)
    private TaskExecutor taskExecutor;

    @Override
    public PagingVO<ScpMrpRespVO> page(ScpMrpPageParam param) {
        PagingVO<ScpMrpRespVO> pagingVO = scpMrpDomainService.searchPage(param);
        if (pagingVO.isNotEmpty()) {
            List<Long> ids = pagingVO.getRecords().stream().map(ScpMrpRespVO::getId).collect(Collectors.toList());
            List<ScpMrpDRespVO> scpMrpDRespVOList = scpMrpDDomainService.findByMasIds(ids);
            if (CollectionUtil.isNotEmpty(scpMrpDRespVOList)) {
                Map<Long, List<ScpMrpDRespVO>> mrpDMap = scpMrpDRespVOList.stream().collect(Collectors.groupingBy(ScpMrpDRespVO::getMasId));
                pagingVO.getRecords().forEach(record -> {
                    List<ScpMrpDRespVO> scpMrpDRespVOS = mrpDMap.get(record.getId());
                    if (CollectionUtil.isNotEmpty(scpMrpDRespVOS)) {
                        record.setOuName(scpMrpDRespVOS.stream().map(ScpMrpDRespVO::getOuName).distinct().collect(Collectors.joining("；")));
                    }
                });
            }
        }
        return pagingVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            throw new BusinessException("ids不能为空");
        }
        List<ScpMrpDO> scpMrpDOList = scpMrpDomainService.findByIds(ids);
        if (CollectionUtil.isEmpty(scpMrpDOList)) {
            throw new BusinessException("MRP计划不存在，请检查");
        }
        List<String> wrongMrpLotNos = scpMrpDOList.stream().filter(mrp -> !ScpUdcEnum.MRP_PUSH_STATUS_NOT.getValueCode().equals(mrp.getPushStatus())).map(ScpMrpDO::getMrpLotNo).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(wrongMrpLotNos)) {
            throw new BusinessException("只能删除未推送的MRP计划，请检查：" + String.join(",", wrongMrpLotNos));
        }
        scpMrpDomainService.deleteByIds(ids);

        // 明细ids
        List<ScpMrpDRespVO> scpMrpDRespVOList = scpMrpDDomainService.findByMasIds(ids);
        if (CollectionUtil.isNotEmpty(scpMrpDRespVOList)) {
            List<Long> mrpDIds = scpMrpDRespVOList.stream().map(ScpMrpDRespVO::getId).collect(Collectors.toList());
            List<Long> predIds = scpMrpDRespVOList.stream().map(ScpMrpDRespVO::getRelateDocDid).collect(Collectors.toList());

            // 删除明细和采购计划
            scpMrpDDomainService.deleteByMasIds(ids);
            if (CollectionUtil.isNotEmpty(mrpDIds)) {
                scpMrpDPlanDomainService.deleteByMasIds(mrpDIds);
            }
            // 回写预测安全库存MRP标记
            scpPredictStStockRepoProc.updateMrpFlag(predIds, Boolean.FALSE);
        }


    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void pushPo(Long id) {
        if (id == null) {
            throw new BusinessException("id不能为空");
        }
        ScpMrpDO scpMrpDO = scpMrpDomainService.findById(id);
        if (scpMrpDO == null) {
            throw new BusinessException("MRP计划不存在，请检查");
        }
        if (ScpUdcEnum.MRP_PUSH_STATUS_COMPLETE.getValueCode().equals(scpMrpDO.getPushStatus())) {
            throw new BusinessException("MRP计划" + scpMrpDO.getMrpLotNo() + "已推送完成，请勿重复推送");
        }
        List<ScpMrpDetailRespVO> mrpDetailRespVOS = scpMrpDDomainService.findDetailByMasId(id);
        if (CollectionUtil.isEmpty(mrpDetailRespVOS)) {
            throw new BusinessException("MRP计划" + scpMrpDO.getMrpLotNo() + "采购计划明细为空，请检查");
        }
        mrpDetailRespVOS = mrpDetailRespVOS.stream().filter(d -> ScpUdcEnum.MRP_D_PUSH_STATUS_NOT.getValueCode().equals(d.getPushStatus())
                || ScpUdcEnum.MRP_D_PUSH_STATUS_FAIL.getValueCode().equals(d.getPushStatus())).collect(Collectors.toList());

        if (CollectionUtil.isEmpty(mrpDetailRespVOS)) {
            throw new BusinessException("MRP计划" + scpMrpDO.getMrpLotNo() + "已推送完成，请勿重复推送");
        }

        Collection<List<ScpMrpDetailRespVO>> mrpDetailGroup = mrpDetailRespVOS.stream().collect(Collectors.groupingBy(d -> d.getPurOuCode() + d.getSuppCode() + d.getWhCode())).values();
        List<String> purOuCodes = mrpDetailRespVOS.stream().map(ScpMrpDetailRespVO::getPurOuCode).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        Map<String, OrgOuRpcSimpleDTO> ouCodeBuIdMap = rmiOrgOuService.findOuDtoMapByOuCodes(purOuCodes);

        List<String> suppCodes = mrpDetailRespVOS.stream().map(ScpMrpDetailRespVO::getSuppCode).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        PurSuppBaseRpcParam purSuppBaseRpcParam = new PurSuppBaseRpcParam();
        purSuppBaseRpcParam.setSuppCodes(suppCodes);
        Map<String, PurSuppBaseRpcDTO> suppMap = purSuppProvider.findBaseRpcDtoByParam(purSuppBaseRpcParam).getData().stream().collect(Collectors.toMap(PurSuppBaseRpcDTO::getSuppCode, s -> s));

        List<PurPoSaveDTO> purPoSaveDTOList = new ArrayList<>();
        mrpDetailGroup.forEach(group -> {
            Collection<List<ScpMrpDetailRespVO>> mrpPlanMap = group.stream().collect(Collectors.groupingBy(ScpMrpDetailRespVO::getItemCode)).values();
            ScpMrpDetailRespVO scpMrpDetailRespVO = group.get(0);
            PurPoSaveDTO purPoSaveDTO = new PurPoSaveDTO();
            purPoSaveDTO.setOuCode(scpMrpDetailRespVO.getPurOuCode());
            purPoSaveDTO.setWhCode(scpMrpDetailRespVO.getWhCode());
            purPoSaveDTO.setSuppCode(scpMrpDetailRespVO.getSuppCode());
            purPoSaveDTO.setPoSource(UdcEnum.PUR_PO_GEN_TYPE_MRP.getValueCode());
            purPoSaveDTO.setSceneTypeCode(ScpConstant.PUR_SCENE_TYPE_CODE_001);
            purPoSaveDTO.setBuId(ouCodeBuIdMap.get(scpMrpDetailRespVO.getPurOuCode()).getBuId());
            purPoSaveDTO.setAgentEmpId(getEmployeeId());
            purPoSaveDTO.setDocStatus("CF");
            purPoSaveDTO.setRelateDocId(scpMrpDetailRespVO.getMasId());
            // 无需支付
            purPoSaveDTO.setRelateDocNo(scpMrpDetailRespVO.getMrpLotNo());
            purPoSaveDTO.setPaymentStatus("45");
            purPoSaveDTO.setRelateDocCls("MRP");
            PurSuppBaseRpcDTO suppDTO = suppMap.get(scpMrpDetailRespVO.getSuppCode());
            if (suppDTO == null) {
                throw new BusinessException("供应商不存在，请检查：" + scpMrpDetailRespVO.getSuppCode());
            }

            List<PurPoDSaveDTO> purPoDSaveDTOList = mrpPlanMap.stream().map(planList -> {
                PurPoDSaveDTO purPoDSaveDTO = new PurPoDSaveDTO();
                ScpMrpDetailRespVO planVO = planList.get(0);
                purPoDSaveDTO.setItemCode(planVO.getItemCode());
                purPoDSaveDTO.setItemId(planVO.getItemId());
                purPoDSaveDTO.setOuId(planVO.getPurOuId());
                purPoDSaveDTO.setSuppId(planVO.getSuppId());
                purPoDSaveDTO.setCurrCode(suppDTO.getCurrCode());
                purPoDSaveDTO.setQty(planList.stream().map(ScpMrpDetailRespVO::getPurQty).reduce(BigDecimal.ZERO, BigDecimal::add));
                purPoDSaveDTO.setUom(planVO.getPurUom());

                purPoDSaveDTO.setRelateDocId(scpMrpDetailRespVO.getMasId());
                purPoDSaveDTO.setRelateDocNo(scpMrpDetailRespVO.getMrpLotNo());
                purPoDSaveDTO.setRelateDocDid(scpMrpDetailRespVO.getDId());
                purPoDSaveDTO.setRelateDocCls("MRP");
                return purPoDSaveDTO;
            }).collect(Collectors.toList());
            purPoSaveDTO.setPurPoDCreateParamVOList(purPoDSaveDTOList);
            purPoSaveDTOList.add(purPoSaveDTO);
        });

        // 生成采购订单
        List<PurPoSaveDTO> resultList;
        try {
            ApiResult<List<PurPoSaveDTO>> apiResult = purPoProvider.createBatch(purPoSaveDTOList);
            if (apiResult.isFailed() || CollectionUtil.isEmpty(apiResult.getData())) {
                throw new BusinessException("MRP计划推送采购订单失败：" + apiResult.getMsg());
            }
            resultList = apiResult.getData();
        } catch (Exception e) {
            String pushFailReason = e.getMessage().length() > 100 ? e.getMessage().substring(0, 100) : e.getMessage();
            List<Long> planIds = mrpDetailRespVOS.stream().map(ScpMrpDetailRespVO::getPlanId).collect(Collectors.toList());
            scpMrpDPlanDomainService.updatePushStatus(planIds, ScpUdcEnum.MRP_D_PUSH_STATUS_FAIL.getValueCode(), null, null, null, pushFailReason);
            scpMrpDomainService.updatePushStatus(id, ScpUdcEnum.MRP_D_PUSH_STATUS_FAIL.getValueCode(), pushFailReason);
            return;
        }

        // 更新采购计划推送状态
        Map<String, List<ScpMrpDetailRespVO>> detailMap = mrpDetailRespVOS.stream().collect(Collectors.groupingBy(d -> d.getPurOuCode() + d.getSuppCode() + d.getWhCode() + d.getItemCode()));
        resultList.forEach(po -> po.getPurPoDCreateParamVOList().forEach(pod -> {
            List<Long> planIds = detailMap.get(po.getOuCode() + po.getSuppCode() + po.getWhCode() + pod.getItemCode()).stream().map(ScpMrpDetailRespVO::getPlanId).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(planIds)) {
                scpMrpDPlanDomainService.updatePushStatus(planIds, ScpUdcEnum.MRP_D_PUSH_STATUS_SUCCESS.getValueCode(), po.getDocNo(), pod.getId(), pod.getLineNo(), null);
            }
        }));
        // 更新MRP计划推送状态
        scpMrpDomainService.updatePushStatus(id, ScpUdcEnum.MRP_PUSH_STATUS_COMPLETE.getValueCode(), null);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void compute(Long predId) {
        if (predId == null) {
            throw new BusinessException("预测批次不能为空");
        }

        Optional<ScpPredictStStockCalcDO> predCalcOptional = scpPredictStStockCalcRepo.findById(predId);
        if (predCalcOptional.isEmpty()) {
            throw new BusinessException("预测批次不能为空");
        }
        ScpPredictStStockCalcDO predCalcDO = predCalcOptional.get();

        // 预测批次明细
        List<ScpPredictStStockRespVO> predList = scpPredictStStockRepoProc.findByMasIdAndMrpFlag(predId, false);

        if (CollectionUtil.isEmpty(predList)) {
            throw new BusinessException("该预测批次明细已计算完成，请检查");
        }

        String mrpLotNo = rmiSysUserRpcService.sysNumberRuleGenerateCode("MRP", new ArrayList<>());

        ScpMrpDO scpMrpDO = new ScpMrpDO();
        scpMrpDO.setMrpLotNo(mrpLotNo);
        scpMrpDO.setPredLotNo(predCalcDO.getPredLotNo());
        scpMrpDO.setPushStatus(ScpUdcEnum.MRP_PUSH_STATUS_NOT.getValueCode());
        scpMrpDO.setCalcStatus(ScpUdcEnum.MRP_CALC_STATUS_PROCESS.getValueCode());

        // 保存MRP计划
        ScpMrpDO scpMrpResult = scpMrpDomainService.save(scpMrpDO);

        //事物提交之后计算具体数据
        transactionCommitHandler.handle(() -> CompletableFuture.runAsync(() -> {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionTemplate.execute(transactionStatus -> {
                try {
                    computeAndSaveDetail(predList, scpMrpResult);
                    scpMrpDomainService.updateCalcStatus(List.of(scpMrpResult.getId()), ScpUdcEnum.MRP_CALC_STATUS_SUCCESS.getValueCode(), null);
                } catch (Exception e) {
                    log.info("MRP计划明细计算失败：{}", e.getMessage());
                    scpMrpDomainService.updateCalcStatus(List.of(scpMrpResult.getId()), ScpUdcEnum.MRP_CALC_STATUS_FAIL.getValueCode(), e.getMessage().length() > 100 ? e.getMessage().substring(0, 100) : e.getMessage());
                }
                return "OK";
            });
        }, taskExecutor));
    }

    private void computeAndSaveDetail(List<ScpPredictStStockRespVO> predList, ScpMrpDO scpMrpDO) {
        log.info("计算并保存MRP计划明细和采购计划，预测库存：{}，MRP计划：{}", JSONObject.toJSONString(predList), JSONObject.toJSONString(scpMrpDO));
        LocalDateTime dateTo = LocalDateTime.now();
        LocalDateTime dateFrom = dateTo.minusMonths(1L);

        Integer qtyPlace = SysUtils.getQtyPlace();

        Map<String, String> uomMap = udcProvider.getValueMapByUdcCode("yst-supp", "UOM");

        // 商品信息查询
        List<Long> itemIds = predList.stream().map(ScpPredictStStockRespVO::getItemId).distinct().collect(Collectors.toList());
        ItmItemBaseRpcParam itemParam = new ItmItemBaseRpcParam();
        itemParam.setItemIds(itemIds);
        List<ItmItemBaseRpcDTO> itemList = rmiItemService.findItemBaseRpcDtoByParam(itemParam);
        if (CollectionUtil.isEmpty(itemList)) {
            throw new BusinessException("商品不存在，请检查");
        }
        Map<Long, ItmItemBaseRpcDTO> itemMap = itemList.stream().collect(Collectors.toMap(ItmItemBaseRpcDTO::getId, c -> c));

        // 查询商品的经营目录
        List<String> ouCodes = predList.stream().map(ScpPredictStStockRespVO::getOuCode).distinct().collect(Collectors.toList());
        ItmItemBusinessRpcDtoParam businessParam = new ItmItemBusinessRpcDtoParam();
        businessParam.setItemIds(itemIds);
        businessParam.setBuCodes(ouCodes);
        Map<String, ItmItemBusinessRpcDTO> businessMap = rmiItemService.findItmItemBusinessByParam(businessParam).stream().collect(Collectors.toMap(t -> t.getBuCode() + t.getItemCode(), t -> t));

        // 查询商品的单位转化率
        Map<Long, BigDecimal> uomConvMap = getUomConvMap(itemList);

        // 查询采购在途数量
        Map<Long, BigDecimal> purRouteMap = getPurRouteMap(predList, dateFrom, dateTo);

        // 查询库存流水相关计算信息
        Map<Long, InvIoCalcRpcDTO> invIoCalcMap = getInvIoCalcMap(predList, dateFrom, dateTo);

        // 供应商份额分配查询
        Set<String> allocationConcatKeys = predList.stream().map(pred -> pred.getWhCode() + pred.getItemCode()).collect(Collectors.toSet());
        Map<String, List<ScpSupplyAllocationDO>> allocationMap = scpSupplyAllocationDomainService.findByConcatKeys(allocationConcatKeys).stream().collect(Collectors.groupingBy(d -> d.getStoreWhCode() + d.getItemCode()));

        List<ScpMrpDDO> scpMrpDDOList = new ArrayList<>();
        List<ScpMrpDPlanDO> scpMrpDPlanDOList = new ArrayList<>();
        for (ScpPredictStStockRespVO d : predList) {
            ScpMrpDDO scpMrpDDO = new ScpMrpDDO();
            scpMrpDDO.setMasId(scpMrpDO.getId());
            scpMrpDDO.setRelateDocDid(d.getId());
            scpMrpDDO.setOuId(d.getOuId());
            scpMrpDDO.setOuCode(d.getOuCode());
            scpMrpDDO.setOuName(d.getOuName());
            scpMrpDDO.setWhId(d.getWhId());
            scpMrpDDO.setWhCode(d.getWhCode());
            scpMrpDDO.setWhName(d.getWhName());
            scpMrpDDO.setItemId(d.getItemId());
            scpMrpDDO.setItemCode(d.getItemCode());
            scpMrpDDO.setItemName(d.getItemName());
            scpMrpDDO.setPredSafetyQty(d.getPredSafetyQty());
            scpMrpDDO.setPredTargetQty(d.getPredTargetQty());
            scpMrpDDO.setUom(d.getPlanUom());
            scpMrpDDO.setUomName(uomMap.get(d.getPlanUom()));
            scpMrpDDO.setIoQty(d.getIoQty());
            scpMrpDDO.setUomRatio(d.getUomRatio());
            scpMrpDDO.setSalQty(d.getIoQty().divide(d.getUomRatio(), qtyPlace, RoundingMode.HALF_UP));

            ItmItemBaseRpcDTO item = itemMap.get(d.getItemId());
            if (item == null) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "商品不存在：" + d.getItemCode());
            }

            ItmItemBusinessRpcDTO itemBusiness = businessMap.get(d.getOuCode() + d.getItemCode());
            if (itemBusiness == null) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "经营目录不存在，公司：" + d.getOuCode() + ",商品：" + d.getItemCode());
            }
            // 采购提前期
            scpMrpDDO.setPurLeadTime(BigDecimal.valueOf(itemBusiness.getPurAheadPeriod()));
            // 预期交货日期
            scpMrpDDO.setExpectArriveDate(dateTo.plusDays(itemBusiness.getPurAheadPeriod()));
            scpMrpDDO.setItemName(item.getItemName());
            scpMrpDDO.setStdInvTurnover(BigDecimal.valueOf(30L).divide(BigDecimal.valueOf(itemBusiness.getPurDelivPeriod()), 2, RoundingMode.HALF_UP));

            // 单位转化率
            BigDecimal uomRatio = item.getUom().equals(item.getUom4()) ? BigDecimal.ONE : uomConvMap.get(d.getItemId());

            // 最小起订量
            scpMrpDDO.setMoq(itemBusiness.getMoq());

            if (uomRatio == null) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "商品单位转化率不存在：" + d.getItemCode());
            }

            InvIoCalcRpcDTO invIoCalcRpcDTO = invIoCalcMap.get(d.getId());
            if (invIoCalcRpcDTO != null) {
                scpMrpDDO.setInvQty(invIoCalcRpcDTO.getInvQty().multiply(uomRatio).setScale(qtyPlace, RoundingMode.HALF_UP));
                scpMrpDDO.setBeforeQty(invIoCalcRpcDTO.getBeforeInvQty());
                // 平均值
                BigDecimal avgQty = (scpMrpDDO.getBeforeQty().add(invIoCalcRpcDTO.getInvQty())).divide(BigDecimal.valueOf(2L), 8, RoundingMode.HALF_UP);
                if (avgQty.compareTo(BigDecimal.ZERO) == 0) {
                    scpMrpDDO.setInvTurnover(BigDecimal.ZERO);
                } else {
                    scpMrpDDO.setInvTurnover(scpMrpDDO.getIoQty().divide(avgQty, qtyPlace, RoundingMode.HALF_UP));
                }
            } else {
                scpMrpDDO.setInvQty(BigDecimal.ZERO);
                scpMrpDDO.setInvTurnover(BigDecimal.ZERO);
            }

            // 采购在途数量
            scpMrpDDO.setPurTransitQty(purRouteMap.getOrDefault(d.getId(), BigDecimal.ZERO).multiply(uomRatio).setScale(qtyPlace, RoundingMode.HALF_UP));
            // 毛需求=预测安全库存+预测目标库存-在库数
            BigDecimal grossDemand = scpMrpDDO.getPredSafetyQty().add(scpMrpDDO.getPredTargetQty()).subtract(scpMrpDDO.getInvQty());
            if (grossDemand.compareTo(BigDecimal.ZERO) <= 0) {
                scpMrpDDO.setGrossDemand(BigDecimal.ZERO);
                scpMrpDDO.setNetDemand(BigDecimal.ZERO);
            } else if (scpMrpDDO.getMoq() == null) {
                scpMrpDDO.setGrossDemand(grossDemand);
                // 净需求
                scpMrpDDO.setNetDemand(scpMrpDDO.getGrossDemand().subtract(scpMrpDDO.getPurTransitQty()));
            } else {
                scpMrpDDO.setGrossDemand(grossDemand.divide(scpMrpDDO.getMoq(), 0, RoundingMode.UP).multiply(scpMrpDDO.getMoq()).setScale(qtyPlace, RoundingMode.HALF_UP));
                // 净需求
                scpMrpDDO.setNetDemand(scpMrpDDO.getGrossDemand().subtract(scpMrpDDO.getPurTransitQty()));
            }

            scpMrpDDOList.add(scpMrpDDO);

            if (scpMrpDDO.getNetDemand().compareTo(BigDecimal.ZERO) > 0) {
                // 根据供应商份额分配生成采购计划
                List<ScpSupplyAllocationDO> allocationList = allocationMap.get(d.getWhCode() + d.getItemCode());
                if (!CollectionUtil.isEmpty(allocationList)) {
                    allocationList.forEach(allocation -> {
                        ScpMrpDPlanDO scpMrpDPlanDO = new ScpMrpDPlanDO();
                        // 标识，方便后续赋值
                        scpMrpDPlanDO.setMasId(d.getId());
                        scpMrpDPlanDO.setMrpId(scpMrpDDO.getMasId());
                        scpMrpDPlanDO.setOuId(allocation.getOuId());
                        scpMrpDPlanDO.setOuCode(allocation.getOuCode());
                        scpMrpDPlanDO.setOuName(allocation.getOuName());
                        scpMrpDPlanDO.setSuppId(allocation.getSuppId());
                        scpMrpDPlanDO.setSuppCode(allocation.getSuppCode());
                        scpMrpDPlanDO.setSuppName(allocation.getSuppName());
                        scpMrpDPlanDO.setPushStatus(ScpUdcEnum.MRP_D_PUSH_STATUS_NOT.getValueCode());
                        scpMrpDPlanDO.setQty(scpMrpDDO.getNetDemand().multiply(allocation.getAllocation().divide(BigDecimal.valueOf(100L), 2, RoundingMode.HALF_UP)));
                        scpMrpDPlanDOList.add(scpMrpDPlanDO);
                    });
                }
            }
        }

        // 批量保存明细和采购计划
        scpMrpDDOList = scpMrpDDomainService.save(scpMrpDDOList);
        if (!CollectionUtil.isEmpty(scpMrpDPlanDOList)) {
            Map<Long, Long> mrpDMap = scpMrpDDOList.stream().collect(Collectors.toMap(ScpMrpDDO::getRelateDocDid, ScpMrpDDO::getId));
            scpMrpDPlanDOList.forEach(plan -> plan.setMasId(mrpDMap.get(plan.getMasId())));
            scpMrpDPlanDomainService.saveDO(scpMrpDPlanDOList);
        }

        // 回写预测安全库存已生成MRP计划
        scpPredictStStockRepoProc.updateMrpFlag(predList.stream().map(ScpPredictStStockRespVO::getId).collect(Collectors.toList()), Boolean.TRUE);
    }

    private Map<Long, BigDecimal> getUomConvMap(List<ItmItemBaseRpcDTO> itemList) {
        List<ItmItemUomConvRpcOtherParam> uomConvDParams = itemList.stream().map(item -> {
            ItmItemUomConvRpcOtherParam convParam = new ItmItemUomConvRpcOtherParam();
            convParam.setItemId(item.getId());
            convParam.setFromUom(item.getUom());
            convParam.setToUom(item.getUom4());
            return convParam;
        }).collect(Collectors.toList());
        ItmItemUomConvRpcDtoParam uomConvParam = new ItmItemUomConvRpcDtoParam();
        uomConvParam.setOtherParam(uomConvDParams);

        // 查询商品单位转化率
        List<ItmItemUomConvRpcDTO> uomConvList = rmiItemService.findItemUomConvDtoByParam(uomConvParam);
        if (CollectionUtil.isEmpty(uomConvList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "商品单位转化率查询失败");
        }
        return uomConvList.stream().collect(Collectors.toMap(ItmItemUomConvRpcDTO::getItemId, ItmItemUomConvRpcDTO::getRatio));
    }

    @NotNull
    private Map<Long, BigDecimal> getPurRouteMap(List<ScpPredictStStockRespVO> predList, LocalDateTime dateFrom, LocalDateTime dateTo) {
        List<PurPoRouteParamDTO> purRouteParams = predList.stream().map(pred -> {
            PurPoRouteParamDTO purPoRouteParamDTO = new PurPoRouteParamDTO();
            purPoRouteParamDTO.setId(pred.getId());
            purPoRouteParamDTO.setOuId(pred.getOuId());
            purPoRouteParamDTO.setWhId(pred.getWhId());
            purPoRouteParamDTO.setItemId(pred.getItemId());
            purPoRouteParamDTO.setDateFrom(dateFrom);
            purPoRouteParamDTO.setDateTo(dateTo);
            return purPoRouteParamDTO;
        }).collect(Collectors.toList());

        // 采购在途查询
        List<PurPoRouteRpcDTO> purRouteList = purPoProvider.findRouteInfo(purRouteParams);
        return purRouteList.stream().collect(Collectors.toMap(PurPoRouteRpcDTO::getId, PurPoRouteRpcDTO::getRouteQty));
    }

    private Map<Long, InvIoCalcRpcDTO> getInvIoCalcMap(List<ScpPredictStStockRespVO> predList, LocalDateTime dateFrom, LocalDateTime dateTo) {
        List<InvIoCalcParamRpcDTO> invIoCalcParams = predList.stream().map(pred -> {
            InvIoCalcParamRpcDTO paramDTO = new InvIoCalcParamRpcDTO();
            paramDTO.setId(pred.getId());
            paramDTO.setOuId(pred.getOuId());
            paramDTO.setWhId(pred.getWhId());
            paramDTO.setItemId(pred.getItemId());
            paramDTO.setDateFrom(dateFrom);
            paramDTO.setDateTo(dateTo);
            paramDTO.setSceneCodes(List.of("SO_OUT", "SO_LOGIC_DELIVERY", "SO_LOGIC_RECEIPT", "SO_RETURN", "TRN003", "TRN004", "TRN008", "TRN_RELEASE _ISSUE", "TRN007"));
            return paramDTO;
        }).collect(Collectors.toList());

        // 库存流水计算
        List<InvIoCalcRpcDTO> invIoCalcList = rmiInvStkRpcService.calcIoByMrp(invIoCalcParams);
        return invIoCalcList.stream().collect(Collectors.toMap(InvIoCalcRpcDTO::getId, c -> c));
    }

    /**
     * 查询当前用户的员工ID
     *
     * @return 员工ID
     */
    private Long getEmployeeId() {
        //获取当前操作人id和name
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        log.info("当前登录用户信息:{}", JSONUtil.toJsonStr(currentUserDTO));
        if (ObjectUtils.isEmpty(currentUserDTO) || currentUserDTO.getDetail() == null) {
            throw new BusinessException("查询当前登录用户失败");
        }
        return currentUserDTO.getDetail().getEmployeeId();
    }
}

