package com.elitesland.scp.domain.service.msg.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.provider.SysNoticeRpcService;
import com.elitescloud.cloudt.system.provider.dto.save.SysNoticeSaveDTO;
import com.elitesland.scp.application.facade.vo.query.msg.ScpMessagePageMngQueryVO;
import com.elitesland.scp.application.facade.vo.resp.msg.BaseScpMessageListRespVO;
import com.elitesland.scp.application.facade.vo.resp.msg.ScpMessageEditRespVO;
import com.elitesland.scp.application.facade.vo.resp.msg.ScpMessagePageRespVO;
import com.elitesland.scp.application.facade.vo.save.msg.ScpMessageSaveVO;
import com.elitesland.scp.common.ScpMessageChannel;
import com.elitesland.scp.common.ScpMessageReceiverType;
import com.elitesland.scp.domain.convert.msg.ScpMessageConvert;
import com.elitesland.scp.domain.entity.msg.ScpMessageChannelDO;
import com.elitesland.scp.domain.entity.msg.ScpMessageDO;
import com.elitesland.scp.domain.entity.msg.ScpMessageReceiverDO;
import com.elitesland.scp.domain.entity.msg.ScpMessageTxtDO;
import com.elitesland.scp.domain.service.msg.ScpMessageMngService;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/28
 */
@Service
public class ScpMessageMngServiceImpl extends BaseScpMessageService implements ScpMessageMngService {
    private static final Logger logger = LoggerFactory.getLogger(ScpMessageMngServiceImpl.class);

    @Autowired
    private SysNoticeRpcService noticeRpcService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(ScpMessageSaveVO saveVO) {
        var messageDO = this.upsert(saveVO);
        return ApiResult.ok(messageDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> delete(Set<Long> ids, Boolean real) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("消息ID为空");
        }

        // 删除
        messageAdapter(ids, t -> {
            // 获取已发布的外部的ID
            var idsOfHasWeb = filterChannelOfWeb(t);
            List<String> idsOfSys = idsOfHasWeb.isEmpty() ? Collections.emptyList() :
                    new ArrayList<>(channelRepoProc.getPublishedId(idsOfHasWeb, ScpMessageChannel.WEB.name()).values());

            if (real == null || Boolean.TRUE.equals(real)) {
                repoProc.delete(t);
                txtRepoProc.deleteByMessageId(t);
                channelRepoProc.deleteByMessageId(t);
                receiverRepoProc.deleteByMessageId(t);
            } else {
                repoProc.updateDeleteFlag(t);
            }

            // 撤回系统公告
            this.revokeFromSys(idsOfHasWeb, idsOfSys);
        });

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> execPublish(Set<Long> ids) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("消息ID为空");
        }

        // 发布
        this.messageAdapter(ids, t -> {
            repoProc.updatePublish(t);
            repoProc.updatePublish(t, LocalDateTime.now());

            // 发布至系统域
            this.publishToSys(t);
        });

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<Long>> execRevoke(Set<Long> ids) {
        if (ids != null) {
            ids = ids.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        }
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("消息ID为空");
        }

        // 撤回
        this.messageAdapter(ids, t -> {
            repoProc.updateRevoke(t);

            // 撤回系统公告
            var idsOfHasWeb = filterChannelOfWeb(t);
            List<String> idsOfSys = idsOfHasWeb.isEmpty() ? Collections.emptyList() :
                    new ArrayList<>(channelRepoProc.getPublishedId(idsOfHasWeb, ScpMessageChannel.WEB.name()).values());
            this.revokeFromSys(idsOfHasWeb, idsOfSys);
        });

        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<ScpMessageEditRespVO> getEditVO(Long id) {
        Assert.notNull(id, "消息ID为空");

        var messageVO = get(id);

        return ApiResult.ok(messageVO);
    }

    @Override
    public ApiResult<PagingVO<ScpMessagePageRespVO>> pageMng(ScpMessagePageMngQueryVO queryVO) {
        var pageData = this.queryByPage(queryVO);

        return ApiResult.ok(pageData);
    }

    private void publishToSys(Collection<Long> ids) {
        var idsOfHasWeb = filterChannelOfWeb(ids);
        if (idsOfHasWeb.isEmpty()) {
            return;
        }

        var messageList = repoProc.get(idsOfHasWeb);
        if (messageList.isEmpty()) {
            return;
        }
        var messageTxtIds = messageList.stream().map(ScpMessageDO::getTxtId).filter(Objects::nonNull).collect(Collectors.toSet());
        Map<Long, ScpMessageTxtDO> messageTxtMap = messageTxtIds.isEmpty() ? Collections.emptyMap() :
                txtRepoProc.get(messageTxtIds).stream().collect(Collectors.toMap(ScpMessageTxtDO::getMessageId, Function.identity(), (t1, t2) -> t1));
        var noticeList = messageList.stream().map(t -> {
            SysNoticeSaveDTO noticeSaveDTO = ScpMessageConvert.INSTANCE.convert2NoticeSaveDTO(t);
            noticeSaveDTO.setSys(false);

            var txt = messageTxtMap.get(t.getId());
            if (txt != null) {
                noticeSaveDTO.setTxt(txt.getTxt());
            }
            return noticeSaveDTO;
        }).collect(Collectors.toList());
        var publishedIds = noticeRpcService.publish(noticeList).computeData();
        int i = 0;
        for (ScpMessageDO messageDO : messageList) {
            var publishId = publishedIds.size() > i ? publishedIds.get(i) : null;
            channelRepoProc.updatePublishedId(messageDO.getId(), ScpMessageChannel.WEB.name(), publishId == null ? null : publishId.toString());
            i++;
        }
    }

    private void revokeFromSys(Collection<Long> messageId, Collection<String> publishIds) {
        if (publishIds.isEmpty()) {
            return;
        }

        channelRepoProc.deletePublishedId(messageId, ScpMessageChannel.WEB.name());

        var noticeIds = publishIds.stream().map(Long::parseLong).collect(Collectors.toSet());
        noticeRpcService.revokeBatch(noticeIds);
    }

    private Set<Long> filterChannelOfWeb(Collection<Long> ids) {
        var channelsMap =  channelRepoProc.getChannels(ids);
        if (channelsMap.isEmpty()) {
            return Collections.emptySet();
        }

        Set<Long> filterIds = new HashSet<>(ids.size());
        for (var entry : channelsMap.entrySet()) {
            if (entry.getValue().contains(ScpMessageChannel.WEB.name())) {
                filterIds.add(entry.getKey());
            }
        }
        return filterIds;
    }

    private void messageAdapter(Collection<Long> ids, Consumer<Collection<Long>> consumer) {
        var normalIds = repoProc.exists(ids);
        if (!normalIds.isEmpty()) {
            consumer.accept(normalIds);

            if (normalIds.size() == ids.size()) {
                return;
            }
        }
    }

    private ScpMessageEditRespVO get(long id) {
        var messageDO = repoProc.get(id);
        if (messageDO == null) {
            return null;
        }
        var messageEditRespVO = ScpMessageConvert.INSTANCE.convert2EditVO(messageDO);
        messageEditRespVO.setPublished(this.hasPublished(messageDO));
        super.fillDetailInfo(messageDO, messageEditRespVO);

        // 是否推送小程序
        var channels = channelRepoProc.getChannels(id);
        messageEditRespVO.setPushApplet(channels.contains(ScpMessageChannel.APPLET.name()));

        // 门店
        messageEditRespVO.setOrgStoreCodes(new ArrayList<>(8));
        messageEditRespVO.setOrgStores(new ArrayList<>(32));
        if (Boolean.TRUE.equals(messageEditRespVO.getPushApplet())) {
            var storeCodes = receiverRepoProc.getReceivers(id, ScpMessageReceiverType.STORE.name());
            if (!storeCodes.isEmpty()) {
                var storeList = orgStoreRpcService.getOrgStoreBaseMap(storeCodes);
                if (CollUtil.isNotEmpty(storeList)) {
                    for (OrgStoreBaseRpcDTO value : storeList.values()) {
                        messageEditRespVO.getOrgStoreCodes().add(value.getStoreCode());
                        messageEditRespVO.getOrgStores().add(new CodeNameParam(value.getStoreCode(), value.getStoreName()));
                    }
                }
            }
        }

        return messageEditRespVO;
    }

    private ScpMessageDO upsert(ScpMessageSaveVO saveVO) {
        ScpMessageDO messageDO = this.checkAndConvert(saveVO);

        // 保存消息信息
        repoProc.save(messageDO);
        // 保存消息内容
        var messageTxtDO = this.saveTxt(saveVO, messageDO);
        repoProc.updateTxtId(messageDO.getId(), messageTxtDO.getId());
        // 保存消息渠道
        this.saveChannels(saveVO, messageDO);
        // 保存消息接收者
        this.saveReceivers(saveVO, messageDO);

        return messageDO;
    }

    private PagingVO<ScpMessagePageRespVO> queryByPage(ScpMessagePageMngQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO).map(t -> {
            var respVO = ScpMessageConvert.INSTANCE.convert2PageVO(t);
            respVO.setPublished(this.hasPublished(t));
            return respVO;
        });
        if (pageData.isEmpty()) {
            return pageData;
        }

        super.fillListInfo(pageData.getRecords());

        var ids = pageData.stream().map(BaseScpMessageListRespVO::getId).filter(Objects::nonNull).collect(Collectors.toSet());
        // 渠道
        var channelsMap = channelRepoProc.getChannels(ids);
        // 接收者
        var receiversMap = receiverRepoProc.getReceivers(ids);
        // 门店
        var storeCodes = receiversMap.values().stream()
                .flatMap(t -> t.stream().filter(tt -> ScpMessageReceiverType.STORE.name().equals(tt.getName())).map(IdCodeNameParam::getCode))
                .collect(Collectors.toList());
        Map<String, OrgStoreBaseRpcDTO> storeMap = storeCodes.isEmpty() ? Collections.emptyMap() : orgStoreRpcService.getOrgStoreBaseMap(storeCodes);

        pageData.each(t -> {
            // 门店
            var channels = channelsMap.get(t.getId());
            if (channels != null && channels.contains(ScpMessageChannel.APPLET.name())) {
                t.setPushApplet(true);
                t.setOrgStoreCodes(new ArrayList<>(8));
                t.setOrgStores(new ArrayList<>(8));
                var receivers = receiversMap.get(t.getId());
                for (var receiver : receivers) {
                    if (ScpMessageReceiverType.STORE.name().equals(receiver.getName())) {
                        var store = storeMap.get(receiver.getCode());
                        if (store == null) {
                            continue;
                        }
                        t.getOrgStoreCodes().add(store.getStoreCode());
                        t.getOrgStores().add(new CodeNameParam(store.getStoreCode(), store.getStoreName()));
                    }
                }
            } else {
                t.setPushApplet(false);
            }
        });

        return pageData;
    }

    private boolean hasPublished(ScpMessageDO messageDO) {
        if (Boolean.FALSE.equals(messageDO.getPublished())) {
            return false;
        }

        return messageDO.getPublishTime() != null && messageDO.getPublishTime().isBefore(LocalDateTime.now());
    }

    private ScpMessageDO checkAndConvert(ScpMessageSaveVO saveVO) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        ScpMessageDO messageDO = null;
        if (saveVO.getId() == null) {
            messageDO = new ScpMessageDO();
        } else {
            messageDO = repoProc.get(saveVO.getId());
            Assert.notNull(messageDO, "消息不存在");

            // 已发布的不可修改
            if (this.hasPublished(messageDO)) {
                throw new BusinessException("消息已发布，不可修改");
            }
        }

        Assert.hasText(saveVO.getTitle(), "标题为空");
        Assert.hasText(saveVO.getTxt(), "内容为空");

        // 默认值处理
        ObjUtil.ifNull(saveVO.getAuthorId(), currentUser.getUserId(), saveVO::setAuthorId);
        ObjUtil.ifNull(saveVO.getTop(), false, saveVO::setTop);
        ObjUtil.ifNull(saveVO.getPushApplet(), false, saveVO::setPushApplet);

        if (Boolean.TRUE.equals(saveVO.getPushApplet()) && CollUtil.isEmpty(saveVO.getOrgStoreCodes())) {
            throw new BusinessException("推送小程序时门店不能为空");
        }

        ScpMessageConvert.INSTANCE.convert2DO(saveVO, messageDO);

        messageDO.setPublished(saveVO.getPublishTime() != null);
        messageDO.setChangeTime(LocalDateTime.now());

        if (CollUtil.isNotEmpty(saveVO.getFileCodes())) {
            messageDO.setFileCodesStr(String.join(",", saveVO.getFileCodes()));
        } else {
            messageDO.setFileCodesStr(null);
        }

        return messageDO;
    }

    private ScpMessageTxtDO saveTxt(ScpMessageSaveVO saveVO, ScpMessageDO messageDO) {
        ScpMessageTxtDO messageTxtDO = messageDO.getTxtId() == null ? new ScpMessageTxtDO() : txtRepoProc.get(messageDO.getTxtId());
        messageTxtDO.setTxt(saveVO.getTxt());
        messageTxtDO.setMessageId(messageDO.getId());

        if (CollUtil.isNotEmpty(saveVO.getTxtFileCodes())) {
            messageTxtDO.setFileCodesStr(String.join(",", saveVO.getTxtFileCodes()));
        } else {
            messageTxtDO.setFileCodesStr(null);
        }

        messageTxtDO.setExtAttr(CollUtil.isEmpty(saveVO.getTxtExtAttr()) ? null : saveVO.getTxtExtAttr());

        txtRepoProc.save(messageTxtDO);
        return messageTxtDO;
    }

    private void saveChannels(ScpMessageSaveVO saveVO, ScpMessageDO messageDO) {
        List<String> channels = new ArrayList<>();
        if (Boolean.TRUE.equals(saveVO.getPushApplet())) {
            channels.add(ScpMessageChannel.APPLET.name());
        } else {
            channels.add(ScpMessageChannel.WEB.name());
        }

        var currentChannels = channelRepoProc.getChannels(messageDO.getId());
        // 删除不存在的
        if (!currentChannels.isEmpty()) {
            var toDelChannels = currentChannels.stream().filter(t -> !channels.contains(t)).collect(Collectors.toList());
            if (!toDelChannels.isEmpty()) {
                channelRepoProc.deleteChannels(messageDO.getId(), toDelChannels);
            }
        }

        // 保存新的
        var channelDoList = channels.stream()
                .filter(t -> !currentChannels.contains(t))
                .map(t -> {
                    ScpMessageChannelDO channelDO = new ScpMessageChannelDO();
                    channelDO.setMessageId(messageDO.getId());
                    channelDO.setChannel(t);

                    return channelDO;
                }).collect(Collectors.toList());
        if (!channelDoList.isEmpty()) {
            channelRepoProc.save(channelDoList);
        }
    }

    private void saveReceivers(ScpMessageSaveVO saveVO, ScpMessageDO messageDO) {
        List<String> receivers = new ArrayList<>();
        if (CollUtil.isNotEmpty(saveVO.getOrgStoreCodes())) {
            receivers.addAll(saveVO.getOrgStoreCodes().stream().filter(StringUtils::hasText).collect(Collectors.toList()));
        }

        // 门店
        var currentReceivers = receiverRepoProc.getReceivers(messageDO.getId(), ScpMessageReceiverType.STORE.name());
        // 删除不存在的
        if (!currentReceivers.isEmpty()) {
            var toDelReceivers = currentReceivers.stream().filter(t -> !receivers.contains(t)).collect(Collectors.toList());
            if (!toDelReceivers.isEmpty()) {
                receiverRepoProc.deleteReceivers(messageDO.getId(), ScpMessageReceiverType.STORE.name(), toDelReceivers);
            }
        }

        // 保存新的
        var receiverDoList = receivers.stream()
                .filter(t -> !currentReceivers.contains(t))
                .map(t -> {
                    ScpMessageReceiverDO receiverDO = new ScpMessageReceiverDO();
                    receiverDO.setMessageId(messageDO.getId());
                    receiverDO.setReceiverType(ScpMessageReceiverType.STORE.name());
                    receiverDO.setReceiverId(t);
                    receiverDO.setViewed(false);

                    return receiverDO;
                }).collect(Collectors.toList());
        if (!receiverDoList.isEmpty()) {
            receiverRepoProc.save(receiverDoList);
        }
    }
}
