package com.elitesland.scp.domain.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.core.common.utils.UUIDUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitesland.sale.api.vo.resp.crm.CustBaseDTO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDListParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderPayInfoRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSaveVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderDConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderRelateDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiSalRpcService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandOrderDDomainServiceImpl implements ScpDemandOrderDDomainService {

    private final EntityManager entityManager;
    private final RmiItemService rmiItemService;
    private final RmiSalRpcService rmiSalRpcService;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;

    @Override
    public Long saveDemandOrderD(ScpDemandOrderDSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpDemandOrderDDO intentDO = ScpDemandOrderDConvert.INSTANCE.saveVoToDO(saveVO);
            return scpDemandOrderDRepo.save(intentDO).getId();
        } else {
            Optional<ScpDemandOrderDDO> option = scpDemandOrderDRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("订货订单明细ID：" + saveVO.getId() + "不存在");
            }
            ScpDemandOrderDConvert.INSTANCE.copySaveParamToDo(saveVO, option.get());
            return scpDemandOrderDRepo.save(option.get()).getId();
        }
    }

    @Override
    @SysCodeProc
    public List<ScpDemandOrderDDTO> findDemandOrderDByMasId(Long masId) {
        return scpDemandOrderDRepo.findByMasId(masId).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandOrderDDO> findDemandOrderDBySpuCodes(Long masId, List<String> spuCode) {
        return scpDemandOrderDRepo.findByMasIdAndSpuItemCodeIn(masId, spuCode);
    }

    @Override
    public Page<ScpDemandOrderDDO> pageDemandOrderDByMasId(ScpDemandOrderDPageParamVO paramVO) {
        Page<ScpDemandOrderDDO> scpDemandOrderDDTOS = scpDemandOrderDRepo.findByMasIdAndCombineItemCodeIsNull(paramVO.getMasId(),
                paramVO.getPageRequest());
        return scpDemandOrderDDTOS;
    }

    @Override
    public List<ScpDemandOrderDDO> findByMasIdAndCombineItemCodeIsNotNull(ScpDemandOrderDPageParamVO paramVO) {
        List<ScpDemandOrderDDO> scpDemandOrderDDTOS = scpDemandOrderDRepo.findByMasIdAndCombineItemCodeIsNotNull(paramVO.getMasId());
        return scpDemandOrderDDTOS;
    }


    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDByIds(List<Long> ids) {
        return scpDemandOrderDRepo.findByIdIn(ids).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpDemandOrderDRepoProc.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePlanQtyAndAmtById(Long id, BigDecimal planQty) {
        scpDemandOrderDRepoProc.updatePlanQtyAndAmtById(id, planQty);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSave(List<ScpDemandOrderDSaveVO> saveVOS, AtomicInteger lineNo, String source) {
        Map<String, List<ScpDemandOrderDSaveVO>> whCodeMap = saveVOS.stream().collect(Collectors.groupingBy(row -> row.getSuppWhCode()));

        List<String> custCodes = saveVOS.stream().map(ScpDemandOrderDSaveVO::getSaleCustCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        Map<String, CustBaseDTO> custMap = rmiSalRpcService.getCustBaseMapByCode(custCodes);
        saveVOS.forEach(row -> {
            if (StrUtil.isBlank(row.getSaleOuCode())) {
                row.setPayStatus(ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode());
            } else {
                CustBaseDTO custBaseInfoDTO = custMap.get(row.getSaleCustCode());
                String payStatus = (!"OUT".equals(custBaseInfoDTO.getInOutCust())) ? ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode() : ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode();
                row.setPayStatus(payStatus);
            }
        });

        //门店订货单获取运费行商品
        String type = scpDemandOrderRepoProc.getOrderTypeById(saveVOS.get(0).getMasId());
        if (ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(type) && source == null) {
            String itemCode = SysUtils.getOrderItemSetting();
            String orderItemImgUrlSetting = SysUtils.getOrderItemImgUrlSetting();
            ItmItemBaseRpcParam itmItemBaseRpcParam = new ItmItemBaseRpcParam();
            itmItemBaseRpcParam.setItemCodes(Arrays.asList(itemCode));
            List<ItmItemBaseRpcDTO> itemBaseRpcDTOList = rmiItemService.findItemBaseRpcDtoByParam(itmItemBaseRpcParam);
            if (CollUtil.isEmpty(itemBaseRpcDTOList)) {
                throw new BusinessException("运费商品编码：" + itemCode + "不存在");
            }
            //门店订货单添加运费行，按照【要货门店】、【供应仓库】or【供应供应商】生成一行运费
            whCodeMap.forEach((whCode, list) -> {
                ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = new ScpDemandOrderDSaveVO();
                ScpDemandOrderDSaveVO oldSaveVO = list.get(0);
                if (!(oldSaveVO.getFreightFlag() && oldSaveVO.getFreightRatio() != null)) {
                    return;
                }
                if (StrUtil.isBlank(oldSaveVO.getSaleOuCode())) {
                    return;
                }
                BigDecimal freightRatio = oldSaveVO.getFreightRatio();
                ItmItemBaseRpcDTO itmItemBaseRpcDTO = itemBaseRpcDTOList.get(0);
                BigDecimal amt = list.stream().filter(row -> row.getPlanAmt() != null).map(row -> SysUtils.processAmtScale(row.getPlanAmt().multiply(freightRatio)))
                        .collect(Collectors.reducing(BigDecimal.ZERO, BigDecimal::add));
                scpDemandOrderDSaveVO.setMasId(saveVOS.get(0).getMasId());
                scpDemandOrderDSaveVO.setDemandQuantity(BigDecimal.ONE);
                scpDemandOrderDSaveVO.setAllocationQuantity(BigDecimal.ONE);
                scpDemandOrderDSaveVO.setAllocationDeQuantity(BigDecimal.ONE);
                scpDemandOrderDSaveVO.setPlanQuantity(BigDecimal.ONE);
                scpDemandOrderDSaveVO.setPrice(SysUtils.processPriceScale(amt));
                scpDemandOrderDSaveVO.setPlanAmt(SysUtils.processAmtScale(amt));
                scpDemandOrderDSaveVO.setPrice(SysUtils.processAmtScale(amt));
                scpDemandOrderDSaveVO.setCurrency(ScpConstant.CNY);
                scpDemandOrderDSaveVO.setPreRootUuid(UUIDUtil.getUUID());
                scpDemandOrderDSaveVO.setItemId(itmItemBaseRpcDTO.getId());
                scpDemandOrderDSaveVO.setItemCode(itmItemBaseRpcDTO.getItemCode());
                scpDemandOrderDSaveVO.setItemName(itmItemBaseRpcDTO.getItemName());
                scpDemandOrderDSaveVO.setSpuItemCode(itmItemBaseRpcDTO.getSpuCode());
                scpDemandOrderDSaveVO.setSpuItemName(itmItemBaseRpcDTO.getSpuName());
                scpDemandOrderDSaveVO.setUom2(itmItemBaseRpcDTO.getUom());
                scpDemandOrderDSaveVO.setSaleCustCode(oldSaveVO.getSaleCustCode());
                scpDemandOrderDSaveVO.setUom2Name(itmItemBaseRpcDTO.getUomName());
                scpDemandOrderDSaveVO.setUnit(itmItemBaseRpcDTO.getUom2());
                scpDemandOrderDSaveVO.setUnitName(itmItemBaseRpcDTO.getUom2Name());
                if (amt.compareTo(BigDecimal.ZERO) == 0) {
                    scpDemandOrderDSaveVO.setPayStatus(ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode());
                } else {
                    scpDemandOrderDSaveVO.setPayStatus(oldSaveVO.getPayStatus());
                }
                scpDemandOrderDSaveVO.setItemType(itmItemBaseRpcDTO.getItemType2());
                scpDemandOrderDSaveVO.setSupplyType(oldSaveVO.getSupplyType());
                scpDemandOrderDSaveVO.setSuppWhId(oldSaveVO.getSuppWhId());
                scpDemandOrderDSaveVO.setSuppWhName(oldSaveVO.getSuppWhName());
                scpDemandOrderDSaveVO.setSuppWhCode(oldSaveVO.getSuppWhCode());
                scpDemandOrderDSaveVO.setRatio(BigDecimal.valueOf(100));
                scpDemandOrderDSaveVO.setUomRatio(itmItemBaseRpcDTO.getUomRatio2());
                scpDemandOrderDSaveVO.setQty2(itmItemBaseRpcDTO.getUomRatio2());
                scpDemandOrderDSaveVO.setLineNo(new BigDecimal(lineNo.getAndIncrement()));
                scpDemandOrderDSaveVO.setFreightLineFlag(Boolean.TRUE);
                scpDemandOrderDSaveVO.setIsCalculated(Boolean.TRUE);
                scpDemandOrderDSaveVO.setOuId(oldSaveVO.getOuId());
                scpDemandOrderDSaveVO.setOuCode(oldSaveVO.getOuCode());
                scpDemandOrderDSaveVO.setOuName(oldSaveVO.getOuName());
                scpDemandOrderDSaveVO.setSaleOuName(oldSaveVO.getSaleOuName());
                scpDemandOrderDSaveVO.setSaleOuCode(oldSaveVO.getSaleOuCode());
                scpDemandOrderDSaveVO.setSaleCustCode(oldSaveVO.getSaleCustCode());
                scpDemandOrderDSaveVO.setImgUrl(orderItemImgUrlSetting);
                saveVOS.add(scpDemandOrderDSaveVO);
            });
        }

        List<ScpDemandOrderDSaveVO> updateList = saveVOS.stream().filter(saveVO -> ObjectUtil.isNotEmpty(saveVO.getId())).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(updateList)) {
            scpDemandOrderDRepo.saveAll(ScpDemandOrderDConvert.INSTANCE.saveVoListToDOList(updateList));
        }
        List<ScpDemandOrderDSaveVO> saveList = saveVOS.stream().filter(saveVO -> ObjectUtil.isEmpty(saveVO.getId())).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(saveList)) {
            this.batchInsert(ScpDemandOrderDConvert.INSTANCE.saveVoListToDOList(saveList));
        }
        List<ScpDemandOrderDDO> saveRootList = scpDemandOrderDRepoProc.findByMasIdAndIsPushed(saveVOS.get(0).getMasId(), Boolean.FALSE);
        Map<String, ScpDemandOrderDDO> rootItemMap = saveRootList.stream()
                .collect(Collectors.groupingBy(ScpDemandOrderDDO::groupByUuid))
                .values().stream()
                .map(list -> list.get(0))
                .collect(Collectors.toMap(ScpDemandOrderDDO::groupByUuid, Function.identity(),
                        (t1, t2) -> t1));
        saveRootList.forEach(row -> {
            ScpDemandOrderDDO item = rootItemMap.get(row.groupByUuid());
            row.setSourceId(item != null ? item.getId() : row.getId());
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSuppAllocPlanQty(Long demandId) {
        scpDemandOrderDRepo.updateSuppAllocPlanQty(demandId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateIsPushed(List<Long> dIds, String msg) {
        String syncMsg = StrUtil.isBlank(msg) ? "成功" : msg;
        scpDemandOrderDRepo.updateIsPushed(dIds, syncMsg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSyncMsg(List<Long> dIds, String msg) {
        scpDemandOrderDRepo.updateSyncMsg(dIds, msg);
    }

    @Override
    public void updateSrcInfo(Long srcDocId, String srcDocNo, Long dId, Integer srcDocLineNo) {
        scpDemandOrderDRepo.updateSrcDocInfoByIds(srcDocId, srcDocNo, dId, srcDocLineNo);
    }

    @Override
    public void updatePOSrcInfo(Long id, Long srcDocId, String srcDocNo, String srcLineNO) {
        scpDemandOrderDRepoProc.updateSrcDocInfoById(id, srcDocId, srcDocNo, srcLineNO, "PO");
    }

    @Override
    public void clearSrcDocInfo(Long docId) {
        scpDemandOrderDRepoProc.clearSrcDocInfo(docId);
    }

    @Override
    public void resetSrcDocQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.resetSrcDocQuantity(dId, qty);
    }

    @Override
    public void updateAllocQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepoProc.updateAllocQtyById(dId, qty);
    }

    @Override
    public void updateQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateQtyById(dId, qty);
    }

    @Override
    public void updateErrorMsg(Long dId, String errorMsg) {
        scpDemandOrderDRepoProc.updateErrorMsgById(dId, errorMsg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteUnPushedItem(Long masId) {
        scpDemandOrderDRepo.deleteByMasIdAndIsPushed(masId, Boolean.FALSE);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBySourceIds(List<Long> sourceIds) {
        scpDemandOrderDRepoProc.deleteBySourceIds(sourceIds);
    }

    @Override
    public BigDecimal findMaxLineNoPushedByMasId(Long masId) {
        return scpDemandOrderDRepoProc.findPushedItemMasId(masId);
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDByMasIds(List<Long> masIds) {
        return scpDemandOrderDRepo.findByMasIdIn(masIds).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<Long> getMasIdByDId(List<Long> dIds) {
        return scpDemandOrderDRepoProc.getMasIdByDId(dIds);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancelOrderByDIds(List<Long> dIds) {
        Optional<ScpDemandOrderDDO> byId = scpDemandOrderDRepo.findById(dIds.get(0));
        String payStatus = ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode().equals(byId.get().getPayStatus()) ? ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode() : ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode();
        scpDemandOrderDRepo.cancelOrderByDIds(dIds, payStatus);
    }

    @Override
    public List<ScpDemandOrderPayInfoRespVO> payInfo(Long masId) {
        return scpDemandOrderDRepoProc.findPayInfo(masId);
    }

    @Override
    public void updateRecvQty(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateRecvQtyById(dId, qty);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void initAllocQuantity(Long masId) {
        scpDemandOrderDRepoProc.initAllocQuantity(masId);
    }

    @Override
    public void updatePayStatus(List<Long> ids, String payStatus) {
        scpDemandOrderDRepo.updatePayStatus(ids, payStatus);
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDBySrcDocId(Long srcDocId) {
        return scpDemandOrderDRepo.findBySrcDocId(srcDocId).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDBySrcDocNo(String srcDocNo) {
        return scpDemandOrderDRepo.findBySrcDocNo(srcDocNo).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public LocalDateTime findDocCreateTimeBySrcDocId(Long srcDocId) {
        return scpDemandOrderDRepoProc.findDocCreateTimeBySrcDocId(srcDocId);
    }

    @Override
    public List<ScpDemandOrderRelateDTO> findRelateOrderByMasId(Long masId) {
        return scpDemandOrderDRepoProc.findRelateOrderByMasId(masId);
    }

    @Override
    public List<ScpDemandOrderDDTO> queryList(ScpDemandOrderDListParamVO scpDemandOrderDParamVO) {
        return scpDemandOrderDRepoProc.queryList(scpDemandOrderDParamVO);
    }

    public void batchInsert(List<ScpDemandOrderDDO> dataList) {
        int index = 0;
        int batchSize = 500;
        for (ScpDemandOrderDDO data : dataList) {
            entityManager.persist(data);
            if (batchSize > 1) {
                // 开启批量
                index++;
                if (index % batchSize == 0) {
                    entityManager.flush();
                    entityManager.clear();
                }
            }
        }
        if (!dataList.isEmpty()) {
            entityManager.flush();
        }
    }
}
