package com.elitesland.scp.application.service.calendar;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetComputeParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetLineParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetParamVO;
import com.elitesland.scp.application.facade.vo.param.setting.ScpOrderSettingParamVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetLineRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetApproveVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetSaveVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetSubmitVO;
import com.elitesland.scp.application.service.setting.ScpOrderSettingService;
import com.elitesland.scp.domain.convert.calendar.ScpStoreCalendarSetConvert;
import com.elitesland.scp.domain.convert.calendar.ScpStoreDemandCalendarConvert;
import com.elitesland.scp.domain.entity.calendar.ScpStoreDemandCalendarDO;
import com.elitesland.scp.domain.service.calendar.ScpStoreCalendarSetDomainService;
import com.elitesland.scp.domain.service.calendar.ScpStoreCalendarSetLineDomainService;
import com.elitesland.scp.domain.service.calendar.ScpStoreDemandCalendarDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.calendar.ScpStoreCalendarSetDTO;
import com.elitesland.scp.utils.CalendarBuilder;
import com.elitesland.workflow.enums.ProcInstStatus;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreCalendarSetServiceImpl implements ScpStoreCalendarSetService {

    private final ScpStoreCalendarSetDomainService scpStoreCalendarSetDomainService;
    private final ScpStoreCalendarSetLineDomainService scpStoreCalendarSetLineDomainService;
    private final TransactionTemplate transactionTemplate;
    private final ScpStoreDemandCalendarDomainService scpStoreDemandCalendarDomainService;
    private final TaskExecutor taskExecutor;
    private final ScpOrderSettingService scpOrderSettingService;

    @Override
    @SysCodeProc
    public PagingVO<ScpStoreCalendarSetPageRespVO> page(ScpStoreCalendarSetPageParamVO paramVO) {
        return scpStoreCalendarSetDomainService.queryCalendarSetList(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveCalendarSet(ScpStoreCalendarSetSaveVO saveVO) {
        // 校验单据类型配置
        checkSetting(saveVO);
        if(CollUtil.isNotEmpty(saveVO.getItemType2s())){
            saveVO.setItemType2(String.join(",", saveVO.getItemType2s()));
        }
        return scpStoreCalendarSetDomainService.saveCalendarSet(saveVO);
    }

    @Override
    @SysCodeProc
    public Optional<ScpStoreCalendarSetRespVO> findCalendarSetById(Long id) {
        return scpStoreCalendarSetDomainService.findCalendarSetById(id).map(ScpStoreCalendarSetConvert.INSTANCE::dtoToRespVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpStoreCalendarSetDomainService.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submit(ScpStoreCalendarSetSubmitVO submitVO) {
        ScpStoreCalendarSetLineParamVO scpStoreCalendarSetLineParamVO = new ScpStoreCalendarSetLineParamVO();
        scpStoreCalendarSetLineParamVO.setMasId(submitVO.getId());
        Long count = scpStoreCalendarSetLineDomainService.countCalendarSetLine(scpStoreCalendarSetLineParamVO);
        if (count == 0) {
            throw new BusinessException("门店订货日历配置明细不能为空");
        }
        scpStoreCalendarSetDomainService.applyChangeUpdate(submitVO.getId(), ScpUdcEnum.CALENDAR_SET_STATUS_APPRING.getValueCode(), ProcInstStatus.APPROVING);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchApprove(ScpStoreCalendarSetApproveVO approveSaveParam) {
        scpStoreCalendarSetDomainService.approveUpdateApplyAlter(approveSaveParam.getIds(), ScpUdcEnum.CALENDAR_SET_STATUS_COMPLETE.getValueCode(), ProcInstStatus.APPROVED,
                approveSaveParam.getApproveComment(), LocalDateTime.now());
        //异步执行写入门店订货日历表
        CompletableFuture.runAsync(() -> {
            for (Long id : approveSaveParam.getIds()) {
                batchSaveCalendar(id);
            }
        }, taskExecutor);
    }

    private void batchSaveCalendar(Long id) {
        Optional<ScpStoreCalendarSetDTO> calendarSetDTO = scpStoreCalendarSetDomainService.findCalendarSetById(id);
        if (calendarSetDTO.isEmpty()) {
            throw new BusinessException("查询不到门店订货日历配置信息");
        }
        ScpStoreCalendarSetDTO scpStoreCalendarSetDTO = calendarSetDTO.get();
        List<String> itemType2s = scpStoreCalendarSetDTO.getItemType2s();
        if(CollUtil.isNotEmpty(itemType2s)&&itemType2s.contains("all")){
            ScpOrderSettingParamVO scpOrderSettingParamVO = new ScpOrderSettingParamVO();
            List<ScpOrderSettingRespVO> scpOrderSettingRespVOS = scpOrderSettingService.queryOrderSettingList(scpOrderSettingParamVO);
            List<String> collect = scpOrderSettingRespVOS.stream().map(ScpOrderSettingRespVO::getDocType).collect(Collectors.toList());
            collect.remove("BF");
            itemType2s=collect;
        }
        Integer startYear = scpStoreCalendarSetDTO.getStartYear();
        LocalDate startOfYear = LocalDate.of(startYear, 1, 1);
        List<ScpStoreCalendarSetLineRespVO> storeInfos = scpStoreCalendarSetLineDomainService.findStoreCodeByMasId(id, startYear);
        List<String> storeCodes = storeInfos.stream().map(ScpStoreCalendarSetLineRespVO::getStoreCode).collect(Collectors.toList());

        //删除门店订货日历历史记录
        scpStoreDemandCalendarDomainService.deleteByStoreAndYear(storeCodes, startYear.toString());
        var computeParamVO = getScpStoreCalendarSetComputeParamVO(id, storeInfos);

        for (int i = 0; i < 12; i++) {
            log.info("开始写入:{}月份门店订货日历", i + 1);
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            int finalI = i;
            List<String> finalItemType2s = itemType2s;
            transactionTemplate.execute(trans -> {
                try {
                    LocalDate startDate = startOfYear.plusMonths(finalI);
                    LocalDate endDate = startDate.with(TemporalAdjusters.lastDayOfMonth());
                    List<ScpStoreDemandCalendarDO> scpStoreDemandCalendarDOS = CalendarBuilder.buildCalendarSetList(startDate, endDate, computeParamVO);
                    List<ScpStoreDemandCalendarDO> typeList = new ArrayList<>();
                    finalItemType2s.forEach(itemType2 -> {
                        List<ScpStoreDemandCalendarDO> copiedList = scpStoreDemandCalendarDOS.stream()
                                .map(original -> {
                                    // 复制 original 的属性到 copy
                                    ScpStoreDemandCalendarDO scpStoreDemandCalendarDO = ScpStoreDemandCalendarConvert.INSTANCE.doToDo(original);
                                    scpStoreDemandCalendarDO.setItemType2(itemType2);
                                    return scpStoreDemandCalendarDO;
                                })
                                .collect(Collectors.toList());
                        typeList.addAll(copiedList);
                    });
                    scpStoreDemandCalendarDomainService.batchInsert(typeList, 10000);
                    return "OK";
                } catch (Exception e) {
                    trans.setRollbackOnly();
                    throw new BusinessException(ApiCode.FAIL, e.getMessage());
                }
            });
            log.info("结束写入:{}月份门店订货日历", i + 1);
        }
    }

    private ScpStoreCalendarSetComputeParamVO getScpStoreCalendarSetComputeParamVO(Long masId, List<ScpStoreCalendarSetLineRespVO> storeInfos) {
        List<ScpStoreCalendarSetLineRespVO> monList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 1);
        List<ScpStoreCalendarSetLineRespVO> tuesList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 2);
        List<ScpStoreCalendarSetLineRespVO> wendsList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 3);
        List<ScpStoreCalendarSetLineRespVO> thursList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 4);
        List<ScpStoreCalendarSetLineRespVO> friList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 5);
        List<ScpStoreCalendarSetLineRespVO> saturList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 6);
        List<ScpStoreCalendarSetLineRespVO> sunList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 7);
        ScpStoreCalendarSetComputeParamVO computeParamVO = new ScpStoreCalendarSetComputeParamVO();
        computeParamVO.setMonList(monList);
        computeParamVO.setTuesList(tuesList);
        computeParamVO.setWendsList(wendsList);
        computeParamVO.setThursList(thursList);
        computeParamVO.setFriList(friList);
        computeParamVO.setSaturList(saturList);
        computeParamVO.setSunList(sunList);
        computeParamVO.setStoreInfos(storeInfos);
        computeParamVO.setNonMonList(findMissingList(monList, storeInfos));
        computeParamVO.setNonTuesList(findMissingList(tuesList, storeInfos));
        computeParamVO.setNonWendsList(findMissingList(wendsList, storeInfos));
        computeParamVO.setNonThursList(findMissingList(thursList, storeInfos));
        computeParamVO.setNonFriList(findMissingList(friList, storeInfos));
        computeParamVO.setNonSaturList(findMissingList(saturList, storeInfos));
        computeParamVO.setNonSunList(findMissingList(sunList, storeInfos));
        return computeParamVO;
    }

    public List<ScpStoreCalendarSetLineRespVO> findMissingList(List<ScpStoreCalendarSetLineRespVO> filterList, List<ScpStoreCalendarSetLineRespVO> storeInfos) {
        Set<String> storeCodesInList = filterList.stream()
                .map(ScpStoreCalendarSetLineRespVO::getStoreCode)
                .collect(Collectors.toCollection(HashSet::new));

        return storeInfos.stream()
                .filter(storeInfo -> !storeCodesInList.contains(storeInfo.getStoreCode()))
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchReject(ScpStoreCalendarSetApproveVO approveSaveParam) {
        scpStoreCalendarSetDomainService.approveUpdateApplyAlter(approveSaveParam.getIds(), ScpUdcEnum.CALENDAR_SET_STATUS_REJECTED.getValueCode(), ProcInstStatus.REJECTED,
                approveSaveParam.getApproveComment(), LocalDateTime.now());
    }

    private void checkSetting(ScpStoreCalendarSetSaveVO saveVO) {
        if (saveVO.getEndYear() == null) {
            saveVO.setEndYear(saveVO.getStartYear());
        }
        ScpStoreCalendarSetParamVO scpStoreCalendarSetParamVO = new ScpStoreCalendarSetParamVO();
        scpStoreCalendarSetParamVO.setSetName(saveVO.getSetName());
        scpStoreCalendarSetParamVO.setSetCode(saveVO.getSetCode());
        scpStoreCalendarSetParamVO.setEndYear(saveVO.getEndYear());
        scpStoreCalendarSetParamVO.setStartYear(saveVO.getStartYear());
        List<ScpStoreCalendarSetRespVO> calendarSets = scpStoreCalendarSetDomainService.findCalendarSet(scpStoreCalendarSetParamVO);
        if (CollUtil.isNotEmpty(calendarSets)) {
            var scpStoreCalendarSetRespVO = calendarSets.get(0);
            if (saveVO.getId() != null && !saveVO.getId().equals(scpStoreCalendarSetRespVO.getId())) {
                throw new BusinessException("门店订货日历配置编码已存在，请检查");
            }
        }
        if (saveVO.getId() == null) {
            saveVO.setDocStatus(ScpUdcEnum.CALENDAR_SET_STATUS_DR.getValueCode());
        }
    }
}
