package com.elitesland.scp.application.service.freight;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.inv.provider.InvWhProvider;
import com.elitesland.scp.application.facade.vo.freight.*;
import com.elitesland.scp.application.facade.vo.resp.freight.ScpStoreOrderFreightPageRespVO;
import com.elitesland.scp.domain.entity.freight.ScpStoreOrderFreightDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.repo.freight.ScpStoreOrderFreightRepo;
import com.elitesland.scp.infr.repo.freight.ScpStoreOrderFreightRepoProc;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreWhDTO;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreOrderFreightServiceImpl implements ScpStoreOrderFreightService{
    private final ScpStoreOrderFreightRepo scpStoreOrderFreightRepo;
    private final ScpStoreOrderFreightRepoProc scpStoreOrderFreightRepoProc;
    private final InvWhProvider invWhProvider;
    private final OrgStoreRpcService orgStoreRpcService;
    private final RmiItemService rmiItemService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;

    // 检验ScpStoreOrderFreightSaveParamVO的合法性
    private void validate(ScpStoreOrderFreightSaveParamVO param){
        if (param.getId() != null && param.getId() == 0L){
            param.setId(null);
        }
        if (param.getOrderType() == null || param.getFreightMethod() == null){
            throw new BusinessException("订货类型和运费收取方式均不能为空");
        }
        // 两种不同的订货类型，直送时不需要校验供货仓库
        if (param.getOrderType().equals(ScpUdcEnum.STORE_ORDER_TYPE_WD.getValueCode())) {
            if (param.getWhCode() == null) {
                throw new BusinessException("直送订货类型，供货仓库不能为空");
            }
        }else if(param.getOrderType().equals(ScpUdcEnum.STORE_ORDER_TYPE_DD.getValueCode())){
            param.setWhCode(null);
            param.setWhName(null);
        }else{
            throw new BusinessException("订货类型不合法");
        }
        // 两种不同的运费收取方式，对应两个不可缺少的字段
        if (param.getFreightMethod().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_PR.getValueCode())) {
            if (param.getFreightRate() == null) {
                throw new BusinessException("PRPR运费收取方式，运费收取比例不能为空");
            }
        } else if (param.getFreightMethod().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_QU.getValueCode())) {
            if (param.getAmountPerQuantity() == null) {
                throw new BusinessException("QU运费收取方式，每件运费收取金额不能为空");
            }
        }else{
            throw new BusinessException("运费收取方式不合法");
        }
        // 判断状态
        if (param.getStatus() == null) {
            throw new BusinessException("状态不能为空");
        }else if(param.getStatus() != 0 && param.getStatus() != 1){
            throw new BusinessException("状态不合法");
        }
        if (param.getStatus() == 0){
            return; // 是一个失效逻辑，不做其他校验
        }
        // 判重逻辑：订货类型+仓库（包含空值）+门店+商品（包含空值）只能有一条
        // 重复维护/导入时，覆盖老数据
        List<ScpStoreOrderFreightDO> byParam2 = scpStoreOrderFreightRepoProc.findByParam2(param);
        if (Objects.isNull(param.getId()) && !byParam2.isEmpty()) {
            // 删除数据库该条目
            for (ScpStoreOrderFreightDO scpStoreOrderFreightDO : byParam2) {
                deleteStoreOrderFreight(scpStoreOrderFreightDO.getId());
            }
        }
    }
    public ScpStoreOrderFreightSaveParamVO validateAndToSaveVO(ScpStoreOrderFreightImportVO param){
        ScpStoreOrderFreightSaveParamVO scpStoreOrderFreightSaveParamVO = BeanUtils.copyProperties(param, ScpStoreOrderFreightSaveParamVO.class);
        if (param.getOrderTypeName() == null || param.getFreightMethodName() == null){
            throw new BusinessException("订货类型和运费收取方式均不能为空");
        }
        if (param.getFreightMethodName().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_PR.getValueCodeName())){
            scpStoreOrderFreightSaveParamVO.setFreightMethod(ScpUdcEnum.STORE_ORDER_FREIGHT_PR.getValueCode());
        }
        if (param.getFreightMethodName().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_QU.getValueCodeName())){
            scpStoreOrderFreightSaveParamVO.setFreightMethod(ScpUdcEnum.STORE_ORDER_FREIGHT_QU.getValueCode());
        }
        if (param.getOrderTypeName().equals(ScpUdcEnum.STORE_ORDER_TYPE_WD.getValueCodeName())){
            scpStoreOrderFreightSaveParamVO.setOrderType(ScpUdcEnum.STORE_ORDER_TYPE_WD.getValueCode());
        }
        if (param.getOrderTypeName().equals(ScpUdcEnum.STORE_ORDER_TYPE_DD.getValueCodeName())){
            scpStoreOrderFreightSaveParamVO.setOrderType(ScpUdcEnum.STORE_ORDER_TYPE_DD.getValueCode());
        }
        scpStoreOrderFreightSaveParamVO.setStatus(1);
        if (param.getSupplyWhCode() != null) {
            scpStoreOrderFreightSaveParamVO.setWhCode(param.getSupplyWhCode());
        }
        if (param.getDemandWhStCode() != null){
            if("ALL".equals(param.getDemandWhStCode())){
                scpStoreOrderFreightSaveParamVO.setStoreCode("all");
            }else{
                scpStoreOrderFreightSaveParamVO.setStoreCode(param.getDemandWhStCode());
            }
        }
        scpStoreOrderFreightSaveParamVO.setDemandType("门店");
        validate(scpStoreOrderFreightSaveParamVO);
        return scpStoreOrderFreightSaveParamVO;
    }


    @Override
    @Transactional
    public ScpStoreOrderFreightDO saveStoreOrderFreight(ScpStoreOrderFreightSaveParamVO param){
        validate(param);
        // 映射
        ScpStoreOrderFreightDO scpStoreOrderFreightDO = BeanUtils.copyProperties(param, ScpStoreOrderFreightDO.class);
        return scpStoreOrderFreightRepo.save(scpStoreOrderFreightDO);
    }
    @Override
    @Transactional
    public List<ScpStoreOrderFreightDO> saveAllStoreOrderFreight(List<ScpStoreOrderFreightSaveParamVO> param){
        List<ScpStoreOrderFreightDO> scpStoreOrderFreightDOS = new ArrayList<>();
        // 判重
        Set<String> orderCodeAndStoreTyprAndWhCodeAndItemCode = new HashSet<>();
        for(ScpStoreOrderFreightSaveParamVO paramVO : param){
            validate(paramVO);
            ScpStoreOrderFreightDO scpStoreOrderFreightDO = BeanUtils.copyProperties(paramVO, ScpStoreOrderFreightDO.class);
            String orderCodeAndStoreTyprAndWhCodeAndItemCodeStr = scpStoreOrderFreightDO.getOrderType() + '-'
                    + scpStoreOrderFreightDO.getWhCode() + '-' + scpStoreOrderFreightDO.getStoreCode() + '-'
                    + scpStoreOrderFreightDO.getItemCode();
            if (orderCodeAndStoreTyprAndWhCodeAndItemCode.contains(orderCodeAndStoreTyprAndWhCodeAndItemCodeStr)){
                throw new BusinessException("保存/导入失败：重复的条目");
            }
            orderCodeAndStoreTyprAndWhCodeAndItemCode.add(orderCodeAndStoreTyprAndWhCodeAndItemCodeStr);
            scpStoreOrderFreightDOS.add(scpStoreOrderFreightDO);
        }
        return scpStoreOrderFreightRepo.saveAll(scpStoreOrderFreightDOS);
    }

    @Override
    public void deleteStoreOrderFreight(Long id){
        scpStoreOrderFreightRepo.deleteById(id);
    }

    @Override
    public PagingVO<ScpStoreOrderFreightPageRespVO> pageStoreOrderFreight(ScpStoreOrderFreightPageParamVO param){
        PagingVO<ScpStoreOrderFreightPageRespVO> scpStoreOrderFreightRespVOPagingVO = scpStoreOrderFreightRepoProc.findByParam(param);
        log.info("分页查询结果：{}", scpStoreOrderFreightRespVOPagingVO);
        if (scpStoreOrderFreightRespVOPagingVO == null || scpStoreOrderFreightRespVOPagingVO.isEmpty()){
            return scpStoreOrderFreightRespVOPagingVO;
        }
        for (ScpStoreOrderFreightPageRespVO scpStoreOrderFreightRespVO : scpStoreOrderFreightRespVOPagingVO.getRecords()){
            scpStoreOrderFreightRespVO.update();
        }
        return scpStoreOrderFreightRespVOPagingVO;
    }

    @Override
    public List<ScpStoreOrderFreightPageRespVO> listStoreOrderFreight(ScpStoreOrderFreightPageParamVO param) {
        List<ScpStoreOrderFreightPageRespVO> scpStoreOrderFreightRespVOList = scpStoreOrderFreightRepoProc.listByParam(param);
        if (scpStoreOrderFreightRespVOList == null || scpStoreOrderFreightRespVOList.isEmpty()){
            return scpStoreOrderFreightRespVOList;
        }
        for (ScpStoreOrderFreightPageRespVO scpStoreOrderFreightRespVO : scpStoreOrderFreightRespVOList){
            scpStoreOrderFreightRespVO.update();
        }
        return scpStoreOrderFreightRespVOList;
    }

    @Override
    @Transactional(rollbackOn = BusinessException.class)
    public ScpStoreOrderFreightQuaryRespVO freightBatchQuery(ScpStoreOrderFreightQuaryParamVO param){
        // 建立返回值
        ScpStoreOrderFreightQuaryRespVO scpStoreOrderFreightQuaryRespVO = new ScpStoreOrderFreightQuaryRespVO();
        BeanUtils.copyProperties(param, scpStoreOrderFreightQuaryRespVO);
        // 等待填充，查不到表示运费是0
        scpStoreOrderFreightQuaryRespVO.setFreightAmt(new BigDecimal(0));
        if (param.getStoreCode() == null){
            throw new BusinessException("门店不能为空");
        }
        if (param.getItemCode() == null || param.getItemCode().isEmpty()){
            throw new BusinessException("商品不能为空");
        }
        ScpStoreOrderFreightDO scpStoreOrderFreightDO = null;
        // 如果订货类型是直送，通过门店+商品作为第一优先级，门店作为第二优先级，获取运费.
        if (param.getOrderType().equals(ScpUdcEnum.STORE_ORDER_TYPE_DD.getValueCode())) {
            // 查询门店+商品和门店+空值
            scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndItemCode(param.getStoreCode(), param.getItemCode());
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndItemCode("all", param.getItemCode());
            }
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndItemCode(param.getStoreCode(), "");
            }
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndItemCode("all", "");
            }
        }
        // 如果订货类型是仓配，通过门店+仓库+商品作为第一优先级查询，门店+仓库作为第二优先级，获取运费.
        else if (param.getOrderType().equals(ScpUdcEnum.STORE_ORDER_TYPE_WD.getValueCode())) {
            if (param.getWhCode() == null || param.getWhCode().isEmpty()) {
                throw new BusinessException("订货类型是仓配，仓库不能为空");
            }
            // 查询门店+仓库+商品和门店+仓库+空值
            scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndWhCodeAndItemCode(param.getStoreCode(), param.getWhCode(), param.getItemCode());
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndWhCodeAndItemCode("all", param.getWhCode(), param.getItemCode());
            }
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndWhCodeAndItemCode(param.getStoreCode(), param.getWhCode(), "");
            }
            if (scpStoreOrderFreightDO == null) {
                scpStoreOrderFreightDO = scpStoreOrderFreightRepo.findByStoreCodeAndWhCodeAndItemCode("all", param.getWhCode(), "");
            }
        }
        // 1、如果运费是固定模式，此行的运费金额是订货数量*运费收取金额；
        // 2、如果运费是比例模式，运费金额=订货含税金额*运费收取比例*0.01。
        if (scpStoreOrderFreightDO != null) {
            if (scpStoreOrderFreightDO.getFreightMethod().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_PR.getValueCode())) {
                scpStoreOrderFreightQuaryRespVO.setFreightAmt(param.getDemandQuantity().multiply(scpStoreOrderFreightDO.getFreightRate().multiply(BigDecimal.valueOf(0.01f))));
            } else if (scpStoreOrderFreightDO.getFreightMethod().equals(ScpUdcEnum.STORE_ORDER_FREIGHT_QU.getValueCode())) {
                scpStoreOrderFreightQuaryRespVO.setFreightAmt(param.getPlanAmt().multiply(scpStoreOrderFreightDO.getAmountPerQuantity()));
            }else {
                throw new BusinessException("数据错误：运费收取方式未知");
            }
        }
        return scpStoreOrderFreightQuaryRespVO;
    }

    @Override
    @Transactional(rollbackOn = BusinessException.class)
    public List<ScpStoreOrderFreightQuaryRespVO> freightBatchQueryAll(List<ScpStoreOrderFreightQuaryParamVO> param){
        // 调用freightBatchQuery单条查询，然后拼接结果
        List<ScpStoreOrderFreightQuaryRespVO> scpStoreOrderFreightQuaryRespVOList = new ArrayList<>();
        for (ScpStoreOrderFreightQuaryParamVO paramVO : param) {
            scpStoreOrderFreightQuaryRespVOList.add(freightBatchQuery(paramVO));
        }
        return scpStoreOrderFreightQuaryRespVOList;
    }
    @Override
    public ScpStoreOrderFreightRespVO getStoreOrderFreightDetail(Long id){
        var s = scpStoreOrderFreightRepo.findById(id).orElse(null);
        return ScpStoreOrderFreightRespVO.from(s);
    }

    @Override
    @org.springframework.transaction.annotation.Transactional(rollbackFor = Exception.class)
    public List<String> executeImport(List<ScpStoreOrderFreightImportVO> scpStoreOrderFreightImportVOList, int startRowIndex){
        // 数据合规性校验
        checkImportData(scpStoreOrderFreightImportVOList,startRowIndex);

        if (CollUtil.isEmpty(scpStoreOrderFreightImportVOList)) {
            return Collections.emptyList();
        }
        log.info("开始导入门店运费数据,{}",scpStoreOrderFreightImportVOList);
        List<ScpStoreOrderFreightSaveParamVO> scpStoreOrderFreightSaveParamVOList = new ArrayList<>();
        List<String> errorList = new ArrayList<>();
        Map<String, String> successMapForWh = new HashMap<>(), successMapForStore = new HashMap<>();
        Set<String> errorSetForWh = new HashSet<>(), errorSetForStore = new HashSet<>();
        // 判重用
        Set<String> orderCodeAndStoreTyprAndWhCodeAndItemCode = new HashSet<>();

        List<String> itemCodes = scpStoreOrderFreightImportVOList.stream().map(ScpStoreOrderFreightImportVO::getItemCode).distinct().collect(Collectors.toList());
        List<ItmItemBaseRpcDTO> itemBaseRpcDtoByParam = new ArrayList<>();
        if(CollUtil.isNotEmpty(itemCodes)) {
            itemBaseRpcDtoByParam = rmiItemService.findItemBaseRpcDtoByParam(ItmItemBaseRpcParam.builder().itemCodes(itemCodes).build());
        }
        for (ScpStoreOrderFreightImportVO scpStoreOrderFreightImportVO : scpStoreOrderFreightImportVOList) {
            try {
                ScpStoreOrderFreightSaveParamVO scpStoreOrderFreightSaveParamVO = validateAndToSaveVO(scpStoreOrderFreightImportVO);
                // 填充仓库信息
                if (successMapForWh.containsKey(scpStoreOrderFreightSaveParamVO.getWhCode())) {
                    scpStoreOrderFreightSaveParamVO.setWhName(successMapForWh.get(scpStoreOrderFreightSaveParamVO.getWhCode()));
                } else if (errorSetForWh.contains(scpStoreOrderFreightSaveParamVO.getWhCode())) {
                    throw new BusinessException("仓库编码不存在");
                }else{
                    try {
                        var i = invWhProvider.findSimpleWhByWhCodes(List.of(scpStoreOrderFreightSaveParamVO.getWhCode()));
                        if (i.getData() != null && !i.getData().isEmpty()) {
                            successMapForWh.put(scpStoreOrderFreightSaveParamVO.getWhCode(), i.getData().get(0).getWhName());
                            scpStoreOrderFreightSaveParamVO.setWhName(i.getData().get(0).getWhName());
                        }else {
                            throw new BusinessException("仓库编码不存在");
                        }
                    }catch (Exception e){
                        errorSetForWh.add(scpStoreOrderFreightSaveParamVO.getWhCode());
                        throw new BusinessException("仓库编码不存在");
                    }
                }
                // 填充门店信息
                if (scpStoreOrderFreightSaveParamVO.getStoreCode().equals("all")){
                    scpStoreOrderFreightSaveParamVO.setStoreName("全部");
                } else if (successMapForStore.containsKey(scpStoreOrderFreightSaveParamVO.getStoreCode())) {
                    scpStoreOrderFreightSaveParamVO.setStoreName(successMapForStore.get(scpStoreOrderFreightSaveParamVO.getStoreCode()));
                } else if (errorSetForStore.contains(scpStoreOrderFreightSaveParamVO.getStoreCode())) {
                    throw new BusinessException("门店编码不存在");
                }else{
                    try {
                        var o = orgStoreRpcService.getByCode(scpStoreOrderFreightSaveParamVO.getStoreCode());
                        successMapForStore.put(scpStoreOrderFreightSaveParamVO.getStoreCode(), o.getStoreName());
                        scpStoreOrderFreightSaveParamVO.setStoreName(o.getStoreName());
                    }catch (Exception e){
                        errorSetForStore.add(scpStoreOrderFreightSaveParamVO.getStoreCode());
                        throw new BusinessException("门店编码不存在");
                    }
                }
                if(CollUtil.isNotEmpty(itemBaseRpcDtoByParam)){
                    Optional<ItmItemBaseRpcDTO> first = itemBaseRpcDtoByParam.stream().filter(i -> i.getItemCode().equals(scpStoreOrderFreightSaveParamVO.getItemCode())).findFirst();
                    if(first.isEmpty()){
                        throw new BusinessException("商品编码不存在");
                    }
                    scpStoreOrderFreightSaveParamVO.setItemName(first.get().getItemName());
                }
                // 判重
                if (orderCodeAndStoreTyprAndWhCodeAndItemCode.contains(
                        scpStoreOrderFreightSaveParamVO.getOrderType() + "-" +
                        scpStoreOrderFreightSaveParamVO.getStoreCode() +  "-" +
                        scpStoreOrderFreightSaveParamVO.getWhCode() +  "-" +
                        scpStoreOrderFreightSaveParamVO.getItemCode())) {
                    throw new BusinessException("重复导入");
                }
                orderCodeAndStoreTyprAndWhCodeAndItemCode.add(
                        scpStoreOrderFreightSaveParamVO.getOrderType() + "-" +
                        scpStoreOrderFreightSaveParamVO.getStoreCode() + "-" +
                        scpStoreOrderFreightSaveParamVO.getWhCode() + "-" +
                        scpStoreOrderFreightSaveParamVO.getItemCode());
                scpStoreOrderFreightSaveParamVOList.add(scpStoreOrderFreightSaveParamVO);
                errorList.add("");
            } catch (Exception e) {
                // 用catch的方式捕获错误，同时重用判重代码
                errorList.add("第" + (startRowIndex + scpStoreOrderFreightImportVOList.indexOf(scpStoreOrderFreightImportVO)) + "行导入失败：" + e.getMessage());
            }
        }
        log.info("开始保存门店运费数据,{}",scpStoreOrderFreightSaveParamVOList);
        if (CollUtil.isNotEmpty(scpStoreOrderFreightSaveParamVOList)){
            saveAllStoreOrderFreight(scpStoreOrderFreightSaveParamVOList);
        }
        log.info("保存门店运费数据完成,{}",errorList);
        return errorList;
    }

    private void checkImportData(List<ScpStoreOrderFreightImportVO> scpStoreOrderFreightImportVOList,int startRowIndex) {
        // 仓库编码
        List<String> supplyWhCodeList = scpStoreOrderFreightImportVOList.stream().map(e -> e.getSupplyWhCode()).collect(Collectors.toList());
        // 门店编码
        List<String> demandWhStCodeList = scpStoreOrderFreightImportVOList.stream().map(e -> e.getDemandWhStCode()).collect(Collectors.toList());
        // 商品编码
        List<String> itemCodeList = scpStoreOrderFreightImportVOList.stream().map(e -> e.getItemCode()).collect(Collectors.toList());

        // 仓库列表
        ApiResult<List<InvWhRpcSimpleDTO>> orgWhDTOSResult = invWhProvider.findSimpleWhByWhCodes(supplyWhCodeList);
        List<InvWhRpcSimpleDTO> orgWhDTOS = new ArrayList<>();
        if (orgWhDTOSResult.getData() != null && !orgWhDTOSResult.getData().isEmpty()) {
            orgWhDTOS = orgWhDTOSResult.getData();
        }
        // 门店列表
        List<OrgStoreDetailRpcDTO> orgStoreDTOS = rmiOrgStoreRpcService.queryByStoreCodes(demandWhStCodeList);

        // 商品列表
        List<ItmItemBaseRpcDTO> itmItemBaseDTOS = rmiItemService.findItemBaseRpcDtoByParam(ItmItemBaseRpcParam.builder().itemCodes(itemCodeList).build());

        // 提取仓库仓库编号，仓库名称map(whCode,whName)
        Map<String, String> whMap = orgWhDTOS.stream()
                .collect(Collectors.toMap(InvWhRpcSimpleDTO::getWhCode, InvWhRpcSimpleDTO::getWhName));
        // 提取门店编号，门店名称map(storeCode,storeName)
        Map<String, String> storeMap = orgStoreDTOS.stream()
                .collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode, OrgStoreDetailRpcDTO::getStoreName));
        // 提取商品编号，商品名称map(itemCode,itemName)
        Map<String, String> itemMap = itmItemBaseDTOS.stream()
                .collect(Collectors.toMap(ItmItemBaseRpcDTO::getItemCode, ItmItemBaseRpcDTO::getItemName));

        // 遍历scpStoreOrderFreightImportVOList，匹配每一个仓库，门店，商品的code与名称是否匹配，不匹配直接返回报错信息
        for (int i = 0; i < scpStoreOrderFreightImportVOList.size(); i++) {

            ScpStoreOrderFreightImportVO vo = scpStoreOrderFreightImportVOList.get(i);
            int rowNum = startRowIndex; // Excel 行号

            // 校验仓库
            if(vo.getSupplyWhCode()!=null){
                if(vo.getSupplyWhName()==null){
                    throw new BusinessException("第 " + rowNum + " 行：仓库编码 " + vo.getSupplyWhCode() + " 对应仓库名称为空，请检查！");
                }else if (!whMap.containsKey(vo.getSupplyWhCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：仓库编码 " + vo.getSupplyWhCode() + " 不存在，请检查！");
                }else if(!whMap.get(vo.getSupplyWhCode()).equals(vo.getSupplyWhName())){
                    throw new BusinessException("第 " + rowNum + " 行：仓库编码 " + vo.getSupplyWhCode() + " 与仓库名称 " + vo.getSupplyWhName()+"不匹配，请检查！");
                }
            }

            // 校验门店
            if (vo.getDemandWhStCode() != null) {
                if (vo.getDemandWhStName() == null) {
                    throw new BusinessException("第 " + rowNum + " 行：门店编码 " + vo.getDemandWhStCode() + " 对应门店名称为空，请检查！");
                } else if (!storeMap.containsKey(vo.getDemandWhStCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：门店编码 " + vo.getDemandWhStCode() + " 不存在，请检查！");
                } else if (!storeMap.get(vo.getDemandWhStCode()).equals(vo.getDemandWhStName())) {
                    throw new BusinessException("第 " + rowNum + " 行：门店编码 " + vo.getDemandWhStCode() + " 与门店名称 " + vo.getDemandWhStName() + " 不匹配，请检查！");
                }
            }else{
                throw new BusinessException("第 " + rowNum + " 行：门店编码 " + vo.getDemandWhStCode() + " 为空，请检查！");
            }


            // 校验商品
            if (vo.getItemCode() != null) {
                if (vo.getItemName() == null) {
                    throw new BusinessException("第 " + rowNum + " 行：商品编码 " + vo.getItemCode() + " 对应商品名称为空，请检查！");
                } else if (!itemMap.containsKey(vo.getItemCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：商品编码 " + vo.getItemCode() + " 不存在，请检查！");
                } else if (!itemMap.get(vo.getItemCode()).equals(vo.getItemName())) {
                    throw new BusinessException("第 " + rowNum + " 行：商品编码 " + vo.getItemCode() + " 与商品名称 " + vo.getItemName() + " 不匹配，请检查！");
                }
            }
        }
    }

    @Override
    public PagingVO<ScpStoreOrderFreightExportRespVO> exportSearch(ScpStoreOrderFreightPageParamVO queryParam){
        var res = scpStoreOrderFreightRepoProc.findAll(queryParam);
        // 编码不暴露
        List<ScpStoreOrderFreightExportRespVO> list = new ArrayList<>();
        res.getRecords().forEach(e ->{
            e.update();
            list.add(BeanUtils.copyProperties(e, ScpStoreOrderFreightExportRespVO.class));
        });
        PagingVO<ScpStoreOrderFreightExportRespVO> pagingVO = new PagingVO<>();
        pagingVO.setRecords(list);
        pagingVO.setTotal(res.getTotal());
        pagingVO.setAggregatedData(res.getAggregatedData());
        return pagingVO;
    }
}