package com.elitesland.scp.domain.service.msg.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.query.msg.CommonPublishedMessagePageQueryVO;
import com.elitesland.scp.application.facade.vo.resp.msg.BaseScpMessageListRespVO;
import com.elitesland.scp.application.facade.vo.resp.msg.CommonPublishedScpMessagePageRespVO;
import com.elitesland.scp.application.facade.vo.resp.msg.CommonPublishedScpMessageRespVO;
import com.elitesland.scp.common.ScpMessageReceiverType;
import com.elitesland.scp.domain.convert.msg.ScpMessageConvert;
import com.elitesland.scp.domain.service.msg.ScpMessageQueryService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/29
 */
@Service
public class ScpMessageQueryServiceImpl extends BaseScpMessageService implements ScpMessageQueryService {
    private static final Logger logger = LoggerFactory.getLogger(ScpMessageQueryServiceImpl.class);

    @Override
    public ApiResult<Long> countUnView(ScpMessageReceiverType receiverType, String receiverId) {
        if (receiverType == null || CharSequenceUtil.isBlank(receiverId)) {
            return ApiResult.fail("接收人为空");
        }

        // 统计数量
        var count = receiverRepoProc.countUnView(receiverType.name(), receiverId);

        return ApiResult.ok(count);
    }

    @Override
    public ApiResult<PagingVO<CommonPublishedScpMessagePageRespVO>> pagePublished(ScpMessageReceiverType receiverType,
                                                                                  CommonPublishedMessagePageQueryVO queryVO) {
        if (receiverType == null || CharSequenceUtil.isBlank(queryVO.getReceiverId())) {
            return ApiResult.fail("接收人为空");
        }

        // 分页查询
        var pageData = this.pageQuery(receiverType, queryVO);

        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<CommonPublishedScpMessageRespVO> viewPublished(Long id, ScpMessageReceiverType receiverType, String receiverId) {
        Assert.notNull(id, "消息ID为空");

        // 获取消息
        var message = this.getMessage(id);

        // 更新已读标识
        if (receiverType != null && StringUtils.hasText(receiverId)) {
            taskExecutor.execute(() -> receiverRepoProc.updateViewed(id, receiverType.name(), receiverId));
        }

        return ApiResult.ok(message);
    }

    private PagingVO<CommonPublishedScpMessagePageRespVO> pageQuery(ScpMessageReceiverType receiverType,
                                                                    CommonPublishedMessagePageQueryVO queryVO) {
        var pageData = repoProc.pagePublished(receiverType.name(), queryVO).map(ScpMessageConvert.INSTANCE::convert2PublishedPageVO);
        if (pageData.isEmpty()) {
            return pageData;
        }

        // 设置列表详情
        super.fillListInfo(pageData.getRecords());

        var ids = pageData.stream().map(BaseScpMessageListRespVO::getId).filter(Objects::nonNull).collect(Collectors.toSet());
        // 获取已读标识
        var viewedMap = receiverRepoProc.queryViewed(ids, receiverType.name(), queryVO.getReceiverId());

        pageData.each(t -> t.setViewed(viewedMap.getOrDefault(t.getId(), false)));

        return pageData;
    }

    private CommonPublishedScpMessageRespVO getMessage(long id) {
        var messageDO = repoProc.get(id);
        if (messageDO == null || Boolean.FALSE.equals(messageDO.getPublished()) || messageDO.getPublishTime().isAfter(LocalDateTime.now())) {
            return null;
        }

        var noticeVO = ScpMessageConvert.INSTANCE.convert2PublishedVO(messageDO);
        super.fillDetailInfo(messageDO, noticeVO);


        return noticeVO;
    }
}
