package com.elitesland.scp.application.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.invwh.InvWhRpcDTO;
import com.elitesland.inv.dto.invwh.InvWhRpcDtoParam;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.inv.provider.InvWhProvider;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcDTO;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcParam;
import com.elitesland.pur.provider.PurSuppProvider;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.supalloc.*;
import com.elitesland.scp.application.facade.vo.whnet.ScpWhNetRelationImportVO;
import com.elitesland.scp.application.service.scpsman.BaseScpmanAuthorityParam;
import com.elitesland.scp.application.service.supalloc.ScpSupplyAllocationService;
import com.elitesland.scp.domain.convert.ScpSupplyAllocationConvert;
import com.elitesland.scp.domain.entity.supalloc.ScpSupplyAllocationDO;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.domain.service.supalloc.ScpSupplyAllocationDomainService;
import com.elitesland.scp.dto.supalloc.ScpSupplyAllocationRpcDTO;
import com.elitesland.scp.dto.supalloc.ScpSupplyAllocationRpcDtoParam;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.dto.supalloc.ScpSupplyAllocationDTO;
import com.elitesland.scp.infr.repo.supalloc.ScpSupplyAllocationRepoProc;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgOuService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemCateSimpleTreeRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import com.elitesland.support.provider.path.SupportTransactionPathRpcService;
import com.elitesland.support.provider.path.dto.SupportTransactionPathDRpcDTO;
import com.elitesland.support.provider.path.dto.SupportTransactionPathRpcDTO;
import com.elitesland.support.provider.path.param.SupportTransactionPathRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

/**
 * @author jeesie
 * @description:
 * @datetime 2024年 03月 25日 10:46
 * @version: 1.0
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScpSupplyAllocationServiceImpl implements ScpSupplyAllocationService {

    private final ScpSupplyAllocationDomainService scpSupplyAllocationDomainService;

    private final UdcProvider udcProvider;

    private final ItmItemRpcService itmItemRpcService;

    private final ScpSupplyAllocationRepoProc scpSupplyAllocationRepoProc;

    private final PurSuppProvider purSuppProvider;

    private final ItmItemRpcService itemRpcService;

    private final InvWhProvider whProvider;

    private final OrgStoreRpcService orgStoreRpcService;

    private final UdcProvider systemService;

    private final ScpDemandAuthorityService scpDemandAuthorityService;

    private final RmiOrgOuService rmiOrgOuService;

    private final RmiInvStkRpcService rmiInvStkService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final SupportTransactionPathRpcService supportTransactionPathRpcService;
    private final RmiItemService rmiItemService;


    @Override
    public PagingVO<ScpSupplyAllocationPageVO> page(ScpSupplyAllocationPageParamVO param) {
        if (Boolean.TRUE.equals(param.getScpmanAuthority())) {
            PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
            if (userAuthority.isEmpty()) {
                return PagingVO.<ScpSupplyAllocationPageVO>builder()
                        .total(0L)
                        .records(new ArrayList<>())
                        .build();
            }
            extractedAuthorityParam(param, userAuthority);
        }
        return scpSupplyAllocationDomainService.page(param);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult saveSupplyAllocation(ScpSupplyAllocatioSaveVO createParam) {
        var allocatioList = createParam.getScpSupplyAllocatios();
        Map<String, List<ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio>> groupMap = allocatioList
                .stream()
                .collect(Collectors.groupingBy(d -> d.getStoreWhCode() + d.getItemCode()));
        Set<String> businessKeys = groupMap.keySet();
        List<ScpSupplyAllocationDO> existList = scpSupplyAllocationDomainService.findByConcatKeys(businessKeys);
        Map<String, List<ScpSupplyAllocationDO>> existMap = existList.stream().collect(Collectors.groupingBy(d -> d.getStoreWhCode() + d.getItemCode()));
        Set<String> updateList = new HashSet<>();
        List<ScpSupplyAllocationResultVO.ResultVO> failList = new ArrayList<>();
        List<ScpSupplyAllocationResultVO.ResultVO> successList = new ArrayList<>();
        ScpSupplyAllocationResultVO result = new ScpSupplyAllocationResultVO();
        // 校验结算路径
        checkTransactionPath(allocatioList);

        for (var allocatio : groupMap.entrySet()) {
            String allocatioKey = allocatio.getKey();
            List<ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio> allocatios = allocatio.getValue();
            var activateList = allocatios.stream().filter(d -> Boolean.TRUE.equals(d.getStatus())).collect(Collectors.toList());
            long count1 = activateList.stream().map(d -> d.getSuppCode() + d.getStoreWhCode() + d.getType() + d.getItemCode()).count();
            long count2 = activateList.stream().map(d -> d.getSuppCode() + d.getStoreWhCode() + d.getType() + d.getItemCode()).distinct().count();
            if (count1 != count2) {
                throw new BusinessException(ApiCode.FAIL, String.format("数据重复,仓库编码/门店编码-商品【%s-%s】存在相同【供应商-类型】组合数据，请检查",
                        allocatios.get(0).getStoreWhCode(), allocatios.get(0).getItemCode()));
            }
            long existCount = allocatios.stream()
                    .map(ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio::getId)
                    .filter(Objects::nonNull).count();
            if (CollUtil.isNotEmpty(activateList)) {
                if (existCount == 0 && existMap.containsKey(allocatioKey)) {
                    String errorMsg = String.format("数据重复，仓库编码/门店编码-商品【%s-%s】配置已存在,请修改", allocatios.get(0).getStoreWhCode(), allocatios.get(0).getItemCode());
                    throw new BusinessException(ApiCode.FAIL, errorMsg);

                } else if (existCount > 0 && existMap.containsKey(allocatioKey)) {
                    updateList.add(allocatioKey);
                }
                BigDecimal sumAllocation = activateList.stream()
                        .map(ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio::getAllocation)
                        .filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
                //不满足配置要求
                List<ScpSupplyAllocationResultVO.ResultVO> resultVOS = getResultVOS(allocatios);
                if (sumAllocation.compareTo(new BigDecimal("100")) == 0) {
                    successList.addAll(resultVOS);
                } else {
                    failList.addAll(resultVOS);
                }
            }
        }
        result.setFailList(failList);
        result.setSuccessList(successList);
        if (CollUtil.isEmpty(failList)) {
            if (CollUtil.isNotEmpty(updateList)) {
                List<ScpSupplyAllocationDO> scpSupplyAllocationDOS = scpSupplyAllocationDomainService.findByConcatKeys(updateList);
                List<Long> updateIds = scpSupplyAllocationDOS.stream().map(ScpSupplyAllocationDO::getId).collect(Collectors.toList());
                scpSupplyAllocationDomainService.deleteBatch(updateIds);
            }
            List<ScpSupplyAllocationDTO> allocationDOList = createParam.getScpSupplyAllocatios().stream()
                    .map(ScpSupplyAllocationConvert.INSTANCE::saveVoToDto).collect(Collectors.toList());
            scpSupplyAllocationDomainService.createBatch(allocationDOList);
            return ApiResult.builder().success(true).code(200).data(result).msg("操作成功").build();
        } else {
            return ApiResult.builder().success(false).code(0).data(result).msg("操作失败").build();
        }
    }

    /**
     * 校验结算路径
     *
     * @param allocatioList
     */
    private void checkTransactionPath(List<ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio> allocatioList) {
        List<String> demandWhCodes = allocatioList.stream().filter(row -> ScpUdcEnum.DEMAND_SET_TYPE_1.getValueCode().equals(row.getType())).map(ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio::getStoreWhCode).distinct().collect(Collectors.toList());
        Map<String, InvWhRpcSimpleDTO> demandWhMap = buildInvWhMap(demandWhCodes);

        List<String> demandStoreCodes = allocatioList.stream().filter(row -> ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(row.getType())).map(ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio::getStoreWhCode).distinct().collect(Collectors.toList());
        Map<String, OrgStoreBaseRpcDTO> fStoreMap = getStoreMap(demandStoreCodes);

        List<SupportTransactionPathRpcParam> rpcParams = new ArrayList<>();
        allocatioList.forEach(row -> {

            String demandWhStCode = row.getStoreWhCode();
            Long ouId = ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(row.getType()) ? fStoreMap.get(demandWhStCode).getOuId() : demandWhMap.get(demandWhStCode).getOuId();
            Long whOuId = row.getOuId();
            if (!ouId.equals(whOuId)) {
                SupportTransactionPathRpcParam rpcParam = new SupportTransactionPathRpcParam();
                rpcParam.setCompanyIdStart(whOuId);
                rpcParam.setCompanyIdEnd(ouId);
                rpcParams.add(rpcParam);
            }
        });
        Map<String, List<SupportTransactionPathRpcDTO>> suppTransactionPathMap = checkTransactionPathAndBuild(rpcParams);
        allocatioList.forEach(row -> {
            String demandWhStCode = row.getStoreWhCode();

            Long ouId = ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(row.getType()) ? fStoreMap.get(demandWhStCode).getOuId() : demandWhMap.get(demandWhStCode).getOuId();
            String ouCode = ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(row.getType()) ? fStoreMap.get(demandWhStCode).getOuCode() : demandWhMap.get(demandWhStCode).getOuCode();

            Long whOuId = row.getOuId();
            String whOuCode = row.getOuCode();
            if (!ouId.equals(whOuId)) {
                List<SupportTransactionPathRpcDTO> pathList = suppTransactionPathMap.get(whOuId + "@" + ouId);
                if (CollUtil.isEmpty(pathList) || StrUtil.isBlank(pathList.get(0).getPathCode())) {
                    log.info("从公司 {}到公司 {}的结算路径不存在，请维护", whOuId, ouId);
                    throw new BusinessException("从公司【" + whOuCode + "】至公司【" + ouCode + "】的结算路径不存在，请维护");
                }
                var detailList = pathList.get(0).getDetailList();
                int size = detailList.size();
                SupportTransactionPathDRpcDTO dto = detailList.get(size - 1);
                row.setSaleOuCode(dto.getCompanyCodeStart());
                row.setSaleOuName(dto.getCompanyNameStart());
                row.setSaleCustCode(dto.getCustCodeEnd());
            }
        });
    }

    /**
     * 构建需求仓库信息
     *
     * @param demandWhCodes
     * @return
     */
    private Map<String, InvWhRpcSimpleDTO> buildInvWhMap(List<String> demandWhCodes) {
        if (CollUtil.isEmpty(demandWhCodes)) {
            return new HashMap<>();
        }
        return rmiInvStkService.findInvWhBaseMapByCode(demandWhCodes);
    }

    private Map<String, OrgStoreBaseRpcDTO> getStoreMap(List<String> demandStoreCodes) {
        List<OrgStoreBaseRpcDTO> orgStoreBaseRpcDTOS = rmiOrgStoreRpcService.findOrgStoreBaseByParam(demandStoreCodes);
        if (CollUtil.isEmpty(orgStoreBaseRpcDTOS)) {
            return new HashMap<>();
        }
        return orgStoreBaseRpcDTOS.stream().collect(Collectors.toMap(OrgStoreBaseRpcDTO::getStoreCode, Function.identity(), (k1, k2) -> k1));
    }

    /**
     * 获取结算路径信息
     *
     * @param rpcParams
     * @return
     */
    private Map<String, List<SupportTransactionPathRpcDTO>> checkTransactionPathAndBuild(List<SupportTransactionPathRpcParam> rpcParams) {
        //查询结算路径
        log.info("查询启用状态的结算路径,参数:{}", JSONObject.toJSONString(rpcParams));
        List<SupportTransactionPathRpcDTO> data = supportTransactionPathRpcService.queryListByParam(rpcParams).getData();
        log.info("查询启用状态的结算路径结束,返回:{}", JSONObject.toJSONString(data));
        if (CollUtil.isEmpty(data)) {
            return new HashMap<>();
        }
        return data.stream().collect(Collectors.groupingBy(row -> row.getCompanyIdStart() + "@" + row.getCompanyIdEnd()));
    }

    @Override
    public List<ScpSupplyAllocationRespVO> findSupplyAllocationList(ScpSupplyAllocationParamVO param) {
        String bussinessKey = param.getStoreWhCode() + param.getItemCode();
        List<ScpSupplyAllocationDO> allocationDOS = scpSupplyAllocationDomainService.findAllByConcatKey(List.of(bussinessKey));
        List<ScpSupplyAllocationRespVO> voList = allocationDOS.stream().map(ScpSupplyAllocationConvert.INSTANCE::doToRespVo).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(voList)) {
            Map<String, Map<String, String>> typeMap = udcProvider.getValueMapByUdcCode("yst-suplan", Set.of("DEMAND_SET_TYPE"));
            voList.forEach(d -> {
                d.setTypeName(typeMap.getOrDefault("DEMAND_SET_TYPE", new HashMap<>()).get(d.getType()));
                if (StringUtils.isNotEmpty(d.getItemCateCode())) {
                    List<ItmItemCateSimpleTreeRpcDTO> itemCatePath =
                            itmItemRpcService.findItemCatePath(d.getItemCateCode());
                    d.setItemCatePath(itemCatePath);
                }
            });
        }
        List<String> suppCodes =
                voList.stream().map(ScpSupplyAllocationRespVO::getSuppCode).distinct().collect(Collectors.toList());
        //通过采购域rpc获取供应商归属公司信息
        PurSuppBaseRpcParam purSuppBaseRpcParam = new PurSuppBaseRpcParam();
        purSuppBaseRpcParam.setSuppCodes(suppCodes);
        List<PurSuppBaseRpcDTO> suppList =
                purSuppProvider.findBaseRpcDtoByParam(purSuppBaseRpcParam).getData();
        log.info("供应商信息查询结果:{}", JSONUtil.toJsonStr(suppList));
        if (CollUtil.isEmpty(suppList)) {
            return voList;
        }
        Map<String, PurSuppBaseRpcDTO> suppMap =
                suppList.stream().collect(Collectors.toMap(PurSuppBaseRpcDTO::getSuppCode, Function.identity(),
                        (t1, t2) -> t2));
        voList.forEach(item -> {
                    if (suppMap.containsKey(item.getSuppCode()) && CollUtil.isNotEmpty(suppMap.get(item.getSuppCode()).getSuppOuRespVOList())) {
                        item.setSuppOuRespVOList(ScpSupplyAllocationConvert.INSTANCE.dtosToPurSuppOuRespVOs(suppMap.get(item.getSuppCode()).getSuppOuRespVOList()));
                    }
                }

        );
        return voList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteWhNetRelation(List<Long> ids) {
        scpSupplyAllocationDomainService.deleteBatch(ids);
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpSupplyAllocationExportRespVO> exportSearch(ScpSupplyAllocationPageParamVO queryParam) {
        if (Boolean.TRUE.equals(queryParam.getScpmanAuthority())) {
            PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
            if (userAuthority.isEmpty()) {
                return PagingVO.<ScpSupplyAllocationExportRespVO>builder()
                        .total(0L)
                        .records(new ArrayList<>())
                        .build();
            }
            extractedAuthorityParam(queryParam, userAuthority);
        }
        return scpSupplyAllocationDomainService.exportSearch(queryParam);
    }

    @Override
    public List<ScpSupplyAllocationRpcDTO> findSupplyAllocationRpcByParam(ScpSupplyAllocationRpcDtoParam supplyAllocationRpcDtoParam) {
        return scpSupplyAllocationRepoProc.findSupplyAllocationRpcByParam(supplyAllocationRpcDtoParam);
    }

    @Override
    public List<String> importSupplyAllocation(List<ScpSupplyAllocationImportVO> dataList, int startRowIndex) {
        // 数据合规性校验
        checkImportData(dataList,startRowIndex);

        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errorList = new ArrayList<>();
        checkImportParam(dataList, errorList);
        List<String> result = validResult(errorList);
        List<String> successDocGroup = getSuccessDocGroup(result, dataList);
        if (successDocGroup.isEmpty()) {
            return result;
        }
        List<String> ouCodeList = dataList.stream().map(ScpSupplyAllocationImportVO::getOuCode)
                .distinct().collect(Collectors.toList());
        Map<String, OrgOuRpcDTO> ouDataMap =
                rmiOrgOuService.findOuDtoListByOuCodes(ouCodeList).stream().collect(Collectors.toMap(OrgOuRpcDTO::getOuCode,
                        Function.identity()));

        List<String> suppCodes = dataList.stream().map(ScpSupplyAllocationImportVO::getSuppCode)
                .distinct().collect(Collectors.toList());
        List<String> itemCodes = dataList.stream().map(ScpSupplyAllocationImportVO::getItemCode)
                .distinct().collect(Collectors.toList());
        PurSuppBaseRpcParam purSuppBaseRpcParam = new PurSuppBaseRpcParam();
        purSuppBaseRpcParam.setSuppCodes(suppCodes);
        ApiResult<List<PurSuppBaseRpcDTO>> suppList = purSuppProvider.findBaseRpcDtoByParam(purSuppBaseRpcParam);
        List<PurSuppBaseRpcDTO> suppBaseRpcDTOS = suppList.getData();
        Map<String, PurSuppBaseRpcDTO> suppMap = suppBaseRpcDTOS.stream().collect(Collectors.toMap(PurSuppBaseRpcDTO::getSuppCode, i -> i, (o, n) -> n));

        ItmItemRpcDtoParam itemRpcDtoParam = new ItmItemRpcDtoParam();
        itemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemRpcDTO> itemInfos = itemRpcService.findItemRpcDtoByParam(itemRpcDtoParam);
        Map<String, ItmItemRpcDTO> itemMap = itemInfos.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, i -> i, (o, n) -> n));

        List<String> storeCodes = dataList.stream().filter(d -> UdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(d.getType()))
                .map(ScpSupplyAllocationImportVO::getStoreWhCode).distinct().collect(Collectors.toList());
        Map<String, OrgStoreDetailRpcDTO> storeMap = new HashMap<>();
        if (CollUtil.isNotEmpty(storeCodes)) {
            List<OrgStoreDetailRpcDTO> storeList = orgStoreRpcService.queryByStoreCodes(storeCodes);
            storeMap = storeList.stream().collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode, i -> i, (o, n) -> n));
        }
        List<String> whCodes = dataList.stream().filter(d -> UdcEnum.DEMAND_SET_TYPE_1.getValueCode().equals(d.getType()))
                .map(ScpSupplyAllocationImportVO::getStoreWhCode)
                .distinct().collect(Collectors.toList());
        Map<String, InvWhRpcDTO> whMap = new HashMap<>();
        if (CollUtil.isNotEmpty(whCodes)) {
            InvWhRpcDtoParam whRpcDtoParam = new InvWhRpcDtoParam();
            whRpcDtoParam.setWhCodes(whCodes);
            ApiResult<List<InvWhRpcDTO>> whList = whProvider.findWhDTOByParam(whRpcDtoParam);
            whMap = whList.getData().stream().collect(Collectors.toMap(InvWhRpcDTO::getWhCode, i -> i, (o, n) -> n));
        }
        Map<Integer, List<String>> errorMap = new HashMap<>();
        HashMap<String, List<ScpSupplyAllocationDTO>> supplyAllocationMap = new HashMap<>();
        for (ScpSupplyAllocationImportVO importEntity : dataList) {
            if (!successDocGroup.contains(importEntity.getImportBillNum())) {
                continue;
            }
            errorMap.put(importEntity.getLine(), new ArrayList<>());
            List<ScpSupplyAllocationDTO> allocationDOS = supplyAllocationMap.get(importEntity.getImportBillNum());
            if (CollUtil.isEmpty(allocationDOS)) {
                allocationDOS = new ArrayList<>();
                supplyAllocationMap.put(importEntity.getImportBillNum(), allocationDOS);
            }
            ScpSupplyAllocationDTO allocationDO = new ScpSupplyAllocationDTO();
            allocationDO.setSuppCode(importEntity.getSuppCode());
            allocationDO.setOuCode(importEntity.getOuCode());
            ItmItemRpcDTO itmItemRpcDTO = itemMap.get(importEntity.getItemCode());
            if (itmItemRpcDTO == null) {
                errorMap.get(importEntity.getLine()).add("商品 " + importEntity.getItemCode() + " 不存在");
            } else {
//                if (!itmItemRpcDTO.getItemCateCode().equals(importEntity.getItemCateCode())) {
//                    errorMap.get(importEntity.getLine()).add("商品品类 " + importEntity.getItemCateCode() + " 不包含商品【" + itmItemRpcDTO.getItemCode() + "】");
//                } else {
//                    allocationDO.setItemCode(itmItemRpcDTO.getItemCode());
//                    allocationDO.setItemId(itmItemRpcDTO.getId());
//                    allocationDO.setItemName(itmItemRpcDTO.getItemName());
//                    allocationDO.setItemCateName(itmItemRpcDTO.getItemCateFullName());
//                    allocationDO.setItemCateCode(itmItemRpcDTO.getItemCateCode());
//                }
                allocationDO.setItemCode(itmItemRpcDTO.getItemCode());
                allocationDO.setItemId(itmItemRpcDTO.getId());
                allocationDO.setItemName(itmItemRpcDTO.getItemName());
                allocationDO.setItemCateName(itmItemRpcDTO.getItemCateFullName());
                allocationDO.setItemCateCode(itmItemRpcDTO.getItemCateCode());
            }
            OrgOuRpcDTO orgOuRpcDTO = ouDataMap.get(importEntity.getOuCode());
            if (orgOuRpcDTO == null) {
                errorMap.get(importEntity.getLine()).add("采购公司编码" + importEntity.getOuCode() + " 不存在");
            } else {
                allocationDO.setOuId(orgOuRpcDTO.getId());
                allocationDO.setOuName(orgOuRpcDTO.getOuName());
            }

            if (UdcEnum.DEMAND_SET_TYPE_1.getValueCode().equals(importEntity.getType())) {
                InvWhRpcDTO invWhRpcDTO = whMap.get(importEntity.getStoreWhCode());
                if (invWhRpcDTO == null) {
                    errorMap.get(importEntity.getLine()).add("仓库 " + importEntity.getStoreWhCode() + " 不存在");
                } else {
                    allocationDO.setStoreWhId(invWhRpcDTO.getId());
                    allocationDO.setStoreWhName(invWhRpcDTO.getWhName());
                    allocationDO.setStoreWhCode(importEntity.getStoreWhCode());
                }
                if (!invWhRpcDTO.getOuId().equals(orgOuRpcDTO.getId())) {
                    SupportTransactionPathRpcParam pathRpcParam = new SupportTransactionPathRpcParam();
                    pathRpcParam.setCompanyIdStart(orgOuRpcDTO.getId());
                    pathRpcParam.setCompanyIdEnd(invWhRpcDTO.getOuId());
                    SupportTransactionPathRpcDTO data = supportTransactionPathRpcService.queryByParam(pathRpcParam).getData();
                    log.info("查询启用状态的结算路径结束,返回:{}", JSONObject.toJSONString(data));
                    if (data == null || StrUtil.isBlank(data.getPathCode())) {
                        log.info("从公司 {}到公司 {}的结算路径不存在，请维护", orgOuRpcDTO.getId(), invWhRpcDTO.getOuId());
                        errorMap.get(importEntity.getLine()).add("从公司:" + orgOuRpcDTO.getOuCode() + "到公司:" + invWhRpcDTO.getOuCode() + "的结算路径不存在，请维护");
                    } else {
                        var detailList = data.getDetailList();
                        int size = detailList.size();
                        SupportTransactionPathDRpcDTO dto = detailList.get(size - 1);
                        allocationDO.setSaleOuCode(dto.getCompanyCodeStart());
                        allocationDO.setSaleOuName(dto.getCompanyNameStart());
                        allocationDO.setSaleCustCode(dto.getCustCodeEnd());
                    }
                }
            } else {
                OrgStoreDetailRpcDTO orgStoreWhDTO = storeMap.get(importEntity.getStoreWhCode());
                if (orgStoreWhDTO == null) {
                    errorMap.get(importEntity.getLine()).add("门店 " + importEntity.getStoreWhCode() + " 不存在");
                } else {
                    allocationDO.setStoreWhId(orgStoreWhDTO.getId());
                    allocationDO.setStoreWhName(orgStoreWhDTO.getStoreName());
                    allocationDO.setStoreWhCode(importEntity.getStoreWhCode());
                }
                if (!orgStoreWhDTO.getOuId().equals(orgOuRpcDTO.getId())) {
                    SupportTransactionPathRpcParam pathRpcParam = new SupportTransactionPathRpcParam();
                    pathRpcParam.setCompanyIdStart(orgOuRpcDTO.getId());
                    pathRpcParam.setCompanyIdEnd(orgStoreWhDTO.getOuId());
                    SupportTransactionPathRpcDTO data = supportTransactionPathRpcService.queryByParam(pathRpcParam).getData();
                    log.info("查询启用状态的结算路径结束,返回:{}", JSONObject.toJSONString(data));
                    if (data == null || StrUtil.isBlank(data.getPathCode())) {
                        log.info("从公司 {}到公司 {}的结算路径不存在，请维护", orgOuRpcDTO.getId(), orgStoreWhDTO.getOuId());
                        errorMap.get(importEntity.getLine()).add("从公司:" + orgOuRpcDTO.getOuCode() + "到公司:" + orgStoreWhDTO.getOuCode() + "的结算路径不存在，请维护");
                    } else {
                        var detailList = data.getDetailList();
                        int size = detailList.size();
                        SupportTransactionPathDRpcDTO dto = detailList.get(size - 1);
                        allocationDO.setSaleOuCode(dto.getCompanyCodeStart());
                        allocationDO.setSaleOuName(dto.getCompanyNameStart());
                        allocationDO.setSaleCustCode(dto.getCustCodeEnd());
                    }
                }
            }
            PurSuppBaseRpcDTO suppBaseRpcDTO = suppMap.get(importEntity.getSuppCode());
            if (suppBaseRpcDTO == null) {
                errorMap.get(importEntity.getLine()).add("供应商编码" + importEntity.getSuppCode() + " 不存在");
            } else {
                allocationDO.setSuppId(suppBaseRpcDTO.getId());
                allocationDO.setSuppName(suppBaseRpcDTO.getSuppName());
            }
            allocationDO.setType(importEntity.getType());
            allocationDO.setTypeName(importEntity.getTypeName());
            allocationDO.setAllocation(importEntity.getAllocation());
            allocationDO.setMinimumOrderQuantity(importEntity.getMinimumOrderQuantity());
            allocationDO.setPurchaseLeadTime(importEntity.getPurchaseLeadTime());
            allocationDO.setStartTime(importEntity.getStartTime());
            allocationDO.setEndTime(importEntity.getEndTime());
            allocationDO.setEs1(importEntity.getImportBillNum());
            allocationDO.setLineNo(importEntity.getLine());
            allocationDO.setStatus(Boolean.TRUE);
            allocationDOS.add(allocationDO);
        }
//        同已有数据：
//
//        类型=门店时，门店编码+物料唯一（只有一条）；
//
//        类型=仓库时，仓库编码+物料不唯一，不需要判重；
        Map<String, List<ScpSupplyAllocationImportVO>> storeListMap = dataList.stream().filter(d -> UdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(d.getType()))
                .collect(Collectors.groupingBy(a -> a.getStoreWhCode() + "-" + a.getItemCode()));
        storeListMap.forEach((k, v) -> {
            if (v.size() > 1) {
                v.forEach(a -> {
                    errorMap.get(a.getLine()).add("门店类型的门店编码" + a.getStoreWhCode() + "物料编码" + a.getItemCode() + "已存在");
                });
            }
        });
        //校验数据是否存在
        Set<String> existsList = dataList.stream().map(d -> d.getStoreWhCode() + d.getItemCode()).collect(Collectors.toSet());
        List<ScpSupplyAllocationDO> existList = scpSupplyAllocationDomainService.findByConcatKeys(existsList);
        dataList.forEach(d -> {
            if(!UdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(d.getType())){
                return;
            }
            if (existList.stream().anyMatch(e -> UdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(e.getType())&&e.getStoreWhCode().equals(d.getStoreWhCode()) && e.getItemCode().equals(d.getItemCode()))) {
                deleteWhNetRelation(existList.stream().map(e->e.getId()).collect(Collectors.toList()));
//                errorMap.get(d.getLine()).add("门店类型的门店编码:" + d.getStoreWhCode() + "物料编码:" + d.getItemCode() + "已存在");
            }
        });
        for (String key : supplyAllocationMap.keySet()) {
            List<ScpSupplyAllocationDTO> supplyAllocationDOS = supplyAllocationMap.get(key);
            long count = supplyAllocationDOS.stream().map(d -> d.getStoreWhCode() + d.getType() + d.getItemCode())
                    .distinct().count();
//            BigDecimal sumAlloc = supplyAllocationDOS.stream().map(ScpSupplyAllocationDTO::getAllocation)
//                    .reduce(BigDecimal.ZERO, BigDecimal::add);
//            if (sumAlloc.compareTo(new BigDecimal("100")) != 0) {
//                supplyAllocationDOS.forEach(a -> {
//                    errorMap.get(a.getLineNo()).add("序号组 " + a.getEs1() + "配置比额不符合百分之百");
//                });
//            }
            if(UdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(supplyAllocationDOS.get(0).getType())){
                supplyAllocationDOS.get(0).setAllocation(BigDecimal.valueOf(100));
            }
            //校验数据是否存在
//            Set<String> existsList = supplyAllocationDOS.stream().map(d -> d.getStoreWhCode() + d.getItemCode()).collect(Collectors.toSet());
//            List<ScpSupplyAllocationDO> existList = scpSupplyAllocationDomainService.findByConcatKeys(existsList);
            //查询数据是否重复
//            HashMap<String, Integer> duplicateMap = new HashMap<>();
            for (ScpSupplyAllocationDTO d : supplyAllocationDOS) {
                if (count != 1) {
                    errorMap.get(d.getLineNo()).add("同一序号组 " + d.getEs1() + "【门店/仓库编码-商品编码】必须相同！");
                }
//                if (duplicateMap.get(d.getSuppCode() + d.getStoreWhCode() + d.getType() + d.getItemCode()) == null) {
//                    duplicateMap.put(d.getSuppCode() + d.getStoreWhCode() + d.getType() + d.getItemCode(), d.getLineNo());
//                } else {
//                    errorMap.get(d.getLineNo()).add("序号组 " + d.getEs1() + "数据重复,仓库编码/门店编码-商品存在相同【供应商】组合数据，请检查");
//                }
//                if (CollUtil.isNotEmpty(existList)) {
//                    errorMap.get(d.getLineNo()).add("序号组 " + d.getEs1() + "数据重复，仓库编码/门店编码-商品 配置已存在,请检查");
//                }
            }
        }
        for (Integer line : errorMap.keySet()) {
            if (errorMap.get(line).isEmpty()) {
                continue;
            }
            if (Objects.equals(errorList.get(line - 1), "-")) {
                errorList.set(line - 1, String.join(",", errorMap.get(line)));
            } else {
                errorList.set(line - 1, errorList.get(line - 1) + String.join(",", errorMap.get(line)));
            }
        }
        successDocGroup = getSuccessDocGroup(errorList, dataList);
        List<ScpSupplyAllocationDTO> ddos = new ArrayList<>();
        if (!successDocGroup.isEmpty()) {
            List<String> finalSuccessDocGroup = successDocGroup;
            for (String key : supplyAllocationMap.keySet()) {
                List<ScpSupplyAllocationDTO> supplyAllocationDOS = supplyAllocationMap.get(key);
                List<ScpSupplyAllocationDTO> doList = supplyAllocationDOS.stream()
                        .filter(i -> finalSuccessDocGroup.contains(i.getEs1()))
                        .collect(Collectors.toList());
                if (doList.isEmpty()) {
                    continue;
                }

                ddos.addAll(doList);
            }
//            scpSupplyAllocationDomainService.createBatch(ddos);
        }
        List<String> newResult = new ArrayList<>();
        List<String> errorNewList = new ArrayList<>();
        for (int i = 0; i < errorList.size(); i++) {
            String error = errorList.get(i);
            String importDocGroup = dataList.get(i).getImportBillNum();
            if (!successDocGroup.contains(importDocGroup)) {
                if (Objects.equals(error, "-")) {
                    newResult.add("-");
                } else {
                    newResult.add(error);
                    errorNewList.add(error);
                }
            } else {
                newResult.add(null);
            }
        }
        if(CollUtil.isEmpty(errorNewList)){
            scpSupplyAllocationDomainService.createBatch(ddos);
        }
        return newResult;
    }


    private void checkImportData(List<ScpSupplyAllocationImportVO> dataList, int startRowIndex) {
        // 供应商编码
        List<String> suppCodeList = dataList.stream().map(e -> e.getSuppCode()).collect(Collectors.toList());
        // 门店编码
        List<String> storeWhCodeList = dataList.stream().map(e -> e.getStoreWhCode()).collect(Collectors.toList());
        // 商品编码
        List<String> itemCodeList = dataList.stream().map(e -> e.getItemCode()).collect(Collectors.toList());

        //通过采购域rpc获取供应商归属公司信息
        PurSuppBaseRpcParam purSuppBaseRpcParam = new PurSuppBaseRpcParam();
        purSuppBaseRpcParam.setSuppCodes(suppCodeList);
        List<PurSuppBaseRpcDTO> suppList = purSuppProvider.findBaseRpcDtoByParam(purSuppBaseRpcParam).getData();

        // 门店列表
        List<OrgStoreDetailRpcDTO> orgStoreDTOS = rmiOrgStoreRpcService.queryByStoreCodes(storeWhCodeList);

        // 商品列表
        List<ItmItemBaseRpcDTO> itmItemBaseDTOS = rmiItemService.findItemBaseRpcDtoByParam(ItmItemBaseRpcParam.builder().itemCodes(itemCodeList).build());

        // 提取供应商编号，供应商名称map(whCode,whName)
        Map<String, String> whMap = suppList.stream()
                .collect(Collectors.toMap(PurSuppBaseRpcDTO::getSuppCode, PurSuppBaseRpcDTO::getSuppName));
        // 提取门店编号，门店名称map(storeCode,storeName)
        Map<String, String> storeMap = orgStoreDTOS.stream()
                .collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode, OrgStoreDetailRpcDTO::getStoreName));
        // 提取商品编号，商品名称map(itemCode,itemName)
        Map<String, String> itemMap = itmItemBaseDTOS.stream()
                .collect(Collectors.toMap(ItmItemBaseRpcDTO::getItemCode, ItmItemBaseRpcDTO::getItemName));

        // 遍历scpStoreOrderFreightImportVOList，匹配每一个仓库，门店，商品的code与名称是否匹配，不匹配直接返回报错信息
        for (int i = 0; i < dataList.size(); i++) {

            ScpSupplyAllocationImportVO vo = dataList.get(i);
            int rowNum = startRowIndex+i; // Excel 行号

            // 校验供应商
            if(vo.getSuppCode()!=null){
                if(vo.getSuppName()==null){
                    throw new BusinessException("第 " + rowNum + " 行：供应商编码 " + vo.getSuppCode() + " 对应供应商名称为空，请检查！");
                }else if (!whMap.containsKey(vo.getSuppCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：供应商编码 " + vo.getSuppName() + " 不存在，请检查！");
                }else if(!whMap.get(vo.getSuppCode()).equals(vo.getSuppName())){
                    throw new BusinessException("第 " + rowNum + " 行：供应商编码 " + vo.getSuppCode() + " 与供应商名称 " + vo.getSuppName()+"不匹配，请检查！");
                }
            }else{
                throw new BusinessException("第 " + rowNum + " 行：供应商编码 " + vo.getStoreWhCode() + " 为空，请检查！");
            }

            // 校验门店
            if (vo.getStoreWhCode() != null) {
                if (vo.getStoreWhName() == null) {
                    throw new BusinessException("第 " + rowNum + " 行：需求仓库/门店编码 " + vo.getStoreWhCode() + " 对应需求仓库/门店名称为空，请检查！");
                } else if (!storeMap.containsKey(vo.getStoreWhCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：需求仓库/门店编码 " + vo.getStoreWhCode() + " 不存在，请检查！");
                } else if (!storeMap.get(vo.getStoreWhCode()).equals(vo.getStoreWhName())) {
                    throw new BusinessException("第 " + rowNum + " 行：需求仓库/门店编码 " + vo.getStoreWhCode() + " 与需求仓库/门店名称 " + vo.getStoreWhName() + " 不匹配，请检查！");
                }
            }else{
                throw new BusinessException("第 " + rowNum + " 行：需求仓库/门店编码 " + vo.getStoreWhCode() + " 为空，请检查！");
            }


            // 校验商品
            if (vo.getItemCode() != null) {
                if (vo.getItemName() == null) {
                    throw new BusinessException("第 " + rowNum + " 行：物料编码 " + vo.getItemCode() + " 对应物料名称为空，请检查！");
                } else if (!itemMap.containsKey(vo.getItemCode())) {
                    throw new BusinessException("第 " + rowNum + " 行：物料编码 " + vo.getItemCode() + " 不存在，请检查！");
                } else if (!itemMap.get(vo.getItemCode()).equals(vo.getItemName())) {
                    throw new BusinessException("第 " + rowNum + " 行：物料编码 " + vo.getItemCode() + " 与物料名称 " + vo.getItemName() + " 不匹配，请检查！");
                }
            }else{
                throw new BusinessException("第 " + rowNum + " 行：物料编码 " + vo.getStoreWhCode() + " 为空，请检查！");
            }
        }
    }

    @Override
    public PagingVO<ScpSupplyAllocationPageVO> scpmanAuthorityPage(ScpSupplyAllocationPageParamVO queryParam) {
        PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
        if (userAuthority.isEmpty()) {
            return PagingVO.<ScpSupplyAllocationPageVO>builder()
                    .total(0L)
                    .records(new ArrayList<>())
                    .build();
        }
        extractedAuthorityParam(queryParam, userAuthority);
        return page(queryParam);
    }

    @Override
    public Map<String, List<ScpSupplyAllocationRpcDTO>> findSupplyAllocationByParam(String demandWhStCode, String type, List<String> itemCodes) {
        ScpSupplyAllocationRpcDtoParam param = new ScpSupplyAllocationRpcDtoParam();
        param.setType(type);
        param.setStoreWhCode(demandWhStCode);
        param.setItemCodes(itemCodes);
        List<ScpSupplyAllocationRpcDTO> supplyAllocationList = scpSupplyAllocationRepoProc.findSupplyAllocationRpcByParam(param);
        if (CollUtil.isNotEmpty(supplyAllocationList)) {
            return supplyAllocationList.stream().collect(Collectors.groupingBy(ScpSupplyAllocationRpcDTO::getItemCode));
        }
        return new HashMap<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchDelete(List<Long> ids) {
        List<ScpSupplyAllocationDO> allocationDOS = scpSupplyAllocationDomainService.findByIds(ids);
        Map<String, List<ScpSupplyAllocationDO>> suppMap = allocationDOS.stream().filter(row -> Boolean.TRUE.equals(row.getStatus())).collect(Collectors.groupingBy(row -> row.getStoreWhCode() + row.getItemCode() + row.getSuppCode()));
        for (String key : suppMap.keySet()) {
            List<ScpSupplyAllocationDO> suppList = suppMap.get(key);
            ScpSupplyAllocationDO scpSupplyAllocationDO = suppList.get(0);
            BigDecimal sumAlloc = suppList.stream().map(ScpSupplyAllocationDO::getAllocation)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            if (sumAlloc.compareTo(new BigDecimal("100")) != 0) {
                throw new BusinessException("门店/仓库【" + scpSupplyAllocationDO.getStoreWhName() + "】,商品【  " + scpSupplyAllocationDO.getItemName() + "】配置比额不符合百分之百");
            }
        }
        List<Long> delIds = allocationDOS.stream().map(ScpSupplyAllocationDO::getId).collect(Collectors.toList());
        scpSupplyAllocationDomainService.deleteBatch(delIds);
    }

    private void extractedAuthorityParam(BaseScpmanAuthorityParam queryParam, PagingVO<ScpManAuthorityPageRespVO> userAuthority) {
        List<String> whCodes = userAuthority.stream()
                .filter(v -> 1 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        List<String> stores = userAuthority.stream()
                .filter(v -> 0 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        queryParam.setWhCodes(whCodes);
        queryParam.setStores(stores);
        queryParam.setScpmanAuthority(Boolean.TRUE);
    }


    private List<String> validResult(List<String> errorList) {
        errorList.replaceAll(new UnaryOperator<String>() {
            @Override
            public String apply(String s) {
                if (StringUtils.isEmpty(s)) {
                    return "-";
                }
                return s;
            }
        });
        return errorList;
    }


    private List<String> getSuccessDocGroup(List<String> errorList, List<ScpSupplyAllocationImportVO> dataList) {
        List<String> docGroupNo = dataList.stream()
                .filter(i -> org.springframework.util.StringUtils.hasText(i.getImportBillNum()))
                .map(ScpSupplyAllocationImportVO::getImportBillNum)
                .distinct().collect(Collectors.toList());
        for (int i = 0; i < dataList.size(); i++) {
            ScpSupplyAllocationImportVO entity = dataList.get(i);
            if (!Objects.equals("-", errorList.get(i))) {
                docGroupNo.remove(entity.getImportBillNum());
            }
        }
        return docGroupNo;
    }


    //校验入参
    private void checkImportParam(List<ScpSupplyAllocationImportVO> dataList, List<String> errorList) {
        Map<Integer, List<String>> errorMap = new HashMap<>();
        Map<String, String> typeMap = MapUtil.reverse(systemService.getValueMapByUdcCode("yst-suplan", "DEMAND_SET_TYPE"));
        Map<String, String> uomMap = MapUtil.reverse(systemService.getValueMapByUdcCode("yst-supp", "UOM"));
        for (int i = 0; i < dataList.size(); i++) {
            ScpSupplyAllocationImportVO importEntity = dataList.get(i);
            int line = i + 1;
            importEntity.setLine(line);
            errorMap.put(importEntity.getLine(), new ArrayList<>());
            if (importEntity.getImportBillNum() == null) {
                errorMap.get(line).add("列[序号]不能为空");
            }
            if (StringUtils.isEmpty(importEntity.getItemCode())) {
                errorMap.get(line).add("列[商品编码]不能为空");
            }
//            if (StringUtils.isEmpty(importEntity.getItemCateCode())) {
//                errorMap.get(line).add("列[商品品类]不能为空");
//            }
            if (StringUtils.isEmpty(importEntity.getStoreWhCode())) {
                errorMap.get(line).add("列[仓库编码/门店编码]不能为空");
            }
            if (StringUtils.isEmpty(importEntity.getTypeName())) {
                errorMap.get(line).add("列[类型]不能为空");
            } else {
                String lineType = typeMap.get(importEntity.getTypeName());
                if (StringUtils.isEmpty(lineType)) {
                    errorMap.get(importEntity.getLine()).add("列[类型名]" + importEntity.getTypeName() + " 不存在");
                } else {
                    importEntity.setType(lineType);
                }
            }
            if (StringUtils.isEmpty(importEntity.getSuppCode())) {
                errorMap.get(line).add("列[供应商编码]不能为空");
            }
            if (importEntity.getAllocation() == null) {
                errorMap.get(line).add("列[配额]不能为空");
            }
        }
        for (Integer line : errorMap.keySet()) {
            errorList.add(String.join(",", errorMap.get(line)));
        }
    }

    private List<ScpSupplyAllocationResultVO.ResultVO> getResultVOS(List<ScpSupplyAllocatioSaveVO.ScpSupplyAllocatio> supplyAllocatios) {
        return supplyAllocatios.stream().map(w -> {
            ScpSupplyAllocationResultVO.ResultVO resultVO = new ScpSupplyAllocationResultVO.ResultVO();
            resultVO.setSuppId(w.getId());
            resultVO.setSuppCode(w.getSuppCode());
            resultVO.setSuppName(w.getSuppName());
            resultVO.setType(w.getType());
            resultVO.setStoreWhId(w.getStoreWhId());
            resultVO.setStoreWhCode(w.getStoreWhCode());
            resultVO.setType(w.getType());
            resultVO.setItemCode(w.getItemCode());
            resultVO.setItemId(w.getItemId());
            resultVO.setItemName(w.getItemName());
            resultVO.setItemCateCode(w.getItemCateCode());
            resultVO.setItemCateName(w.getItemCateName());
            resultVO.setAllocation(w.getAllocation());
            resultVO.setMinimumOrderQuantity(w.getMinimumOrderQuantity());
            resultVO.setPurchaseLeadTime(w.getPurchaseLeadTime());
            resultVO.setLineNo(w.getLineNo());
            return resultVO;
        }).collect(Collectors.toList());
    }
}
