package com.elitesland.scp.application.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderImportEntity;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderImportSaveVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemBatchParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDMgmtSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSaveVO;
import com.elitesland.scp.domain.service.order.ScpDemandSetDomainService;
import com.elitesland.scp.domain.service.setting.ScpOrderSettingDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandSetDTO;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.scp.utils.ValidatedList;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: ryan.xu
 * @Date: 2024/9/10 17:01
 * @Description:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class ScpDemandOrderImportService implements DataImport<ScpDemandOrderImportEntity> {


    private static final String LINE = "\n";
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    private final TransactionTemplate transactionTemplate;
    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpDemandOrderDService scpDemandOrderDService;
    private final ScpOrderSettingDomainService scpOrderSettingDomainService;
    private final ScpDemandSetDomainService scpDemandSetDomainService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final RmiItemService rmiItemService;
    private final UdcProvider sysUdcService;

    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_demand_order_import";
    }

    @Override
    public List<String> executeImport(List<ScpDemandOrderImportEntity> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg = checkAndSaveData(dataList, startRowIndex);
            if (errorMsg != null) {
                log.info("订货单导入完成,错误信息如下:{}", JSONUtil.toJsonStr(errorMsg));
            } else {
                log.info("订货单导入完成,未发生错误");
            }
            return errorMsg;
        } catch (Exception e) {
            for (ScpDemandOrderImportEntity entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpDemandOrderImportEntity> dataList, int startRowIndex) {

        Map<Integer, String> errMsgMap = new LinkedHashMap<>();
        Integer index = 0;
        //初始化行号，和错误信息容器
        for (ScpDemandOrderImportEntity entity : dataList) {
            entity.setRowNo(index);
            errMsgMap.put(index, null);
            index++;
        }

        log.info("startRowIndex的值:{}", startRowIndex);
        // 记录明细有错误的订单序号
        Set<String> errorDocNumSet = new HashSet<>();
        List<ScpDemandOrderImportSaveVO> dataToSave = prepareData(dataList, errMsgMap, errorDocNumSet);
        log.info("订货单导入，准备好的数据:{},错误信息:{}", JSONUtil.toJsonStr(dataToSave), JSONUtil.toJsonStr(errMsgMap));
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        //一个一个保存
        for (ScpDemandOrderImportSaveVO vo : dataToSave) {
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                            Long masId = scpDemandOrderService.saveDemandOrder(vo.getHeaderSaveVO());
                            scpDemandOrderDService.batchSaveDemandOrderDMgmt(masId, vo.getDetailSaveVOs());
                        } catch (Exception e) {
                            log.error("订货单导入报错：" + e);
                            // 回滚数据
                            transactionStatus.setRollbackOnly();
                        }
                        return null;
                    }
            );
        }
        long docNumber = dataList.stream().map(ScpDemandOrderImportEntity::getSeqNo).distinct().count();
        String orderMsg = String.format("文件共包含%d条数据，识别出订货单%d单，其中导入成功%d单，失败%d单", dataList.size(), docNumber,
                docNumber - errorDocNumSet.size(),
                errorDocNumSet.size());
        log.info("写入自定义提示语:{}", orderMsg);
        HttpServletUtil.currentRequest().setAttribute("orderMsg", orderMsg);
        return new ArrayList<>(errMsgMap.values());
    }

    private List<ScpDemandOrderImportSaveVO> prepareData(List<ScpDemandOrderImportEntity> dataList,
                                                         Map<Integer, String> errMsgMap, Set<String> errorDocNumSet) {
        List<ScpDemandOrderImportSaveVO> resultList = new ArrayList<>();
        Map<String, List<ScpDemandOrderImportEntity>> seqNoMap = dataList.stream().collect(Collectors.groupingBy(ScpDemandOrderImportEntity::getSeqNo));
        List<String> demandSetNameList = dataList.stream().collect(Collectors.groupingBy(ScpDemandOrderImportEntity::getDemandSetName)).keySet().stream().distinct().collect(Collectors.toList());
        ScpDemandSetParamVO paramVO = ScpDemandSetParamVO.builder().demandNames(demandSetNameList).expireTime(LocalDateTime.now()).build();
        List<ScpDemandSetDTO> scpDemandSetDTOList = scpDemandSetDomainService.findDemandSetByParam(paramVO);
        if (CollUtil.isEmpty(scpDemandSetDTOList)) {
            addErrMsg(errMsgMap, dataList.stream().map(ScpDemandOrderImportEntity::getRowNo).collect(Collectors.toList()), "订货集不存在");
            errorDocNumSet.addAll(dataList.stream().map(ScpDemandOrderImportEntity::getSeqNo).collect(Collectors.toList()));
            return null;
        }
        Map<String, String> uomMap = sysUdcService.getValueMapByUdcCode("yst-supp", "UOM");
        List<ScpOrderSettingRespVO> enabledSetting = scpOrderSettingDomainService.findEnabledSetting();
        Map<String, ScpOrderSettingRespVO> settingMap = enabledSetting.stream().collect(Collectors.toMap(row -> row.getDocTypeName() + "@" + row.getType(), Function.identity()));
        Map<String, ScpDemandSetDTO> scpDemandSetDTOMap = scpDemandSetDTOList.stream().collect(Collectors.toMap(ScpDemandSetDTO::getDemandName, Function.identity()));
        for (Map.Entry<String, List<ScpDemandOrderImportEntity>> entry : seqNoMap.entrySet()) {
            ScpDemandOrderImportSaveVO scpDemandOrderImportSaveVO = new ScpDemandOrderImportSaveVO();
            ScpDemandOrderSaveVO result = new ScpDemandOrderSaveVO();
            List<ScpDemandOrderImportEntity> itemList = entry.getValue();
            ScpDemandOrderImportEntity header = itemList.get(0);
            List<Integer> rowNoList = itemList.stream().map(ScpDemandOrderImportEntity::getRowNo).collect(Collectors.toList());
            //订货集属性
            ScpDemandSetDTO scpDemandSetDTO = scpDemandSetDTOMap.get(header.getDemandSetName());
            if (scpDemandSetDTO == null) {
                addErrMsg(errMsgMap, rowNoList, "订货集:" + header.getDemandSetName() + "不存在");
                errorDocNumSet.add(entry.getKey());
                continue;
            }
            result.setDemandId(scpDemandSetDTO.getId());
            result.setDemandCode(scpDemandSetDTO.getDemandCode());
            result.setDemandName(scpDemandSetDTO.getDemandName());
            result.setDemandDate(scpDemandSetDTO.getDemandDate());
            result.setType(scpDemandSetDTO.getType());
            result.setDemandWhStCode(header.getDemandWhStCode());
            result.setDemandWhStName(header.getDemandWhStName());
            String docCls = ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(scpDemandSetDTO.getType()) ? ScpUdcEnum.OB_DOC_CLS_ST.getValueCode() : ScpUdcEnum.OB_DOC_CLS_WH.getValueCode();
            result.setDocCls(docCls);
            //单据类型配置
            ScpOrderSettingRespVO scpOrderSettingRespVO = settingMap.get(header.getDocTypeName() + "@" + result.getType());
            if (scpOrderSettingRespVO == null) {
                addErrMsg(errMsgMap, rowNoList, "单据类型:" + header.getDocTypeName() + "不存在");
                errorDocNumSet.add(entry.getKey());
                continue;
            }
            result.setDocType(scpOrderSettingRespVO.getDocType());
            if (ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(result.getType())) {
                List<OrgStoreBaseRpcDTO> orgStores = rmiOrgStoreRpcService.findOrgStoreBaseByParam(List.of(result.getDemandWhStCode()));
                if (CollUtil.isEmpty(orgStores)) {
                    addErrMsg(errMsgMap, rowNoList, "门店编码:" + result.getDemandWhStCode() + "不存在");
                    errorDocNumSet.add(entry.getKey());
                    continue;
                }
                OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = orgStores.get(0);
                result.setDemandWhStId(orgStoreBaseRpcDTO.getId());
                result.setDemandWhStName(orgStoreBaseRpcDTO.getStoreName());
                result.setStoreLevel(orgStoreBaseRpcDTO.getStoreLevel());
                result.setStoreType2(orgStoreBaseRpcDTO.getStoreType2());
                result.setOuCode(orgStoreBaseRpcDTO.getOuCode());
                result.setBrandCode(orgStoreBaseRpcDTO.getBrandCode());
            } else {
                InvWhRpcSimpleDTO whRpcSimpleDTO = rmiInvStkRpcService.findSimpleWhByCode(result.getDemandWhStCode());
                if (whRpcSimpleDTO == null) {
                    addErrMsg(errMsgMap, rowNoList, "仓库编码:" + result.getDemandWhStCode() + "不存在");
                    errorDocNumSet.add(entry.getKey());
                    continue;
                }
                result.setDemandWhStId(whRpcSimpleDTO.getWhId());
                result.setDemandWhStName(whRpcSimpleDTO.getWhName());
                result.setOuCode(whRpcSimpleDTO.getOuCode());
            }
            scpDemandOrderImportSaveVO.setHeaderSaveVO(result);

            ScpDemandOrderItemBatchParamVO scpDemandOrderItemBatchParamVO = new ScpDemandOrderItemBatchParamVO();

            scpDemandOrderItemBatchParamVO.setDemandDate(result.getDemandDate());
            scpDemandOrderItemBatchParamVO.setDemandWhStCode(result.getDemandWhStCode());
            scpDemandOrderItemBatchParamVO.setType(result.getType());
            scpDemandOrderItemBatchParamVO.setDocType(result.getDocType());

            List<String> itemCodes = itemList.stream().map(ScpDemandOrderImportEntity::getItemCode).distinct().collect(Collectors.toList());
            List<ItmItemBusinessRpcDTO> itmItemBusinessList =
                    rmiItemService.findItmItemBusinessByBUCodeAndBrandAndItemList(result.getOuCode(), result.getBrandCode(), itemCodes);
            Map<String, List<ItmItemBusinessRpcDTO>> itemMap = itmItemBusinessList.stream().collect(Collectors.groupingBy(ItmItemBusinessRpcDTO::getItemCode));

            List<ScpDemandOrderItemBatchParamVO.Item> batchParamList = new ArrayList<>();
            // 原子integer
            AtomicBoolean isValid = new AtomicBoolean(true);
            itemList.forEach(row -> {
                List<ItmItemBusinessRpcDTO> businessItems = itemMap.get(row.getItemCode());
                if (CollUtil.isEmpty(businessItems)) {
                    addErrMsg(errMsgMap, row.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "商品", "商品编码:" + row.getItemCode() + "关联经营目录不存在"));
                    isValid.getAndSet(false);
                    return;
                }
                ItmItemBusinessRpcDTO businessItem = businessItems.get(0);
                ScpDemandOrderItemBatchParamVO.Item item = new ScpDemandOrderItemBatchParamVO.Item();
                item.setItemId(businessItem.getId());
                item.setItemCode(businessItem.getItemCode());
                item.setItemName(businessItem.getItemName());
                item.setItemCateCode(businessItem.getItemCateCode());
                item.setSpuId(businessItem.getSpuId());
                item.setSpuItemCode(businessItem.getSpuCode());
                item.setSpuItemName(businessItem.getSpuName());
                item.setUom(businessItem.getUom2());
                item.setUomName(uomMap.get(businessItem.getUom2()));
                item.setDemandQuantity(row.getDemandQuantity());
                item.setQty2(SysUtils.processQtyScale(row.getDemandQuantity().multiply(businessItem.getUomRatio2())));
                item.setUom2(businessItem.getUom());
                item.setUom2Name(uomMap.get(businessItem.getUom()));
                item.setDecimalPlaces(businessItem.getDecimalPlaces());
                item.setUomRatio(businessItem.getUomRatio2());
                item.setItemType(businessItem.getItemType2());
                batchParamList.add(item);
            });
            if (!isValid.get()) {
                errorDocNumSet.add(entry.getKey());
                continue;
            }
            scpDemandOrderItemBatchParamVO.setItemList(batchParamList);
            Map<String, List<ScpDemandOrderDRespVO>> itemListBatchMap = scpDemandOrderService.getItemListBatch(scpDemandOrderItemBatchParamVO);


            for (ScpDemandOrderImportEntity entity : itemList) {
                List<ScpDemandOrderDRespVO> orderDRespVOS = itemListBatchMap.get(entity.getItemCode());
                if (CollUtil.isEmpty(orderDRespVOS)) {
                    addErrMsg(errMsgMap, entity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "商品供货关系", "商品供货关系不存在,请联系管理员维护"));
                    isValid.getAndSet(false);
                    break;
                }
                orderDRespVOS.forEach(row -> {
                    String ouCode = row.getOuCode();
                    if (StrUtil.isBlank(ouCode)) {
                        return;
                    }
                    if (row.getPrice() == null) {
                        addErrMsg(errMsgMap, entity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "价格", "商品编码【" + row.getItemCode() + "】,公司编码【" + ouCode + "】" + "未配置结算价格信息"));
                        isValid.getAndSet(false);
                    }
                });
            }
            if (!isValid.get()) {
                errorDocNumSet.add(entry.getKey());
                continue;
            }
            List<ScpDemandOrderDMgmtSaveVO> mgmtSaveVOS = new ArrayList<>();
            itemListBatchMap.forEach((key, value) -> {
                ScpDemandOrderDRespVO scpDemandOrderDRespVO = value.get(0);
                ScpDemandOrderDMgmtSaveVO saveVO = new ScpDemandOrderDMgmtSaveVO();
                saveVO.setSpuItemName(scpDemandOrderDRespVO.getSpuItemName());
                saveVO.setSpuItemCode(scpDemandOrderDRespVO.getSpuItemCode());
                saveVO.setItemId(scpDemandOrderDRespVO.getItemId());
                saveVO.setItemCode(scpDemandOrderDRespVO.getItemCode());
                saveVO.setItemName(scpDemandOrderDRespVO.getItemName());
                saveVO.setItemType(scpDemandOrderDRespVO.getItemType());
                saveVO.setDemandQuantity(scpDemandOrderDRespVO.getDemandQuantity());
                saveVO.setCurrency("CNY");

                ValidatedList<ScpDemandOrderDMgmtSaveVO.SupplyItem> supplyItemList = value.stream().map(row -> {
                    ScpDemandOrderDMgmtSaveVO.SupplyItem supplyItem = new ScpDemandOrderDMgmtSaveVO.SupplyItem();
                    supplyItem.setRatio(row.getRatio());
                    supplyItem.setAllocationDeQuantity(row.getAllocationDeQuantity());
                    supplyItem.setAllocationQuantity(row.getAllocationQuantity());
                    supplyItem.setSupplyType(row.getSupplyType());
                    supplyItem.setSuppWhId(row.getSuppWhId());
                    supplyItem.setSuppWhCode(row.getSuppWhCode());
                    supplyItem.setSuppWhName(row.getSuppWhName());
                    supplyItem.setOuId(row.getOuId());
                    supplyItem.setOuCode(row.getOuCode());
                    supplyItem.setOuName(row.getOuName());
                    supplyItem.setIsCalculated(Boolean.FALSE);
                    supplyItem.setIsPushed(Boolean.FALSE);
                    supplyItem.setUnit(row.getUnit());
                    supplyItem.setUnitName(row.getUnitName());
                    supplyItem.setUom2(row.getUom2());
                    supplyItem.setUom2Name(row.getUom2Name());
                    supplyItem.setUomRatio(row.getUomRatio());
                    supplyItem.setDecimalPlaces(row.getDecimalPlaces());
                    supplyItem.setQty2(row.getQty2());
                    supplyItem.setPrice(row.getPrice());
                    supplyItem.setFreightFlag(row.getFreightFlag());
                    supplyItem.setFreightRatio(row.getFreightRatio());
                    supplyItem.setSaleCustCode(row.getSaleCustCode());
                    supplyItem.setSaleOuName(row.getSaleOuName());
                    supplyItem.setSaleOuCode(row.getSaleOuCode());
                    supplyItem.setAllocationQuantity(row.getAllocationQuantity());
                    return supplyItem;
                }).collect(Collectors.toCollection(ValidatedList::new));
                saveVO.setItemList(supplyItemList);
                mgmtSaveVOS.add(saveVO);
            });
            scpDemandOrderImportSaveVO.setDetailSaveVOs(mgmtSaveVOS);
            resultList.add(scpDemandOrderImportSaveVO);
        }
        return resultList;
    }


    public static void addErrMsg(Map<Integer, String> errMsgMap, Integer rowNo, String msg) {
        addErrMsg(errMsgMap, Collections.singletonList(rowNo), msg);
    }

    public static void addErrMsg(Map<Integer, String> errMsgMap, List<Integer> rowNos, String msg) {
        for (Integer rowNo : rowNos) {
            if (errMsgMap.containsKey(rowNo) && errMsgMap.get(rowNo) != null) {
                String newMsg = errMsgMap.get(rowNo) + LINE + msg;
                errMsgMap.put(rowNo, newMsg);
            } else {
                errMsgMap.put(rowNo, msg);
            }
        }
    }
}
