package com.elitesland.scp.infr.repo.order;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDListParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderPayInfoRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderRespVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.domain.entity.authority.QScpsmanAuthorityDDO;
import com.elitesland.scp.domain.entity.authority.QScpsmanAuthorityDO;
import com.elitesland.scp.domain.entity.order.QScpDemandOrderDDO;
import com.elitesland.scp.domain.entity.order.QScpDemandOrderDO;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.domain.entity.scpsman.QScpsmanInfoDO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderRelateDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
public class ScpDemandOrderDRepoProc {
    private final JPAQueryFactory jpaQueryFactory;
    private final QScpDemandOrderDO scpDemandOrderDO = QScpDemandOrderDO.scpDemandOrderDO;
    private final QScpDemandOrderDDO scpDemandOrderDDO = QScpDemandOrderDDO.scpDemandOrderDDO;
    private static final QScpsmanAuthorityDO scpsmanAuthorityDO = QScpsmanAuthorityDO.scpsmanAuthorityDO;
    private static final QScpsmanAuthorityDDO scpsmanAuthorityDDO = QScpsmanAuthorityDDO.scpsmanAuthorityDDO;
    private static final QScpsmanInfoDO scpsmanInfoDO = QScpsmanInfoDO.scpsmanInfoDO;

    private final QBean<ScpDemandOrderDDTO> itemList = Projections.bean(
            ScpDemandOrderDDTO.class,
            scpDemandOrderDDO.id,
            scpDemandOrderDDO.itemId,
            scpDemandOrderDDO.recvQty,
            scpDemandOrderDDO.quantity,
            scpDemandOrderDDO.planQuantity,
            scpDemandOrderDDO.isPushed,
            scpDemandOrderDDO.payStatus,
            scpDemandOrderDDO.freightLineFlag
    );

    public List<ScpDemandOrderDDTO> findByMasId(Long masId) {
        var jpaQuery = jpaQueryFactory.select(itemList)
                .from(scpDemandOrderDDO)
                .where(scpDemandOrderDDO.masId.eq(masId));
        return jpaQuery.fetch();
    }

    public long deleteByIds(List<Long> ids) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scpDemandOrderDDO.id.in(ids));
        var delete = jpaQueryFactory.delete(scpDemandOrderDDO)
                .where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }

    public void updatePlanQtyAndAmtById(Long id, BigDecimal planQty) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.planQuantity, planQty)
                .set(scpDemandOrderDDO.isCalculated, Boolean.TRUE)
                .set(scpDemandOrderDDO.planAmt, scpDemandOrderDDO.price.multiply(planQty))
                .where(scpDemandOrderDDO.id.eq(id))
                .execute();
    }

    public void updateAllocQtyById(Long id, BigDecimal qty) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.allocationQuantity, qty)
                .where(scpDemandOrderDDO.id.eq(id))
                .execute();
    }

    public void updateErrorMsgById(Long dId, String errorMsg) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.syncMsg, errorMsg)
                .set(scpDemandOrderDDO.isPushed, false)
                .where(scpDemandOrderDDO.id.eq(dId))
                .execute();
    }

    public void updateSrcDocInfoById(Long id, Long srcDocId, String srcDocNo, String srcLineNO, String docCls) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.srcDocId, srcDocId)
                .set(scpDemandOrderDDO.srcDocNo, srcDocNo)
                .set(scpDemandOrderDDO.srcDocLineNo, srcLineNO)
                .set(scpDemandOrderDDO.srcDocCls, docCls)
                .where(scpDemandOrderDDO.id.eq(id))
                .execute();
    }

    public void clearSrcDocInfo(Long docId) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .setNull(scpDemandOrderDDO.srcDocId)
                .setNull(scpDemandOrderDDO.srcDocNo)
                .setNull(scpDemandOrderDDO.srcDocCls)
                .setNull(scpDemandOrderDDO.srcDocLineNo)
                .setNull(scpDemandOrderDDO.allocationQuantity)
                .where(scpDemandOrderDDO.srcDocId.eq(docId))
                .execute();
    }

    public void resetSrcDocQuantity(Long dId, BigDecimal qty) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.quantity, scpDemandOrderDDO.quantity.subtract(qty))
                .where(scpDemandOrderDDO.id.eq(dId))
                .execute();
    }

    public Map<Long, Long> getDemandSumItem(List<Long> demandIds) {
        //查询当前登录用户
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtil.isEmpty(currentUserDTO)) {
            throw new BusinessException("当前登录人查询异常");
        }
        String username = currentUserDTO.getDetail().getUsername();
        var subQuery = jpaQueryFactory.select(scpsmanAuthorityDO.id).distinct()
                .from(scpsmanAuthorityDO)
                .leftJoin(scpsmanInfoDO).on(scpsmanInfoDO.scpsmanNo.eq(scpsmanAuthorityDO.scpsmanNo))
                .where(scpsmanAuthorityDO.enableStatus.eq(Boolean.TRUE).and(scpsmanInfoDO.loginAccount.eq(username)));
        List<Long> authIds = subQuery.fetch();
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ScpDemandOrderRespVO.class,
                        scpDemandOrderDO.demandId,
                        scpDemandOrderDDO.id.count().as("count")))
                .from(scpDemandOrderDO)
                .leftJoin(scpDemandOrderDDO).on(scpDemandOrderDO.id.eq(scpDemandOrderDDO.masId))
                .leftJoin(scpsmanAuthorityDDO).on(scpDemandOrderDO.demandWhStCode.eq(scpsmanAuthorityDDO.stWhCode));
        jpaQuery.where(scpDemandOrderDO.demandId.in(demandIds));
        jpaQuery.where(scpsmanAuthorityDDO.masId.in(authIds));
        jpaQuery.groupBy(scpDemandOrderDO.demandId);
        var orderDRespVOS = jpaQuery.fetch();
        if (orderDRespVOS.isEmpty()) {
            return new HashMap<>();
        }
        return orderDRespVOS.stream().collect(Collectors.toMap(ScpDemandOrderRespVO::getDemandId, ScpDemandOrderRespVO::getCount));
    }

    public long deleteBySourceIds(List<Long> sourceIds) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scpDemandOrderDDO.sourceId.in(sourceIds));
        var delete = jpaQueryFactory.delete(scpDemandOrderDDO)
                .where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }

    public BigDecimal findPushedItemMasId(Long masId) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ScpDemandOrderDDTO.class,
                        scpDemandOrderDDO.lineNo,
                        scpDemandOrderDDO.id))
                .from(scpDemandOrderDDO);
        jpaQuery.where(scpDemandOrderDDO.masId.in(masId));
        jpaQuery.orderBy(scpDemandOrderDDO.lineNo.asc()).limit(1);
        var scpDemandOrderDDTO = jpaQuery.fetchOne();
        return scpDemandOrderDDTO != null ? scpDemandOrderDDTO.getLineNo() : BigDecimal.ZERO;
    }

    @Transactional(rollbackFor = Exception.class)
    public long deleteByMasIdAndIsPushed(Long masId, Boolean isPushed) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scpDemandOrderDDO.masId.eq(masId));
        predicates.add(scpDemandOrderDDO.isPushed.eq(isPushed));
        var delete = jpaQueryFactory.delete(scpDemandOrderDDO)
                .where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }

    public List<Long> getMasIdByDId(List<Long> dIds) {
        return jpaQueryFactory.select(scpDemandOrderDDO.masId)
                .from(scpDemandOrderDDO)
                .where(scpDemandOrderDDO.id.in(dIds))
                .fetch();
    }

    public List<ScpDemandOrderPayInfoRespVO> findPayInfo(Long masId) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ScpDemandOrderPayInfoRespVO.class,
                scpDemandOrderDDO.srcDocId,
                scpDemandOrderDDO.srcDocNo,
                scpDemandOrderDDO.srcDocCls,
                scpDemandOrderDDO.saleOuName,
                scpDemandOrderDDO.saleOuCode,
                scpDemandOrderDDO.payStatus,
                scpDemandOrderDDO.planAmt.sum().as("amt")
        )).from(scpDemandOrderDDO);
        jpaQuery.where(scpDemandOrderDDO.masId.eq(masId));
        jpaQuery.groupBy(scpDemandOrderDDO.srcDocId, scpDemandOrderDDO.srcDocNo, scpDemandOrderDDO.srcDocCls,
                scpDemandOrderDDO.saleOuName, scpDemandOrderDDO.saleOuCode, scpDemandOrderDDO.payStatus);
        return jpaQuery.fetch();
    }

    public List<ScpDemandOrderDDO> findByMasIdAndIsPushed(Long masId, Boolean isPushed) {
        return jpaQueryFactory.select(scpDemandOrderDDO)
                .from(scpDemandOrderDDO)
                .where(scpDemandOrderDDO.masId.eq(masId).and(scpDemandOrderDDO.isPushed.eq(isPushed)))
                .fetch();
    }

    public void initAllocQuantity(Long masId) {
        jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.allocationQuantity, scpDemandOrderDDO.planQuantity)
                .where(scpDemandOrderDDO.masId.eq(masId))
                .execute();
    }

    public LocalDateTime findDocCreateTimeBySrcDocId(Long srcDocId) {
        return jpaQueryFactory.selectDistinct(scpDemandOrderDO.createTime)
                .from(scpDemandOrderDO)
                .leftJoin(scpDemandOrderDDO).on(scpDemandOrderDO.id.eq(scpDemandOrderDDO.masId))
                .where(scpDemandOrderDDO.srcDocId.eq(srcDocId))
                .fetchOne();
    }

    public List<ScpDemandOrderRelateDTO> findRelateOrderByMasId(Long masId) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ScpDemandOrderRelateDTO.class,
                        scpDemandOrderDDO.masId,
                        scpDemandOrderDDO.srcDocId,
                        scpDemandOrderDDO.srcDocNo,
                        scpDemandOrderDDO.srcDocCls,
                        scpDemandOrderDDO.payStatus
                )).from(scpDemandOrderDDO)
                .where(scpDemandOrderDDO.masId.eq(masId));
        return jpaQuery.fetch();
    }

    public List<ScpDemandOrderDDTO> findItemListByMasId(List<Long> masIds) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ScpDemandOrderDDTO.class,
                        scpDemandOrderDDO.masId,
                        scpDemandOrderDDO.itemId,
                        scpDemandOrderDDO.imgUrl
                )).from(scpDemandOrderDDO)
                .where(scpDemandOrderDDO.masId.in(masIds));
        return jpaQuery.fetch();
    }

    public void updatePayStatusBySrcDocId(String srcDocNo, String payStatus) {
        var jpaQuery = jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.payStatus, payStatus);
        jpaQuery.where(scpDemandOrderDDO.srcDocNo.eq(srcDocNo));
        jpaQuery.execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateOnlinePayFlagAndPayerNameBySrcDocNo(String srcDocNo, String payerName, Boolean payStatus) {
        var jpaQuery = jpaQueryFactory.update(scpDemandOrderDDO)
                .set(scpDemandOrderDDO.payerName, payerName)
                .set(scpDemandOrderDDO.onlinePayFlag, payStatus);
        jpaQuery.where(scpDemandOrderDDO.srcDocNo.eq(srcDocNo));
        jpaQuery.execute();
    }
    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ScpDemandOrderDDTO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ScpDemandOrderDDTO.class,
                scpDemandOrderDDO.id,
                scpDemandOrderDDO.masId,
                scpDemandOrderDDO.demandQuantity,
                scpDemandOrderDDO.payStatus,
                scpDemandOrderDDO.itemCode,
                scpDemandOrderDDO.itemId,
                scpDemandOrderDDO.itemName,
                scpDemandOrderDDO.planAmt
        )).from(scpDemandOrderDDO);
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ScpDemandOrderDListParamVO query){
        List<Predicate> list = new ArrayList<>();
        /** 主表idID 精确 */
        if (!ObjectUtils.isEmpty(query.getMasIds())) {
            list.add(scpDemandOrderDDO.masId.in(query.getMasIds()));
        }
        if (!ObjectUtils.isEmpty(query.getPayStatus())) {
            list.add(scpDemandOrderDDO.payStatus.eq(query.getPayStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getPayStatusIn())) {
            list.add(scpDemandOrderDDO.payStatus.in(query.getPayStatusIn()));
        }
        if (!ObjectUtils.isEmpty(query.getItemCode())) {
            list.add(scpDemandOrderDDO.itemCode.eq(query.getItemCode()));
        }
        if (!ObjectUtils.isEmpty(query.getItemCodeNe())) {
            list.add(scpDemandOrderDDO.itemCode.ne(query.getItemCodeNe()));
        }
        return ExpressionUtils.allOf(list);
    }



    public List<ScpDemandOrderDDTO> queryList(ScpDemandOrderDListParamVO scpDemandOrderDParamVO) {
        JPAQuery<ScpDemandOrderDDTO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(scpDemandOrderDParamVO));
        return jpaQuery.fetch();
    }


}
