package com.elitesland.scp.mq;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitesland.scp.boh.StoreReceiveRollbackSendParam;
import com.elitesland.scp.boh.StoreReceiveSendParam;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDO;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 回写订货单关联的门店收货单的审批数量
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class StoreReceiveRollbackListener implements MessageQueueListener<StoreReceiveRollbackSendParam> {

    private final StoreReceiveRepo storeReceiveRepo;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{StoreReceiveRollbackSendParam.ROLLBACK_CHANNEL};
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void onConsume(@NotBlank String s, @NotNull StoreReceiveRollbackSendParam storeReceiveSendParam) {
        log.info("门店收货单撤回动作：" + JSON.toJSONString(storeReceiveSendParam));
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(storeReceiveSendParam.getDocId());
        if (CollectionUtil.isNotEmpty(receiveDOS)) {
            StoreReceiveDO storeReceiveDO = receiveDOS.get(0);
            if (Objects.equals(storeReceiveSendParam.getBackType(), "S")) {
                // 门店收货单出库后做出库作废操作 作废门店收货单
                storeReceiveRepo.deleteById(storeReceiveDO.getId());
            } else if (Objects.equals(storeReceiveSendParam.getBackType(), "O")) {
                // 门店退货单 取消 门店退货改为已取消
                storeReceiveDO.setStatus("CL");
                storeReceiveRepo.save(storeReceiveDO);
            } else {
                // 撤销审批 门店退货单 待退货改为退货待审
                storeReceiveDO.setStatus("DR");
                storeReceiveRepo.save(storeReceiveDO);
            }
        }
        //重置分配数量和发货数量
        Map<Long, BigDecimal> itemMap = storeReceiveSendParam.getItems().stream()
                .collect(Collectors.groupingBy(StoreReceiveRollbackSendParam.OrderItem::getOrderDid,
                        Collectors.mapping(StoreReceiveRollbackSendParam.OrderItem::getQty,
                                Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));

        for (Map.Entry<Long, BigDecimal> item : itemMap.entrySet()) {
            Long did = item.getKey();
            BigDecimal qty = item.getValue();
            scpDemandOrderDDomainService.resetSrcDocQuantity(did, qty);
        }
        scpDemandOrderDomainService.updateDocStatusById(storeReceiveSendParam.getOrderId(), ScpConstant.ZF);
    }

}
