package com.elitesland.scp.utils;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitesland.scp.common.ScpConstant;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Set;

/**
 * @Author: ryan.xu
 * @since 2023/3/21
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MessageDelyQueueService {

    private final RedisUtils redisUtils;

    /**
     * 延迟队列名称，可以根据不通的业务处理设置不同的队列
     */
    private static final String DELY_QUEUE_NAME = "scp_dely_queue";

    /**
     * 发送数据
     *
     * @param message 消息
     * @param dely    延迟多久（毫秒）
     */
    public Boolean pushMessage(String message, long dely) {
        long score = System.currentTimeMillis() + dely;
        return redisUtils.getRedisTemplate().opsForZSet().add(DELY_QUEUE_NAME, message, score);
    }

    /**
     * 拉取最新需要
     * 被消费的消息
     * rangeByScore 根据score范围获取 0-当前时间戳可以拉取当前时间及以前的需要被消费的消息
     *
     * @return
     */
    public String pull() {
        Set<String> strings = redisUtils.getRedisTemplate().opsForZSet().rangeByScore(DELY_QUEUE_NAME, 0, System.currentTimeMillis(), 0, 1);
        if (CollUtil.isNotEmpty(strings)) {
            return strings.iterator().next();
        }
        return null;
    }

    /**
     * 移除消息
     *
     * @param message
     */
    @SneakyThrows
    public Boolean remove(String message) {
        Long remove = redisUtils.getRedisTemplate().opsForZSet().remove(DELY_QUEUE_NAME, message);
        return remove > 0;
    }

    /**
     * 获取购物车倒计时
     *
     * @param message
     * @return
     */
    public String getOrderExpireTime(String message) {
        Double score = redisUtils.getRedisTemplate().opsForZSet().score(DELY_QUEUE_NAME, ScpConstant.SCP_ONLINE_PAY + message);
        if (ObjectUtil.isNotNull(score) && (score - System.currentTimeMillis()) > 0) {
            var time = (score - System.currentTimeMillis()) / 1000D;
            return DateUtil.secondToTime((int) time);
        }
        return "00:00:00";
    }

    public void resettingOrderExpireTime(String orderNo, Long expireTime) {
        String orderKey = ScpConstant.SCP_ONLINE_PAY + orderNo;
        redisUtils.getRedisTemplate().opsForZSet().remove(DELY_QUEUE_NAME, orderKey);

        this.pushMessage(orderKey, expireTime);
    }
}
