package com.elitesland.scp.pay.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.pay.config.EntpayProperties;
import com.elitesland.scp.pay.vo.AccountBindNotifyModel;
import com.elitesland.scp.pay.vo.AccountBindParamVO;
import com.elitesland.scp.pay.vo.BankAccountParamVO;
import com.tenpay.business.entpay.mse.sdk.api.AccountBatchBind;
import com.tenpay.business.entpay.mse.sdk.common.NotifyHandler;
import com.tenpay.business.entpay.mse.sdk.config.EntpayConfig;
import com.tenpay.business.entpay.mse.sdk.exception.EntpayException;
import com.tenpay.business.entpay.mse.sdk.model.AccountBatchBindParam;
import com.tenpay.business.entpay.mse.sdk.model.NotifyUrl;
import com.tenpay.business.entpay.mse.sdk.model.PayeeAccounts;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.configurationprocessor.json.JSONException;
import org.springframework.boot.configurationprocessor.json.JSONObject;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class AccountBatchBindServiceImpl implements AccountBatchBindService {

    private final RedisWrapper redisWrapper;
    private final RedisUtils redisUtils;
    private final NotifyVerifyService notifyVerifyService;

    @Override
    public void accountBind(AccountBindParamVO paramVO) {
        try {
            EntpayProperties entpayProperties = Optional.ofNullable(redisWrapper.apply(() -> redisUtils.get(ScpConstant.ENT_PAY_PROPERTIES), null))
                    .map(Object::toString)
                    .map(o -> JSONUtil.toBean(o, EntpayProperties.class))
                    .orElseThrow(() -> new BusinessException("请联系管理员配置微企付信息"));

            AccountBatchBindParam accountBatchBindParam = new AccountBatchBindParam();
            accountBatchBindParam.setEntId(paramVO.getEntId());
            accountBatchBindParam.setOutRequestNo(paramVO.getOutRequestNo());
            if (CollUtil.isEmpty(paramVO.getBankAccountList())) {
                throw new BusinessException("收款账户信息不能为空");
            }
            List<PayeeAccounts> payeeAccounts = paramVO.getBankAccountList().stream()
                    .map(this::mapToPayeeAccount)
                    .collect(Collectors.toList());
            accountBatchBindParam.setPayeeAccounts(payeeAccounts);
            // 绑定回调url
            NotifyUrl notifyUrl = NotifyUrl.builder()
                    .serverNotifyUrl(entpayProperties.getServerAccountBindNotifyUrl())
                    .webSuccessUrl(entpayProperties.getServerAccountBindNotifyUrl())
                    .build();
            accountBatchBindParam.setNotifyUrl(notifyUrl);
            // 调用批量绑定接口
            log.info("批量绑定账户微企付参数：{}", JSONUtil.toJsonStr(accountBatchBindParam));
            String requestNo = AccountBatchBind.create(accountBatchBindParam).getRequestNo();
            log.info("批量绑定账户微企付请求号：{}", requestNo);
        } catch (EntpayException e) {
            throw new BusinessException("微企付批量绑定账号失败: " + e.getMessage());
        }
    }

    private PayeeAccounts mapToPayeeAccount(BankAccountParamVO row) {
        return PayeeAccounts.builder()
                .accountType(row.getAccountType())
                .bankAccountName(row.getBankAccountName())
                .bankAccountNumber(row.getBankAccountNumber())
                .bankName(row.getBankName())
                .bankBranchId(row.getBankBranchId())
                .bankBranchName(row.getBankBranchName())
                .build();
    }

    @Override
    public String accountBindNotify(String body, String authorization) throws EntpayException, JSONException {
        // 验签并获取model
        AccountBindNotifyModel notifyModel = NotifyHandler.handlerWebhook(body, authorization, AccountBindNotifyModel.class,
                EntpayConfig.getRealTbepPublicKey(null));
        log.info("批量绑定账户回调通知数据：{}", JSONUtil.toJsonStr(notifyModel));
        notifyVerifyService.processAccountBindNotify(notifyModel);
        // 封装响应参数
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("retcode", 0);
        jsonObject.put("retmsg", "SUCCESS");
        return jsonObject.toString();
    }
}
