package com.elitesland.scp.application.service.scpsman;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanImportSaveVO;
import com.elitesland.scp.domain.convert.scpsman.SalesmanInfoConvert;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/3/21 11:24
 * @Description:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class ScpsmanImportServiceImpl implements DataImport<StandardScpsmanImportEntity> {

    private final ScpsmanInfoService scpsmanInfoService;
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_scpsman_import";
    }

    @Override
    public List<String> executeImport(List<StandardScpsmanImportEntity> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg =  checkAndSaveData(dataList, startRowIndex);
            log.info("计划员导入处理完成,错误信息如下:{},错误信息条数{}",JSONUtil.toJsonStr(errorMsg),errorMsg.size());
            return errorMsg;
        } catch (Exception e) {
            for (StandardScpsmanImportEntity entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<StandardScpsmanImportEntity> dataList, int startRowIndex) {
        log.info("startRowIndex的值:{}",startRowIndex);
        List<SalesmanImportSaveVO> recommendRelateCustVOS = SalesmanInfoConvert.INSTANCE.toSalesmanImportSaveVOS(dataList);

        Map<Integer, String> indexScpsmanNoMapping = new HashMap<>();
        for (int i = 0; i < dataList.size(); i++) {
            indexScpsmanNoMapping.put(i + startRowIndex, dataList.get(i).getScpsmanNo());
        }
        log.info("indexScpsmanNoMapping的值:{}", JSONUtil.toJsonStr(indexScpsmanNoMapping));

        List<String> scpsmanNoList =
                dataList.stream().map(StandardScpsmanImportEntity::getScpsmanNo).collect(Collectors.toList());
        Map<String, List<Integer>> indexMap = processDuplicate(scpsmanNoList, startRowIndex);
        log.info("indexMap的值:{}", JSONUtil.toJsonStr(indexMap));
        Map<String, String> errorInfoMap = scpsmanInfoService.salesmanImport(recommendRelateCustVOS).getErrorInfoMap();
        log.info("errorInfoMap的值:{}",JSONUtil.toJsonStr(errorInfoMap));
        return  processErrorList(indexScpsmanNoMapping, indexMap, errorInfoMap, startRowIndex,
                startRowIndex + dataList.size() - 1);
    }

    private Map<String, List<Integer>> processDuplicate(List<String> scpsmanNoList, int startRowIndex) {
        Map<String, List<Integer>> indexMap = new HashMap<>();
        for (int i = 0; i < scpsmanNoList.size(); i++) {
            String element = scpsmanNoList.get(i);
            if (!indexMap.containsKey(element)) {
                indexMap.put(element, new ArrayList<>());
            }
            indexMap.get(element).add(i + startRowIndex);
        }
        return indexMap;
    }

    private List<String> processErrorList(Map<Integer, String> indexScpsmanNoMapping, Map<String, List<Integer>> indexMap,
                                          Map<String, String> errorInfoMap, int start, int end) {

        log.info("errorInfoMap的值:{}",JSONUtil.toJsonStr(errorInfoMap));
        List<String> errorList = new ArrayList<>();

        for (int i = start; i <= end; i++) {
            String scpsmanNo = indexScpsmanNoMapping.get(i);
            if (indexMap.get(scpsmanNo).size() > 1) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i, "计划员编号", "编号重复"));
            } else if (errorInfoMap.containsKey(scpsmanNo) && StrUtil.isNotBlank(errorInfoMap.get(scpsmanNo))) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i, "",
                        errorInfoMap.get(scpsmanNo)));
            } else {
                //重要，没有错误信息也需要返回，返回结果需和数据条数对齐
                errorList.add(null);
            }
        }



        if (CollUtil.isEmpty(errorList)) {
            return new ArrayList<>();
        } else {
            // 如果该条数据存在错误 添加错误信息
            return errorList;
        }
    }
}
