package com.elitesland.scp.application.service.setting;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.setting.*;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.setting.ScpOrderSettingSaveVO;
import com.elitesland.scp.domain.convert.setting.ScpOrderSettingConvert;
import com.elitesland.scp.domain.service.setting.ScpOrderSettingDomainService;
import com.elitesland.scp.infr.repo.item.ScpCateItemRepoProc;
import com.elitesland.scp.rmi.RmiItemService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpOrderSettingServiceImpl implements ScpOrderSettingService {

    private final ScpOrderSettingDomainService scpOrderSettingDomainService;
    private final ScpCateItemRepoProc scpCateItemRepoProc;
    private final RmiItemService rmiItemService;

    @Override
    public PagingVO<ScpOrderSettingPageRespVO> page(ScpOrderSettingPageParamVO paramVO) {
        return scpOrderSettingDomainService.queryOrderSettingList(paramVO);
    }

    @Override
    public Optional<ScpOrderSettingRespVO> findOrderSettingById(Long id) {
        return scpOrderSettingDomainService.findOrderSettingById(id).map(ScpOrderSettingConvert.INSTANCE::dtoToRespVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrderSetting(ScpOrderSettingSaveVO saveVO) {
        // 校验单据类型配置
        checkSetting(saveVO);
        return scpOrderSettingDomainService.saveOrderSetting(saveVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpOrderSettingDomainService.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeOrderSettingStatus(ScpOrderSettingStatusParamVO paramVO) {
        scpOrderSettingDomainService.enableOrderSetting(paramVO.getIds(), paramVO.getStatus());
    }

    @Override
    @SysCodeProc
    public List<ScpOrderSettingRespVO> queryOrderSettingList(ScpOrderSettingParamVO paramVO) {
        List<ScpOrderSettingRespVO> settingList = scpOrderSettingDomainService.findByParam(paramVO);
        if (CollUtil.isEmpty(settingList)) {
            return CollUtil.newArrayList();
        }
        settingList.forEach(setting -> setting.setItemCates(StrUtil.split(setting.getItemCate(), ',')));
        return settingList;
    }

    @Override
    public String getFirstItemCate(ScpOrderSettingBaseParamVO paramVO) {
        ScpOrderSettingRespVO cacheByDocType = scpOrderSettingDomainService.findCacheByDocType(paramVO.getDocType(), "0");
        List<String> itemTypes = StrUtil.split(cacheByDocType.getItemCate(), ",");
        List<String> cateCodes = scpCateItemRepoProc.getFirstCateByType(itemTypes, paramVO.getBrands());
        return rmiItemService.getFirstCateByCodes(cateCodes);
    }

    private void checkSetting(ScpOrderSettingSaveVO saveVO) {
        if (saveVO.getFirstPriority().equals(saveVO.getSecPriority())) {
            throw new BusinessException("第一优先级和第二优先级不能相同");
        }
        ScpOrderSettingParamVO paramVO = new ScpOrderSettingParamVO();
        List<ScpOrderSettingRespVO> settings = scpOrderSettingDomainService.findByParam(paramVO);
        if (CollUtil.isEmpty(settings)) {
            return;
        }
        Map<String, List<ScpOrderSettingRespVO>> typeMap = settings.stream().collect(Collectors.groupingBy(row -> row.getDocType() + "@" + row.getType()));
        Map<String, Long> itemCateMap = settings.stream().flatMap(row -> StrUtil.split(row.getItemCate(), ",")
                .stream().map(cate -> {
                    ScpOrderSettingBaseVO result = new ScpOrderSettingBaseVO();
                    result.setId(row.getId());
                    result.setItemCate(cate);
                    result.setType(row.getType());
                    return result;
                })).collect(Collectors.toMap(row -> row.getItemCate() + "@" + row.getType(), ScpOrderSettingBaseVO::getId));
        if (saveVO.getId() == null) {
            if (typeMap.get(saveVO.getDocType() + "@" + saveVO.getType()) != null) {
                throw new BusinessException("存在单据类型和要货类型,不能重复添加");
            }
            for (String itemCate : saveVO.getItemCates()) {
                if (itemCateMap.containsKey(itemCate + "@" + saveVO.getType())) {
                    throw new BusinessException("商品分类已存在,不能重复添加");
                }
            }
        } else {
            if (CollUtil.isEmpty(settings)) {
                return;
            }
            List<ScpOrderSettingRespVO> settingRespVOS = typeMap.get(saveVO.getDocType() + "@" + saveVO.getType());
            if (CollUtil.isEmpty(settingRespVOS)) {
                return;
            }
            List<Long> ids = settingRespVOS.stream().map(ScpOrderSettingRespVO::getId).collect(Collectors.toList());
            if (!ids.contains(saveVO.getId())) {
                throw new BusinessException("存在单据类型和要货类型,不能重复添加");
            }
            for (String itemCate : saveVO.getItemCates()) {
                String cateType = itemCate + "@" + saveVO.getType();
                if (itemCateMap.containsKey(cateType) && !itemCateMap.get(cateType).equals(saveVO.getId())) {
                    throw new BusinessException("商品分类已存在,不能重复添加");
                }
            }

        }
    }
}
