package com.elitesland.scp.application.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.TimeInterval;
import cn.hutool.core.util.PageUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.inv.dto.invwh.InvWhRpcSimpleDTO;
import com.elitesland.scp.application.facade.vo.param.order.*;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetSelectRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandSetSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandUrgentSetSaveVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandSetConvert;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandSetDomainService;
import com.elitesland.scp.domain.service.setting.ScpOrderSettingDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandSetDTO;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.rmi.RmiSysUserRpcService;
import com.elitesland.scp.utils.TransactionCommitHandler;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreWhDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandOrderSetServiceImpl implements ScpDemandOrderSetService {

    private final RedisTemplate redisTemplate;
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private final TransactionCommitHandler transactionCommitHandler;
    private final ScpDemandOrderService scpDemandOrderService;
    private final MessageQueueTemplate messageQueueTemplate;
    private final ScpOrderSettingDomainService scpOrderSettingDomainService;
    private final ScpDemandSetDomainService scpDemandSetDomainService;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final TaskExecutor taskExecutor;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RmiSysUDCService rmiSysUDCService;

    @Override
    public PagingVO<ScpDemandSetPageRespVO> queryDemandSetList(ScpDemandSetPageParamVO paramVO) {
        return scpDemandSetDomainService.queryDemandSetList(paramVO);
    }

    @Override
    public Optional<ScpDemandSetRespVO> findDemandSetById(Long id) {
        Map<String, String> createTypeMap = rmiSysUDCService.getCodeMap("yst-scp", UdcEnum.SCP_ORDER_SET_CREATE_TYPE_01.getCode());
        return scpDemandSetDomainService.findDemandSetById(id).map(row -> {
            ScpDemandSetRespVO scpDemandSetRespVO = ScpDemandSetConvert.INSTANCE.dtoToRespVO(row);
            if (row.getExpireTime() != null) {
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
                scpDemandSetRespVO.setExpireTimeStr(row.getExpireTime().toLocalTime().format(formatter));
                scpDemandSetRespVO.setCreateTypeName(createTypeMap.get(row.getCreateType()));
            }
            return scpDemandSetRespVO;
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveDemandSet(ScpDemandSetSaveVO saveVO) {
        // 校验订货集保存参数
        this.checkDemandSetParam(saveVO);
        if (saveVO.getId() == null) {
            saveVO.setDemandCode(orderSetNo(saveVO.getType()));
        }
        if (StrUtil.isNotBlank(saveVO.getExpireTimeStr())) {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
            LocalDateTime expireTime = LocalDateTime.of(saveVO.getDemandDate().toLocalDate(), LocalTime.parse(saveVO.getExpireTimeStr(), formatter));
            saveVO.setExpireTime(expireTime);
        }
        return scpDemandSetDomainService.saveDemandSet(saveVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveUrgentDemandSet(ScpDemandUrgentSetSaveVO saveVO) {
        // 校验订货集保存参数
        saveVO.setType(ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode());
        saveVO.setDemandDate(LocalDateTime.now());
        saveVO.setStatus(false);
        if (saveVO.getId() == null) {
            saveVO.setDemandCode(orderSetNo(saveVO.getType()));
        }
        saveVO.setDemandName(ScpConstant.MDDHJ + convertDate(LocalDate.now().atStartOfDay()) + saveVO.getDemandCode().substring(saveVO.getDemandCode().length() - 3));
        if (StrUtil.isNotBlank(saveVO.getExpireTimeStr())) {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
            LocalDateTime expireTime = LocalDateTime.of(saveVO.getDemandDate().toLocalDate(), LocalTime.parse(saveVO.getExpireTimeStr(), formatter));
            saveVO.setExpireTime(expireTime);
        }
        ScpDemandSetSaveVO scpDemandSetSaveVO = ScpDemandSetConvert.INSTANCE.saveVoToUrgentSaveVO(saveVO);
        scpDemandSetSaveVO.setCreateType(UdcEnum.SCP_ORDER_SET_CREATE_TYPE_03.getValueCode());
        return scpDemandSetDomainService.saveDemandSet(scpDemandSetSaveVO);
    }

    /**
     * 格式化日期
     *
     * @return
     */
    private String convertDate(LocalDateTime demandDate) {
        // 定义日期格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy年M月d号");
        // 格式化日期
        return demandDate.format(formatter);
    }



    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeDemandSetStatus(ScpDemandSetStatusParamVO paramVO) {
        if (CollUtil.isEmpty(paramVO.getIds())) {
            throw new BusinessException("订货集id不能为空");
        }
        scpDemandSetDomainService.enableDemandSet(paramVO.getIds(), paramVO.getStatus());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        List<ScpDemandSetDTO> demandSetDTOList = scpDemandSetDomainService.findDemandSetByIds(ids);
        if (CollUtil.isEmpty(demandSetDTOList)) {
            throw new BusinessException("订货集不存在");
        }
        //校验订货集是否存在订货日期
        Map<LocalDateTime, List<ScpDemandSetDTO>> demandDateMap =
                demandSetDTOList.stream().collect(Collectors.groupingBy(ScpDemandSetDTO::getDemandDate));
        for (Map.Entry<LocalDateTime, List<ScpDemandSetDTO>> entry : demandDateMap.entrySet()) {
            ScpDemandSetParamVO paramVO = ScpDemandSetParamVO.builder().demandDate(entry.getKey()).build();
            List<ScpDemandSetDTO> scpDemandSetDTOList = scpDemandSetDomainService.findDemandSetByParam(paramVO);
            if (scpDemandSetDTOList.size() <= entry.getValue().size()) {
                throw new BusinessException("需求日期" + entry.getKey() + "必须有一个订货集，不能删除");
            }
        }
        var demandOrderRespVOS = scpDemandOrderService.findDemandOrderByDemandIds(ids);
        if (CollUtil.isNotEmpty(demandOrderRespVOS)) {
            //如果存在订货订单引用
            throw new BusinessException("订货集:" + demandOrderRespVOS.get(0).getDemandName() + "已被订货订单引用，不能删除");
        }
        demandSetDTOList.forEach(demandSetDTO -> {
            if (demandSetDTO.getStatus()) {
                throw new BusinessException("订货集" + demandSetDTO.getDemandCode() + "已启用，不能删除");
            }
        });
        //删除订货集
        scpDemandSetDomainService.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void compute(List<Long> ids) {
        List<ScpDemandSetDTO> demandSetDTOS = scpDemandSetDomainService.findDemandSetByIds(ids);
        demandSetDTOS.stream().filter(row -> !row.getStatus()).forEach(row -> {
            throw new BusinessException("订货集" + row.getDemandCode() + "未启用");
        });
        //更新订货集计算信息
        scpDemandSetDomainService.updateDemandSetMsgByIds(ids, ScpConstant.COMPUTE_DOING);
        // 事务提交之后执行
        transactionCommitHandler.handle(() -> {
            CompletableFuture.supplyAsync(() -> {
                ids.forEach(row -> scpDemandOrderService.compute(row));
                return 1;
            }, taskExecutor);
        });
    }

    @Override
    public List<ScpDemandSetSelectRespVO> demandSetSelectList(ScpDemandSetParamVO paramVO) {
        //查询启用状态的订货集
        paramVO.setExpireTime(LocalDateTime.now());
        return scpDemandSetDomainService.findDemandSetByParam(paramVO).stream().map(ScpDemandSetConvert.INSTANCE::dtoToSelectRespVO)
                .collect(Collectors.toList());
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandSetPageRespVO> commonPage(ScpDemandSetPageParamVO paramVO) {
        return scpDemandSetDomainService.commonPage(paramVO);
    }

    @Override
    @SysCodeProc
    public List<ScpDemandSetPageRespVO> commonNewPage(ScpDemandSetPageParamVO paramVO) {
        //查询启用状态的订货集
        paramVO.setExpireTime(LocalDateTime.now());
        OrgStoreDetailRpcDTO byCode = rmiOrgStoreRpcService.getByCode(paramVO.getStoreCode());
        if(byCode != null){
            paramVO.setRegionCode(byCode.getRegion());
        }
        return scpDemandSetDomainService.commonNewPage(paramVO);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void push(List<Long> ids) {
        List<ScpDemandSetDTO> demandSetDTOS = scpDemandSetDomainService.findDemandSetByIds(ids);
        demandSetDTOS.stream().filter(row -> !row.getStatus()).forEach(row -> {
            throw new BusinessException("订货集" + row.getDemandCode() + "未启用");
        });
        scpDemandSetDomainService.updateDemandSetPushMsgById(ids, ScpConstant.PUSH_DOING);
        // 事务提交之后执行
        transactionCommitHandler.handle(() -> {
            CompletableFuture.supplyAsync(() -> {
                ids.forEach(row -> this.pushToOtherDomain(row));
                return 1;
            }, taskExecutor);
        });
    }

    /**
     * 推送订单订单到其他业务域
     *
     * @param demandSetId 订货集ID
     */
    private void pushToOtherDomain(Long demandSetId) {
        TimeInterval timer = new TimeInterval();
        log.info("【订货集推送其他业务域】订货集ID：{},开始推送", demandSetId);
        String redisKey = "PUSH_TO_OTHER@".concat(demandSetId.toString());
        if (redisTemplate.hasKey(redisKey)) {
            log.info("【pushToOtherDomain】订货集ID:{},正在推送中...");
            return;
        }
        // 设置防重复操作限时标记(前置通知)
        redisTemplate.opsForValue().set(redisKey, "PUSH_TO_OTHER", 300, TimeUnit.SECONDS);
        try {
            Long count = scpDemandOrderDomainService.countUnPushedOrderByDemandId(demandSetId);
            if (count <= 0) {
                log.info("【订货集推送其他业务域】订货集ID：{}，无需推送", demandSetId);
                return;
            }
            //获取当前操作人id和name
            CurrentUserDTO currentUserDTO = UserService.currentUser();
            log.info("当前登录用户信息:{}", JSONUtil.toJsonStr(currentUserDTO));
            if (ObjectUtils.isEmpty(currentUserDTO) || currentUserDTO.getDetail() == null) {
                throw new BusinessException("查询当前登录用户失败");
            }

            Integer totalPage = PageUtil.totalPage(count, 500);
            for (int i = 1; i <= totalPage; i++) {
                ScpComputeDemandOrderPageParamVO pageParamVO = new ScpComputeDemandOrderPageParamVO();
                pageParamVO.setDemandId(demandSetId);
                pageParamVO.setSize(500);
                pageParamVO.setCurrent(i);
                List<OrderItem> orders = new ArrayList<>();
                OrderItem orderItem = new OrderItem();
                orderItem.setAsc(false);
                orderItem.setColumn("storeLevel");
                orders.add(orderItem);
                pageParamVO.setOrders(orders);
                List<Long> ids = scpDemandOrderDomainService.pageDemandOrderIds(pageParamVO);
                List<ScpDemandOrderDTO> demandOrderDTOS = scpDemandOrderDomainService.findDemandOrderByIds(ids);
                Map<Long, ScpDemandOrderDTO> orderMap = demandOrderDTOS.stream().collect(Collectors.toMap(ScpDemandOrderDTO::getId, Function.identity()));
                List<ScpDemandOrderDDTO> demandOrderDList = scpDemandOrderDDomainService.findDemandOrderDByMasIds(ids);
                Map<Long, List<ScpDemandOrderDDTO>> orderDMap = demandOrderDList.stream().collect(Collectors.groupingBy(ScpDemandOrderDDTO::getMasId));
                Map<String, List<OrgStoreWhDTO>> orgStoreMap = scpDemandOrderService.getOrgStoreMap(demandOrderDTOS);
                Map<Long, InvWhRpcSimpleDTO> whMap = scpDemandOrderService.getWhMap(demandOrderDTOS);

                for (Long id : ids) {
                    ScpDemandOrderDTO scpDemandOrderDTO = orderMap.get(id);
                    //获取单据类型配置信息
                    ScpOrderSettingRespVO setting = scpOrderSettingDomainService.findCacheByDocType(scpDemandOrderDTO.getDocType(), scpDemandOrderDTO.getType());
                    //准备调拨单
                    var storeOrderTrnRpcDTOS = scpDemandOrderService.prepareDataTrnPush(scpDemandOrderDTO, orgStoreMap, whMap, currentUserDTO.getDetail().getEmployeeId(), orderDMap.get(id), setting);
                    //准备采购订单
                    var purPoSaveDTOS = scpDemandOrderService.prepareDataPoPush(scpDemandOrderDTO, orgStoreMap, currentUserDTO.getDetail().getEmployeeId(), orderDMap.get(id), setting);
                    //准备销售订单
                    var salSoSaveDTOS = scpDemandOrderService.prepareDataSoPush(scpDemandOrderDTO, orgStoreMap, orderDMap.get(id));

                    //推送消息队列
                    if (CollUtil.isNotEmpty(storeOrderTrnRpcDTOS)) {
                        //写入mq队列
                        ScpTrnOrderCreateMqParam orderCreateMqParam = new ScpTrnOrderCreateMqParam();
                        orderCreateMqParam.setTrnList(storeOrderTrnRpcDTOS);
                        orderCreateMqParam.setBusinessKey(ScpTrnOrderCreateMqParam.CREATE_CHANNEL);
                        messageQueueTemplate.publishMessage("yst-suplan", ScpTrnOrderCreateMqParam.CREATE_CHANNEL, orderCreateMqParam);
                        log.info("【订货集推送其他业务域】订货单ID：{},调拨单推送成功", id);
                    }
                    if (CollUtil.isNotEmpty(purPoSaveDTOS)) {
                        //写入mq队列
                        ScpPoOrderCreateMqParam poCreateMqParam = new ScpPoOrderCreateMqParam();
                        poCreateMqParam.setPoList(purPoSaveDTOS);
                        poCreateMqParam.setBusinessKey(ScpPoOrderCreateMqParam.CREATE_CHANNEL);
                        messageQueueTemplate.publishMessage("yst-suplan", ScpPoOrderCreateMqParam.CREATE_CHANNEL, poCreateMqParam);
                        log.info("【订货集推送其他业务域】订货单ID：{},采购订单推送成功", id);
                    }

                    if (CollUtil.isNotEmpty(salSoSaveDTOS)) {
                        //写入mq队列
                        ScpSaleOrderCreateMqParam saleCreateMqParam = new ScpSaleOrderCreateMqParam();
                        saleCreateMqParam.setSaleList(salSoSaveDTOS);
                        saleCreateMqParam.setBusinessKey(ScpPoOrderCreateMqParam.CREATE_CHANNEL);
                        messageQueueTemplate.publishMessage("yst-suplan", ScpSaleOrderCreateMqParam.CREATE_CHANNEL, saleCreateMqParam);
                        log.info("【订货集推送其他业务域】订货单ID：{},销售订单推送成功", id);
                    }
                }
            }
            //更新订货集推送状态为已完成
            scpDemandSetDomainService.completePushById(demandSetId);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        } finally {
            redisTemplate.delete(redisKey);
        }
        log.info("【订货集推送其他业务域】订货集ID：{},耗时：{}", demandSetId, timer.intervalMs());
    }

    /**
     * 校验订货集保存参数
     *
     * @param saveVO
     */
    private void checkDemandSetParam(ScpDemandSetSaveVO saveVO) {
        ScpDemandSetParamVO paramVO = ScpDemandSetParamVO.builder().demandName(saveVO.getDemandName()).build();
        boolean exist =
                scpDemandSetDomainService.findDemandSetByParam(paramVO).stream().anyMatch(demandSetDTO -> !demandSetDTO.getId().equals(saveVO.getId()));
        if (exist) {
            throw new BusinessException("订货集名称已存在");
        }
        if (saveVO.getId() == null) {
            return;
        }
        var demandOrderRespVO = scpDemandOrderService.findDemandOrderByDemandIds(List.of(saveVO.getId()));
        if (CollUtil.isNotEmpty(demandOrderRespVO)) {
            //如果存在订货订单引用
            throw new BusinessException("订货集:" + saveVO.getDemandName() + "已被订货订单引用，不能编辑");
        }
    }

    private String orderSetNo(String type) {
        if (ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(type)) {
            return rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.BKST, new ArrayList<>());
        } else {
            return rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.BKWH, new ArrayList<>());
        }
    }
}
