package com.elitesland.scp.pay.controller;


import cn.hutool.extra.servlet.ServletUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitesland.scp.pay.service.PaymentService;
import com.elitesland.scp.pay.vo.OnlinePayOrderParamVO;
import com.tenpay.business.entpay.mse.sdk.api.Payment;
import com.tenpay.business.entpay.mse.sdk.api.Redirect;
import com.tenpay.business.entpay.mse.sdk.api.Withdraw;
import com.tenpay.business.entpay.mse.sdk.exception.EntpayException;
import com.tenpay.business.entpay.mse.sdk.model.PaymentClose;
import com.tenpay.business.entpay.mse.sdk.model.PaymentCloseParam;
import com.tenpay.business.entpay.mse.sdk.model.RetrieveBalanceGetParam;
import com.tenpay.business.entpay.mse.sdk.model.WithdrawAccountTypeConstant;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;


@Slf4j
@RestController
@RequestMapping("/payments")
@RequiredArgsConstructor
@Validated
public class PaymentController {

    private final PaymentService paymentService;

    @ApiOperation(value = "小程序支付下单")
    @PostMapping("/online/mp-pay")
    @BusinessObjectOperation(operationType = OperationTypeEnum.ELSE, operationDescription = "小程序支付下单", allowRepeatRequest = false)
    public Redirect createOnlineMPPay(HttpServletRequest request, @RequestBody @Validated OnlinePayOrderParamVO param) {
        log.info("小程序支付下单请求参数：{}", JSONUtil.toJsonStr(param));
        String clientIP = ServletUtil.getClientIP(request);
        param.setClientIp(clientIP);
        return paymentService.createMpPayOnline(param);
    }

    @ApiOperation(value = "二维码支付下单")
    @PostMapping(value = "/online/qrcode_pay")
    @BusinessObjectOperation(operationType = OperationTypeEnum.ELSE, operationDescription = "二维码支付下单", allowRepeatRequest = false)
    public Redirect createOnlineQRPay(HttpServletRequest request, @RequestBody @Validated OnlinePayOrderParamVO param) {
        log.info("二维码支付下单请求参数：{}", param);
        // 封装二维码支付接口入参
        String clientIP = ServletUtil.getClientIP(request);
        param.setClientIp(clientIP);
        return paymentService.createQrCodePayOnline(param);
    }

    // 查询订单明细(内单号)
    @GetMapping(value = "/{paymentId}")
    public Payment innerQuery(@PathVariable("paymentId") String paymentId) throws EntpayException {
        log.info("查询订单明细(内单号)请求参数：{}", paymentId);
        return Payment.retrieve(paymentId);
    }

    // 查询订单明细(外单号)
    @GetMapping(value = "/out-payment-id/{outPaymentId}")
    public Payment externalQuery(@PathVariable("outPaymentId") String outPaymentId) throws EntpayException {
        log.info("查询订单明细(外单号)请求参数：{}", outPaymentId);
        return Payment.retrieveByOutPaymentId(outPaymentId);
    }

    // 支付关单
    @GetMapping(value = "/{paymentId}/close")
    public PaymentClose close(@PathVariable("paymentId") String paymentId) throws EntpayException {
        PaymentCloseParam paymentCloseParam = PaymentCloseParam.builder().closeReason("关单原因").build();
        return Payment.close(paymentId, paymentCloseParam);
    }

    // 查询订单手续费
    @GetMapping(value = "/fee/{paymentId}")
    public Payment retrieveFee(@PathVariable("paymentId") String paymentId) throws EntpayException {
        log.info("查询订单手续费请求参数：{}", paymentId);
        return Payment.retrieveFee(paymentId);
    }

    // 余额查询
    @GetMapping(value = "/withdraws/amounts")
    public Withdraw retrieveBalance() throws EntpayException {
        RetrieveBalanceGetParam param = RetrieveBalanceGetParam.builder()
                .accountType(WithdrawAccountTypeConstant.RECEIVER_ACCOUNT)
                .entId("100403095693")
                .entAcctId("61141004030956933600000020009131")
                .build();
        return Withdraw.retrieveBalance(param);
    }
}
