package com.elitesland.scp.rmi;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.provider.org.EmployeeRpcService;
import com.elitesland.sale.api.vo.param.crm.AllowShipCustGroupDParam;
import com.elitesland.sale.api.vo.param.crm.AllowShipCustGroupParam;
import com.elitesland.sale.api.vo.param.salesman.SalesmanInfoSimpleQueryVO;
import com.elitesland.sale.api.vo.resp.crm.*;
import com.elitesland.sale.api.vo.resp.salesman.SalesmanInfoBaseRespVO;
import com.elitesland.sale.constant.AllowShipStatus;
import com.elitesland.sale.dto.CrmCustDTO;
import com.elitesland.sale.dto.CrmCustRespDTO;
import com.elitesland.sale.dto.SalesmanInfoDTO;
import com.elitesland.sale.dto.param.AllowShipRpcParam;
import com.elitesland.sale.dto.param.CrmCustRpcDtoParam;
import com.elitesland.sale.dto.param.CustBaseRpcParam;
import com.elitesland.sale.dto.query.SalesmanQueryDTO;
import com.elitesland.sale.service.AllowShipRpcService;
import com.elitesland.sale.service.CrmCustRpcService;
import com.elitesland.sale.service.MktDiscountGiftRpcService;
import com.elitesland.sale.service.SalesmanRpcService;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 销售支持中心接口调用
 */
@Slf4j
@Component
public class RmiSalService {

    @Autowired
    private CrmCustRpcService crmCustRpcService;
    @Autowired
    private SalesmanRpcService salesmanRpcService;
    @Autowired
    private AllowShipRpcService allowShipRpcService;
    @Autowired
    private MktDiscountGiftRpcService mktDiscountGiftRpcService;
    @Autowired
    private EmployeeRpcService employeeRpcService;

    public ApiResult<CrmCustDTO> getCustInfo(String custCode) {
        log.info("查询客户信息入参：" + custCode);
        try {
            return crmCustRpcService.getCustInfo(custCode);
        } catch (Exception e) {
            log.error("调用销售支持中心，查询客户信息异常", e);
            throw new BusinessException("调用销售支持中心，查询客户信息异常" + e);
        }

    }

    public List<LmSaveCustRespVO> findInvCust(List<String> custCodeList) {
        ApiResult<List<LmSaveCustRespVO>> apiResult = crmCustRpcService.findInvCust(custCodeList);
        Assert.notNull(apiResult, "调用2B销售域查询主客户信息为空,客户编码{}", custCodeList);
        Assert.isTrue(apiResult.isSuccess(), "调用2B销售域查询主客户信息失败，客户编码{}", custCodeList);
        return apiResult.getData();
    }

    /**
     * 客户名称查询
     *
     * @param custId
     * @return
     */
    public ApiResult<String> getCustName(@NonNull Long custId) {
        try {
            log.info("查询客户名称{}", custId);
            return crmCustRpcService.getCustName(custId);
        } catch (Exception e) {
            log.error("客户名称查询异常:{}", e.getMessage());
            throw new BusinessException("客户名称查询接口异常", e);
        }
    }

    /**
     * 查询客户数据信息
     *
     * @param custIds
     * @return
     */
    public ApiResult<List<CrmCustDTO>> listCustById(@NonNull List<Long> custIds) {
        try {
            ApiResult<List<CrmCustDTO>> listApiResult = crmCustRpcService.listCustById(custIds);
            return listApiResult;
        } catch (Exception e) {
            log.error("查询客户数据信息异常:{}", e);
            throw new BusinessException("查询客户数据信息接口异常", e);
        }
    }

    /**
     * 根据客户查询参数获取客户信用检查类型、信用额度等基本信息
     *
     * @param crmCustRpcDtoParam 查询参数
     * @return 客户基本信息
     */
    public ApiResult<List<CrmCustRespDTO>> getCustByParam(@NonNull CrmCustRpcDtoParam crmCustRpcDtoParam) {
        log.info("查询客户信息数据入参{}", JSON.toJSONString(crmCustRpcDtoParam));
        ApiResult<List<CrmCustRespDTO>> result = crmCustRpcService.getCustByParam(crmCustRpcDtoParam);
        log.info("查询客户信息数据返回结果{}", JSON.toJSONString(result));
        return result;
    }

    /**
     * 根据编号或名称混合查询
     *
     * @param custs 编号或名称
     * @return
     */
    public ApiResult<List<CrmCustRespDTO>> getCustByCodeOrNames(List<String> custs) {
        CrmCustRpcDtoParam crmCustRpcDtoParam = new CrmCustRpcDtoParam();
        crmCustRpcDtoParam.setCust(custs);
        log.info("混合查询客户信息数据入参{}", JSON.toJSONString(crmCustRpcDtoParam));
        ApiResult<List<CrmCustRespDTO>> result = crmCustRpcService.getCustByParam(crmCustRpcDtoParam);
        log.info("混合查询客户信息数据返回结果{}", JSON.toJSONString(result));
        if (!result.isSuccess()) {
            throw new BusinessException(result.getErrorMsg());
        }
        return result;
    }

    /**
     * 更新发货状态
     *
     * @param orderCodes 参数
     * @return
     */
    public ApiResult<Object> updateOrderIsSend(@NonNull List<String> orderCodes) {
        // TODO sale域代码，暂时设置空实现
        return null;
    }

    /**
     * 发货发送消息
     *
     * @param bipShipMentOrderCodes 参数
     * @return
     */
    public ApiResult<Object> shipMentMessagePush(@NonNull List<String> bipShipMentOrderCodes) {
        // TODO sale域代码，暂时设置空实现
        return null;
    }

    /**
     * 根据多个业务员id 或者 多个业务员编码查询业务员信息
     *
     * @param salesmanIds
     * @return
     */
    public List<SalesmanInfoDTO> querySalesmanInfoList(List<Long> salesmanIds) {
        log.info("调用销售域-根据多个业务员id 或者 多个业务员编码查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salesmanIds);
        if (CollUtil.isEmpty(salesmanIds)) {
            return new ArrayList<>();
        }
        try {
            SalesmanQueryDTO param = new SalesmanQueryDTO();
            param.setSalesmanIdList(salesmanIds);
            ApiResult<List<SalesmanInfoDTO>> sysUser = salesmanRpcService.querySalesmanInfoByIds(param);
            if (sysUser != null && sysUser.isSuccess()) {
                return sysUser.getData();
            }
        } catch (Exception e) {
            log.error("查询业务员信息错误:", e);

        }
        return null;
    }

    public Map<Long, SalesmanInfoDTO> querySalesmanInfoMap(List<Long> salesmanIds) {
        log.info("调用销售域-根据多个业务员id 或者 多个业务员编码查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salesmanIds);
        if (CollUtil.isEmpty(salesmanIds)) {
            return new HashMap<>();
        }
        SalesmanQueryDTO param = new SalesmanQueryDTO();
        param.setSalesmanIdList(salesmanIds);
        ApiResult<List<SalesmanInfoDTO>> salesmanInfo = salesmanRpcService.querySalesmanInfoByIds(param);
        if (salesmanInfo != null && salesmanInfo.isSuccess()) {
            return salesmanInfo.getData().stream().collect(Collectors.toMap(SalesmanInfoDTO::getId, Function.identity()));
        }
        return new HashMap<>();
    }

    /**
     * 根据多个业务员id 或者 多个业务员编码查询业务员信息
     *
     * @param salesmanCodes
     * @return
     */
    public List<SalesmanInfoDTO> querySalesmanInfoListByCodes(List<String> salesmanCodes) {
        log.info("调用销售域-根据多个业务员编码 或者 多个业务员编码查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salesmanCodes);
        if (CollUtil.isEmpty(salesmanCodes)) {
            return new ArrayList<>();
        }
        SalesmanQueryDTO param = new SalesmanQueryDTO();
        param.setSalesmanCodeList(salesmanCodes);
        ApiResult<List<SalesmanInfoDTO>> sysUser = salesmanRpcService.querySalesmanInfoList(param);
        if (!sysUser.isSuccess() || null == sysUser.getData()) {
            throw new BusinessException("销售域服务异常:" + sysUser.getMsg());
        }
        return sysUser.getData();
    }

    @Nullable
    private List<SalesmanInfoDTO> getSalesmanInfoDTOS(EmployeeQueryDTO param) {
        ApiResult<List<SysEmployeeBasicDTO>> listApiResult = employeeRpcService.queryList(param);
        log.info("调用销售域-根据多个员工编码 或者 多个员工编码查询员工信息,时间：{}，入参：{}，返回结果：{}", LocalDateTime.now(), param, JSON.toJSONString(listApiResult));
        if (listApiResult.isSuccess() && CollUtil.isNotEmpty(listApiResult.getData())) {
            var data = listApiResult.getData();
            return data.stream().map(row -> {
                SalesmanInfoDTO result = new SalesmanInfoDTO();
                result.setId(row.getId());
                result.setSalesmanNo(row.getCode());
                result.setFullName(row.getFullName());
                return result;
            }).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    /**
     * 根据编码或者名称任意条件查询业务员信息
     *
     * @param salemans 编号或名称
     * @return
     */
    public List<SalesmanInfoDTO> querySalesmanInfoListByCodeOrNames(List<String> salemans) {
        log.info("调用销售域-根据多个业务员编码 或者 多个业务员姓名查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salemans);
        if (CollUtil.isEmpty(salemans)) {
            return new ArrayList<>();
        }
        SalesmanQueryDTO param = new SalesmanQueryDTO();
        param.setSalemans(salemans);
        ApiResult<List<SalesmanInfoDTO>> sysUser = salesmanRpcService.querySalesmanInfoList(param);
        if (!sysUser.isSuccess()) {
            throw new BusinessException(sysUser.getErrorMsg());
        }
        if (sysUser != null && sysUser.isSuccess()) {
            log.info("业务员查询结果:{}", JSONUtil.toJsonStr(sysUser.getData()));
            return sysUser.getData();
        }
        return new ArrayList<>();
    }


    /**
     * 获取客户允发期
     *
     * @param custCode
     * @param itemCodes
     * @return
     */
    public List<AllowShipRuleCustItemVO> getAllowShipSale(String custCode, List<String> itemCodes) {
        log.info("调用销售域-获取客户允发期信息,时间：{}，入参：{}", LocalDateTime.now(), custCode, itemCodes);
        if (CollUtil.isEmpty(itemCodes)) {
            throw new BusinessException("调用销售域-获取客户允发期信息，商品编码为空");
        }
        try {
            List<AllowShipRpcParam> paramList = itemCodes.stream().map(row -> {
                AllowShipRpcParam allowShipRpcParam = new AllowShipRpcParam();
                allowShipRpcParam.setCustCode(custCode);
                allowShipRpcParam.setItemCode(row);
                return allowShipRpcParam;
            }).collect(Collectors.toList());
            return allowShipRpcService.getAllowShipSale(paramList);
        } catch (Exception e) {
            log.error("获取客户允发期信息:", e);
            throw new BusinessException("销售域服务异常", e);
        }
    }


    /**
     * 获取客户组信息
     *
     * @param groupCodes
     * @return
     */
    public List<AllowShipCustGroupRespVO> getCustGroup(List<String> groupCodes) {
        log.info("调用销售域-获取客户组信息,时间：{}，入参：{}", LocalDateTime.now(), groupCodes);
        if (CollUtil.isEmpty(groupCodes)) {
            throw new BusinessException("调用销售域-获取客户组信息，客户编码为空");
        }
        try {
            AllowShipCustGroupParam param = new AllowShipCustGroupParam();
            param.setCodes(groupCodes);
            return allowShipRpcService.getCustGroup(param);
        } catch (Exception e) {
            log.error("获取客户组信息:", e);
            throw new BusinessException("销售域服务异常", e);
        }
    }

    /**
     * 获取客户组信息
     *
     * @param custCodes
     * @param type
     * @return
     */
    public List<AllowShipCustGroupDVO> getCustGroupInfo(List<String> custCodes, String type) {
        log.info("调用销售域-获取客户组信息,时间：{}，入参：{}", LocalDateTime.now(), custCodes);
        if (CollUtil.isEmpty(custCodes)) {
            throw new BusinessException("调用销售域-获取客户组信息，客户编码为空");
        }
        try {
            AllowShipCustGroupDParam param = new AllowShipCustGroupDParam();
            param.setCustCodes(custCodes);
            param.setType(type);
            param.setStatus(AllowShipStatus.ACTIVE.getType());
            return allowShipRpcService.getCustGroupInfo(param);
        } catch (Exception e) {
            log.error("获取客户组信息:", e);
            throw new BusinessException("销售域服务异常", e);
        }
    }


    /**
     * 商品满赠信息
     *
     * @return
     */
    public MktDiscountGiftRpcDTO findDiscountGift(MktDiscountGiftRpcParam giftRpcParam) {
        log.info("调用销售域-获取满赠信息,时间：{}，入参：{}", LocalDateTime.now(), giftRpcParam);
        ApiResult<MktDiscountGiftRpcDTO> giftResult = mktDiscountGiftRpcService.getGiftByParam(giftRpcParam);
        return giftResult.getData();
    }

    /**
     * 条件模糊满足的商品满赠信息
     *
     * @return
     */
    public List<MktDiscountGiftRpcDTO> findDiscountGiftList(MktDiscountGiftRpcParam giftRpcParam) {
        log.info("调用销售域-获取满赠信息,时间：{}，入参：{}", LocalDateTime.now(), giftRpcParam);
        ApiResult<List<MktDiscountGiftRpcDTO>> giftResult = mktDiscountGiftRpcService.getGiftListByParam(giftRpcParam);
        return giftResult.getData();
    }

    /**
     * 锁定或者释放赠品数量
     *
     * @param giftRpcParam
     */
    public void updateGiftLockNum(MktGiftUpdateNumRpcParam giftRpcParam) {
        log.info("调用销售域-锁定或者释放赠品数量,时间：{}，入参：{}", LocalDateTime.now(), giftRpcParam);
        ApiResult<Object> result = mktDiscountGiftRpcService.updateGiftLocknum(giftRpcParam);
        if (!result.isSuccess()) {
            throw new BusinessException("锁定或者释放赠品数量失败：" + result.getMsg());
        }
    }

    public Map<Long, SalesmanInfoBaseRespVO> findBaseMapByIds(List<Long> salesmanIds) {
        log.info("调用销售域-根据多个业务员id 或者 多个业务员编码查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salesmanIds);
        if (CollUtil.isEmpty(salesmanIds)) {
            return new HashMap<>();
        }
        SalesmanInfoSimpleQueryVO param = new SalesmanInfoSimpleQueryVO();
        param.setIds(salesmanIds);
        ApiResult<List<SalesmanInfoBaseRespVO>> salesmanInfo = salesmanRpcService.findBaseByParam(param);
        if (!salesmanInfo.isSuccess()) {
            throw new BusinessException("销售域服务异常:" + salesmanInfo.getMsg());
        }
        return salesmanInfo.getData().stream().collect(Collectors.toMap(SalesmanInfoBaseRespVO::getId, Function.identity()));
    }

    public SalesmanInfoBaseRespVO findBaseMapById(Long salesmanId) {
        log.info("调用销售域-根据多个业务员id 或者 多个业务员编码查询业务员信息,时间：{}，入参：{}", LocalDateTime.now(), salesmanId);
        if (salesmanId == null) {
            return new SalesmanInfoBaseRespVO();
        }
        SalesmanInfoSimpleQueryVO param = new SalesmanInfoSimpleQueryVO();
        param.setIds(List.of(salesmanId));
        ApiResult<List<SalesmanInfoBaseRespVO>> salesmanInfo = salesmanRpcService.findBaseByParam(param);
        if (!salesmanInfo.isSuccess()) {
            throw new BusinessException("销售域服务异常:" + salesmanInfo.getMsg());
        }
        return salesmanInfo.getData().get(0);
    }

    public Map<Long, CustBaseDTO> findCustBaseMap(List<Long> custIds) {
        log.info("查询客户信息入参:{}", JSONUtil.toJsonStr(custIds));
        if (CollUtil.isEmpty(custIds)) {
            return new HashMap<>();
        }
        CustBaseRpcParam custBaseRpcParam = new CustBaseRpcParam();
        custBaseRpcParam.setCustIdList(custIds);
        ApiResult<List<CustBaseDTO>> custInfo = crmCustRpcService.findBaseByParam(custBaseRpcParam);
        if (!custInfo.isSuccess()) {
            throw new BusinessException("查询客户数据信息接口异常");
        }
        log.info("查询客户信息结果:{}", JSONUtil.toJsonStr(custInfo.getData()));

        return custInfo.getData().stream().collect(Collectors.toMap(CustBaseDTO::getId, Function.identity()));
    }

    public CustBaseDTO findCustBaseByCustCode(String custCode) {
        log.info("查询客户信息入参:{}", JSONUtil.toJsonStr(custCode));
        CustBaseRpcParam custBaseRpcParam = new CustBaseRpcParam();
        custBaseRpcParam.setCustCodeList(List.of(custCode));
        ApiResult<List<CustBaseDTO>> custInfo = crmCustRpcService.findBaseByParam(custBaseRpcParam);
        if (!custInfo.isSuccess()) {
            throw new BusinessException("查询客户数据信息接口异常");
        }
        log.info("查询客户信息结果:{}", JSONUtil.toJsonStr(custInfo.getData()));

        return custInfo.getData().get(0);
    }
}
