package com.elitesland.scp.domain.service.order;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetPageRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandSetSaveVO;
import com.elitesland.scp.domain.convert.order.ScpDemandSetConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.entity.order.ScpDemandSetDO;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandSetDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandSetRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandSetRepoProc;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.support.provider.org.service.OrgRegionRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandSetDomainServiceImpl implements ScpDemandSetDomainService {

    private final ScpDemandSetRepo scpDemandSetRepo;
    private final ScpDemandSetRepoProc scpDemandSetRepoProc;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;
    private final OrgRegionRpcService orgRegionRpcService;
    private final ScpDemandOrderRepo scpDemandOrderRepo;
    private final RmiSysUDCService rmiSysUDCService;


    @Override
    @SysCodeProc
    public PagingVO<ScpDemandSetPageRespVO> queryDemandSetList(ScpDemandSetPageParamVO paramVO) {
        long count = scpDemandSetRepoProc.countDemandSet(paramVO);
        if (count > 0) {
            var demandSet = scpDemandSetRepoProc.queryDemandSet(paramVO);
            List<Long> demandIds = demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList());
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
            // 查询订货集下的订货明细单数量
            Map<Long, Long> itemQtyMap = scpDemandOrderDRepoProc.getDemandSumItem(demandIds);
            Set<String> regionCodes = demandSet.stream().map(ScpDemandSetPageRespVO::getRegionCode).filter(Objects::nonNull).collect(Collectors.toSet());
            Map<String, String> regionMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(regionCodes)) {
                regionMap = orgRegionRpcService.getNameByCode(regionCodes).computeData();
            }
            Map<String, String> finalRegionMap = regionMap;
            Map<String, String> createTypeMap = rmiSysUDCService.getCodeMap("yst-suplan", UdcEnum.SCP_ORDER_SET_CREATE_TYPE_01.getCode());

            demandSet.forEach(row -> {
                int qty = itemQtyMap.get(row.getId()) == null ? 0 : itemQtyMap.get(row.getId()).intValue();
                row.setDemandItemQty(qty);
                row.setCreateTypeName(createTypeMap.get(row.getCreateType()));
                if (row.getExpireTime() != null) {
                    row.setExpireTimeStr(row.getExpireTime().toLocalTime().format(formatter));
                }
                row.setRegionName(finalRegionMap.get(row.getRegionCode()));
            });
            return new PagingVO<>(count, demandSet);
        }
        return new PagingVO<>();
    }

    @Override
    @SysCodeProc
    public Optional<ScpDemandSetDTO> findDemandSetById(Long id) {
        return scpDemandSetRepo.findById(id).map(ScpDemandSetConvert.INSTANCE::doToDto);
    }

    @Override
    public Long saveDemandSet(ScpDemandSetSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpDemandSetDO intentDO = ScpDemandSetConvert.INSTANCE.saveVoToDO(saveVO);
            return scpDemandSetRepo.save(intentDO).getId();
        } else {
            Optional<ScpDemandSetDO> option = scpDemandSetRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("订货集ID：" + saveVO.getId() + "不存在");
            }
            ScpDemandSetConvert.INSTANCE.copySaveParamToDo(saveVO, option.get());
            Long masId = scpDemandSetRepo.save(option.get()).getId();
            return masId;
        }
    }

    @Override
    public void enableDemandSet(List<Long> ids, Boolean enable) {
        scpDemandSetRepoProc.enableDemandSet(ids, enable);
    }

    @Override
    public void deleteByIds(List<Long> ids) {
        scpDemandSetRepoProc.deleteByIds(ids);
    }

    @Override
    public List<ScpDemandSetDTO> findDemandSetByIds(List<Long> ids) {
        return scpDemandSetRepo.findByIdIn(ids).stream().map(ScpDemandSetConvert.INSTANCE::doToDto).collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandSetDTO> findDemandSetByParam(ScpDemandSetParamVO paramVO) {
        List<ScpDemandSetDTO> scpDemandSetDTOS = scpDemandSetRepoProc.findDemandSetByParam(paramVO);
        Set<Long> demandIds = scpDemandOrderRepo.findByDemandIdIn(scpDemandSetDTOS.stream().map(ScpDemandSetDTO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).distinct().collect(Collectors.toSet());
        scpDemandSetDTOS.removeIf(row -> demandIds.contains(row.getId()));
        return scpDemandSetDTOS;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDemandSetMsgByIds(List<Long> ids, String msg) {
        scpDemandSetRepoProc.updateDemandSetMsgByIds(ids, msg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDemandSetPushMsgById(List<Long> ids, String msg) {
        scpDemandSetRepoProc.updateDemandSetPushMsgById(ids, msg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void completePushById(Long id) {
        scpDemandSetRepoProc.completePushById(id);
    }

    @Override
    public Boolean getDemandSetStatusById(Long id) {
        return scpDemandSetRepoProc.getDemandSetStatusById(id);
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandSetPageRespVO> commonPage(ScpDemandSetPageParamVO paramVO) {
        long count = scpDemandSetRepoProc.countDemandSet(paramVO);
        if (count > 0) {
            var demandSet = scpDemandSetRepoProc.queryDemandSet(paramVO);
            Set<String> regionCodes = demandSet.stream().map(ScpDemandSetPageRespVO::getRegionCode).filter(Objects::nonNull).collect(Collectors.toSet());
            Map<String, String> regionMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(regionCodes)) {
                regionMap = orgRegionRpcService.getNameByCode(regionCodes).computeData();
            }
            Map<String, String> finalRegionMap = regionMap;
            demandSet.forEach(row -> {
                row.setRegionName(finalRegionMap.get(row.getRegionCode()));
            });
            return new PagingVO<>(count, demandSet);
        }
        return new PagingVO<>();
    }

    /**
     *      新增查询匹配订货集的公共服务接口，入参：订货单编码+类型（门店）+单据类型+门店+所属区域+下单时间+订货单主键
     *
     *      匹配优先级如下：
     *
     *      优先级一.1：类型+单据类型=下单单据类型+门店+下单时间 早于 订单截单时间；
     *
     *      优先级一.2：类型+单据类型=ALL+门店+下单时间 早于 订单截单时间；
     *
     *      优先级二.1：类型+单据类型=下单单据类型+区域+下单时间 早于 订单截单时间；
     *
     *      优先级二.2：类型+单据类型=ALL+区域+下单时间 早于 订单截单时间；
     *
     *      优先级三.1：类型+单据类型=下单单据类型+门店/区域=ALL+下单时间 早于 订单截单时间；
     *
     *      优先级三.2：类型+单据类型=ALL+门店/区域=ALL+下单时间 早于 订单截单时间；
     *
     *      出参：订货单编码+类型（门店）+单据类型+门店+所属区域+下单时间+订货单主键+订货集编码
     * @param paramVO
     * @return
     */
    @Override
    @SysCodeProc
    public List<ScpDemandSetPageRespVO> commonNewPage(ScpDemandSetPageParamVO paramVO) {

        List<ScpDemandSetPageRespVO> demandSet = new ArrayList<>();
        String regionCode = paramVO.getRegionCode();
        if(StringUtils.hasLength(paramVO.getStoreCode())){
            paramVO.setRegionCode(null);
            demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
            Set<Long> demandIds = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
            demandSet.removeIf(row -> demandIds.contains(row.getId()));
            //如果查询不到，查询订单类型为 all 的
            if(demandSet.isEmpty()){
                paramVO.setDocType("all");
                demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
                Set<Long> demandId2s = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
                demandSet.removeIf(row -> demandId2s.contains(row.getId()));
            }
        }
        if(demandSet.isEmpty()&&StringUtils.hasLength(regionCode)){
            paramVO.setStoreCode(null);
            paramVO.setRegionCode(regionCode);
            demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
            Set<Long> demandIds = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
            demandSet.removeIf(row -> demandIds.contains(row.getId()));
            if(demandSet.isEmpty()){
                paramVO.setDocType("all");
                demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
                Set<Long> demandId2s = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
                demandSet.removeIf(row -> demandId2s.contains(row.getId()));
            }
        }
        if(demandSet.isEmpty()){
            paramVO.setStoreCode(null);
            paramVO.setRegionCode(null);
            paramVO.setAllFlag(true);
            demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
            Set<Long> demandIds = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
            demandSet.removeIf(row -> demandIds.contains(row.getId()));
            if(demandSet.isEmpty()){
                paramVO.setDocType("all");
                demandSet = scpDemandSetRepoProc.queryDemandSetList(paramVO);
                Set<Long> demandId2s = scpDemandOrderRepo.findByDemandIdIn(demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList())).stream().map(ScpDemandOrderDO::getDemandId).collect(Collectors.toSet());
                demandSet.removeIf(row -> demandId2s.contains(row.getId()));
            }
        }

        return demandSet;
    }
}
