package com.elitesland.scp.application.web.order;

import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.dto.account.AccountAmtFlowRpcVO;
import com.elitesland.fin.param.account.AccountAppRpcParam;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemBatchParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderPageParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.*;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSaveVO;
import com.elitesland.scp.application.service.order.ScpDemandOrderService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;


@Slf4j
@Api(tags = "订货订单接口")
@RestController
@RequestMapping(value = "/order", produces = MediaType.APPLICATION_JSON_VALUE)
@AllArgsConstructor
public class ScpDemandOrderController {

    private final ScpDemandOrderService scpDemandOrderService;

    @ApiOperation(value = "订货订单 - 分页查询")
    @PostMapping(value = "/page")
    public ApiResult<PagingVO<ScpDemandOrderTitlePageRespVO>> page(@RequestBody ScpDemandOrderPageParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] page  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.page(paramVO));
    }

    @ApiOperation(value = "订货订单 - 订货单明细分页查询")
    @PostMapping(value = "/detail/page")
    public ApiResult<PagingVO<ScpDemandOrderPageRespVO>> queryDemandOrderList(@RequestBody ScpDemandOrderPageParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] queryDemandOrderList  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.queryDemandOrderPage(paramVO));
    }

    @ApiOperation("订货订单 - 订货订单详情信息")
    @GetMapping("/findIdOne/{id}")
    public ApiResult<ScpDemandOrderRespVO> findIdOne(@PathVariable Long id) {
        log.info("[SCP-DEMAND-ORDER] findIdOne, id ={}", id);
        return scpDemandOrderService.findDemandOrderCateById(id).map(ApiResult::ok)
                .orElse(ApiResult.fail(ApiCode.NOT_FOUND));
    }

    @PostMapping("/save")
    @ApiOperation("订货订单表头 - 保存")
    public ApiResult<Long> saveDemandOrder(@RequestBody @Validated ScpDemandOrderSaveVO saveVO) {
        log.info("[SCP-DEMAND-SET] saveDemandOrder  saveVO ={}", JSONUtil.toJsonStr(saveVO));
        return ApiResult.ok(scpDemandOrderService.saveDemandOrder(saveVO));
    }

    @ApiOperation(value = "订货订单 - 获取商品明细行信息")
    @PostMapping(value = "/item/list")
    public ApiResult<List<ScpDemandOrderDRespVO>> getItemList(@RequestBody @Validated ScpDemandOrderItemParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] getItemList  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.getItemList(paramVO));
    }

    @ApiOperation(value = "订货订单 - 批量获取商品明细行信息")
    @PostMapping(value = "/item/list/batch")
    public ApiResult<Map<String, List<ScpDemandOrderDRespVO>>> getItemListBatch(@RequestBody @Validated ScpDemandOrderItemBatchParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] getItemListBatch  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.getItemListBatch(paramVO));
    }

    @ApiOperation(value = "订货订单 - 取消")
    @PostMapping(value = "/cancel/{id}")
    public ApiResult<Object> getItemList(@PathVariable Long id) {
        log.info("[SCP-DEMAND-ORDER] getItemList  param ={}", id);
        scpDemandOrderService.cancelScpDemand(id);
        return ApiResult.ok();
    }

    @ApiOperation("订货单 - 推送")
    @PostMapping("/push/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "订货单推送", allowRepeatRequest = false)
    public ApiResult<Object> push(@PathVariable Long id) {
        log.info("[SCP-DEMAND-ORDER] push id = {}", JSONUtil.toJsonStr(id));
        scpDemandOrderService.push(id);
        return ApiResult.ok();
    }

    @ApiOperation(value = "订货订单 - 支付弹窗")
    @PostMapping(value = "/pay/{id}")
    public ApiResult<List<ScpDemandOrderPayInfoRespVO>> payInfo(@PathVariable Long id) {
        log.info("[SCP-DEMAND-ORDER] payInfo  param ={}", id);
        return ApiResult.ok(scpDemandOrderService.payInfo(id));
    }

    @ApiOperation(value = "订货订单 - 待付款列表")
    @PostMapping(value = "/pay/page")
    public ApiResult<PagingVO<ScpPayOrderPageRespVO>> payPage(@RequestBody ScpDemandOrderPageParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] page  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.payPage(paramVO));
    }

    @ApiOperation(value = "订货订单 - 关闭")
    @PostMapping(value = "/close/{id}")
    public ApiResult<List<ScpDemandOrderPayInfoRespVO>> close(@PathVariable Long id) {
        log.info("[SCP-DEMAND-ORDER] close  param ={}", id);
        scpDemandOrderService.close(id);
        return ApiResult.ok();
    }

    @ApiOperation(value = "订货子订单 - 关闭")
    @PostMapping(value = "/close/{id}/{type}")
    public ApiResult<List<ScpDemandOrderPayInfoRespVO>> close(@PathVariable Long id,@PathVariable String type) {
        log.info("[SCP-DEMAND-ORDER] close  param ={}", id);
        scpDemandOrderService.closePoOrTro(id, type);
        return ApiResult.ok();
    }

    @ApiOperation("订货订单 - 获取强配商品信息")
    @PostMapping("/alloc")
    public ApiResult<Map<String, List<ScpDemandOrderDRespVO>>> findEnableAllocSetting(@RequestBody @Validated ScpAllocSettingParamVO paramVO) {
        log.info("[SCP-DEMAND-ORDER] findEnableAllocSetting paramVO = {}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.findEnableAllocSetting(paramVO));
    }

}
