package com.elitesland.yst.production.aftersale.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.yst.production.aftersale.Application;
import com.elitesland.yst.production.aftersale.convert.PhoneRecClsConvert;
import com.elitesland.yst.production.aftersale.convert.PhoneRecordConvert;
import com.elitesland.yst.production.aftersale.model.entity.phonerecord.PhoneRecClsDO;
import com.elitesland.yst.production.aftersale.model.vo.PhoneRecClsRespVO;
import com.elitesland.yst.production.aftersale.model.vo.query.PhoneRecClsQueryVO;
import com.elitesland.yst.production.aftersale.model.vo.save.PhoneRecClsSaveVO;
import com.elitesland.yst.production.aftersale.service.PhoneRecClsService;
import com.elitesland.yst.production.aftersale.service.repo.PhoneRecClsRepo;
import com.elitesland.yst.production.aftersale.service.repo.PhoneRecClsRepoProc;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author zhengyue
 * @description: 400电话记录分类
 * @date 2024/4/18 16:56
 */
@Slf4j
@Service
public class PhoneRecClsServiceImpl implements PhoneRecClsService {

    @Autowired
    private PhoneRecClsRepo phoneRecClsRepo;
    @Autowired
    private PhoneRecClsRepoProc phoneRecClsRepoProc;
    @Autowired
    private SeqNumProvider seqNumProvider;

    @Override
    @Caching(evict = {
            @CacheEvict(cacheNames = "PHONE_REC_CLS", allEntries = true),
    })
    @Transactional(rollbackFor = {Exception.class})
    public Long create(PhoneRecClsSaveVO saveVO) {
        log.info("新建分类入参 {}", JSON.toJSONString(saveVO));
        if (saveVO.getPid().equals(saveVO.getId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "不允许将自己选择为上级");
        }
        // ============= 校验 相同父类下名称不能重复 =============
        if (phoneRecClsRepo.existsByNameAndPid(saveVO.getName(), saveVO.getPid())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分类名称已存在");
        }
        PhoneRecClsDO phoneRecClsDO = PhoneRecClsConvert.INSTANCE.saveVOToDO(saveVO);
        phoneRecClsDO.setCode(seqNumProvider.generateCode(Application.NAME, "phoneRecClsCode" , new ArrayList<>()));
        PhoneRecClsDO result = phoneRecClsRepo.save(phoneRecClsDO);
        return result.getId();
    }

    @Override
    @Caching(evict = {
            @CacheEvict(cacheNames = "PHONE_REC_CLS", allEntries = true),
    })
    public Long update(PhoneRecClsSaveVO saveVO) {
        log.info("编辑分类入参 {}", JSON.toJSONString(saveVO));
        val optional = phoneRecClsRepo.findById(saveVO.getId());
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据异常，未找到分类：" + saveVO.getName());
        }
        if (saveVO.getId().equals(saveVO.getPid())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "不允许将自己选择为上级");
        }
        // 校验相同父类下名称不能重复
        if (phoneRecClsRepo.existsByNameAndPidAndIdNot(saveVO.getName(), saveVO.getPid(), saveVO.getId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分类名称已存在");
        }
        val itmItemCateDO = optional.get();

        PhoneRecClsConvert.INSTANCE.updatePhoneRecClsDO(saveVO, itmItemCateDO);
        val result = phoneRecClsRepo.save(itmItemCateDO);

        return result.getId();
    }

    @Override
    @Caching(evict = {
            @CacheEvict(cacheNames = "PHONE_REC_CLS", allEntries = true),
    })
    @Transactional(rollbackFor = Exception.class)
    public Long delete(List<Long> ids) {
        log.info("删除分类入参 {}", JSON.toJSONString(ids));
        List<PhoneRecClsDO> allById = phoneRecClsRepo.findAllById(ids);
        if (CollUtil.isEmpty(allById)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分类不存在，请刷新后重试");
        }
        //判断是否有下级分类
        if (phoneRecClsRepo.existsByPidIn(ids)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "选中的分类存在下级，无法删除");
        }
        List<Long> deleteIds = allById.stream().map(PhoneRecClsDO::getId).collect(Collectors.toList());

        //软删除选中的分类信息
        return phoneRecClsRepoProc.updateDeleteFlag(deleteIds);
    }

    @Override
    public List<PhoneRecClsRespVO> query() {
        List<PhoneRecClsDO> phoneRecClsDOS = phoneRecClsRepo.findAll();
        if (CollUtil.isEmpty(phoneRecClsDOS)){
            return Collections.emptyList();
        }
        List<PhoneRecClsRespVO> comClgClassRespVOS = PhoneRecClsConvert.INSTANCE.phoneRecClsDOToRespVOs(phoneRecClsDOS);
        //一级分类 按照sortNo降序排列
        List<PhoneRecClsRespVO> base = comClgClassRespVOS.stream().filter(s -> s.getPid() == 0).sorted(Comparator.comparing(PhoneRecClsRespVO::getSortNo).reversed()).collect(Collectors.toList());
        //二级分类
        Map<Long, List<PhoneRecClsRespVO>> under = comClgClassRespVOS.stream().filter(s -> s.getPid() != 0).collect(Collectors.groupingBy(PhoneRecClsRespVO::getPid));
        if (CollUtil.isEmpty(under)){
            return base;
        }
        base.forEach(s->{
            //按照sortNo降序排列
            if (CollUtil.isNotEmpty(under.get(s.getId()))) {
                s.setUnder(under.get(s.getId()).stream().sorted(Comparator.comparing(PhoneRecClsRespVO::getSortNo).reversed()).collect(Collectors.toList()));
                s.setHasUnder(true);
            }else {
                s.setHasUnder(false);
            }
        });
        return base;
    }

    @Override
    public PhoneRecClsRespVO queryDetail(Long id) {
        log.info("分类明细查询入参 {}", id);
        Optional<PhoneRecClsDO> phoneRecClsDOOptional = phoneRecClsRepo.findById(id);
        if (phoneRecClsDOOptional.isEmpty()){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到对应的信息");
        }
        PhoneRecClsDO phoneRecClsDO = phoneRecClsDOOptional.get();
        List<PhoneRecClsDO> byPid = phoneRecClsRepo.findByPid(phoneRecClsDO.getId());
        PhoneRecClsRespVO comClgClassRespVO = PhoneRecClsConvert.INSTANCE.phoneRecClsDOToRespVO(phoneRecClsDO);
        comClgClassRespVO.setHasUnder(CollUtil.isNotEmpty(byPid));
        return comClgClassRespVO;
    }

    @Override
    public PagingVO<PhoneRecClsRespVO> queryByParam(PhoneRecClsQueryVO param) {
        log.info("分页查询参数 {}", JSON.toJSONString(param));
        List<PhoneRecClsRespVO> query = query();
        if (CollUtil.isEmpty(query)){
            return PagingVO.builder();
        }
        if (StringUtils.isNotBlank(param.getName())) {
            return searchByName(param, query);
        }
        query = query.stream().distinct().filter(s -> s.getPid() == 0)
                .sorted(Comparator.comparing(PhoneRecClsRespVO::getSortNo).reversed()).collect(Collectors.toList());
        List<PhoneRecClsRespVO> page = ListUtil.page(param.getCurrent(), param.getSize(), query);

        return PagingVO.<PhoneRecClsRespVO>builder()
                .total(query.size())
                .records(page)
                .build();
    }

    private static PagingVO<PhoneRecClsRespVO> searchByName(PhoneRecClsQueryVO param, List<PhoneRecClsRespVO> query) {
        List<PhoneRecClsRespVO> respVOS = new ArrayList<>();
        for (PhoneRecClsRespVO comClgClassRespVO : query) {
            if (comClgClassRespVO.getName().contains(param.getName())) {
                respVOS.add(comClgClassRespVO);
            }
            if (CollUtil.isNotEmpty(comClgClassRespVO.getUnder())) {
                for (PhoneRecClsRespVO clgClassRespVO : comClgClassRespVO.getUnder()) {
                    if (clgClassRespVO.getName().contains(param.getName())) {
                        respVOS.add(clgClassRespVO);
                        respVOS.add(comClgClassRespVO);
                    }
                }
            }
        }
        respVOS = respVOS.stream().distinct().filter(s -> s.getPid() == 0).sorted(Comparator.comparing(PhoneRecClsRespVO::getSortNo).reversed()).collect(Collectors.toList());
        List<PhoneRecClsRespVO> page = ListUtil.page(param.getCurrent(), param.getSize(), respVOS);

        return PagingVO.<PhoneRecClsRespVO>builder()
                .total(respVOS.size())
                .records(page)
                .build();
    }

    @Override
    public List<PhoneRecClsRespVO> querySub(Long id) {
        if (id == null){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "id不能为空！");
        }
        List<PhoneRecClsDO> classRepoByPid = phoneRecClsRepo.findByPid(id);
        if (CollUtil.isEmpty(classRepoByPid)){
            return Collections.emptyList();
        }
        return PhoneRecClsConvert.INSTANCE.phoneRecClsDOToRespVOs(classRepoByPid);
    }

    @Override
    public List<PhoneRecClsRespVO> queryBase() {
        List<PhoneRecClsDO> byPid = phoneRecClsRepo.findByPid(0L);
        if (CollUtil.isEmpty(byPid)){
            return Collections.emptyList();
        }
        return PhoneRecClsConvert.INSTANCE.phoneRecClsDOToRespVOs(byPid);
    }
}
