package com.elitesland.yst.production.aftersale.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;

import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitesland.yst.production.aftersale.Application;
import com.elitesland.yst.production.aftersale.convert.ComplaintSetConvert;
import com.elitesland.yst.production.aftersale.model.entity.ComplaintSetDO;
import com.elitesland.yst.production.aftersale.model.entity.phonerecord.PhoneRecClsDO;
import com.elitesland.yst.production.aftersale.model.param.BusinessComplaintSetParam;
import com.elitesland.yst.production.aftersale.model.param.ComplaintSetPageParam;
import com.elitesland.yst.production.aftersale.model.param.ComplaintSetParam;
import com.elitesland.yst.production.aftersale.model.vo.ComplaintSetVO;
import com.elitesland.yst.production.aftersale.service.ComplaintSetService;
import com.elitesland.yst.production.aftersale.service.repo.ComplaintSetRepo;
import com.elitesland.yst.production.aftersale.service.repo.ComplaintSetRepoProc;
import com.elitesland.yst.production.aftersale.util.UdcEnum;
import lombok.extern.log4j.Log4j2;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author elvis.hong
 * @title ComplaintSetServiceImpl
 * @Date 2023/3/1
 */
@Service
@Log4j2
public class ComplaintSetServiceImpl implements ComplaintSetService {

    @Autowired
    private ComplaintSetRepo complaintSetRepo;

    @Autowired
    private ComplaintSetRepoProc complaintSetRepoProc;

    @Autowired
    private SeqNumProvider seqNumProvider;

    @Override
    @SysCodeProc
    public List<ComplaintSetVO> complaintSetQuery(ComplaintSetPageParam complaintSetPageParam) {
        List<ComplaintSetVO> complaintSetList = complaintSetRepoProc.complaintSetQuery(complaintSetPageParam);
        return complaintSetList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long complaintSetSave(ComplaintSetParam complaintSetParam) {
        //分类编码，分类名称 非空校验
        Assert.notBlank(complaintSetParam.getClassifyCode(), "分类编码不能为空！");
        Assert.notBlank(complaintSetParam.getClassifyName(), "分类名称不能为空！");
        //新增 分类编码，分类名称不能重复
        if (complaintSetParam.getId() == null) {
            List<ComplaintSetVO> complaintSetList = complaintSetRepoProc.complaintSetExist(complaintSetParam);
            if (!CollUtil.isEmpty(complaintSetList)) {
                throw new BusinessException("分类编码，分类名称不能重复");
            }
        }
        //vo转do 并保存数据
        ComplaintSetDO complaintSetDO = ComplaintSetConvert.INSTANCE.complaintSetSaveVOToDO(complaintSetParam);
        Long id = complaintSetRepo.save(complaintSetDO).getId();
        return id;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void complaintSetDelete(List<Long> ids) {
        complaintSetRepoProc.complaintSetDelete(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void complaintSetEnabledToC(String enableFlag) {
        complaintSetRepoProc.complaintSetEnabledToC(enableFlag, "3", UdcEnum.BASIC_SET_ENABLE.getValueCode());
    }

    @Override
    public String complaintSetFlagByC(String complaintType) {
        ComplaintSetPageParam complaintSetPageParam = new ComplaintSetPageParam();
        complaintSetPageParam.setComplaintType(complaintType);
        complaintSetPageParam.setClassifyCode(UdcEnum.BASIC_SET_ENABLE.getValueCode());
        List<ComplaintSetVO> complaintSetList = complaintSetRepoProc.complaintSetQuery(complaintSetPageParam);
        //此处用于小程序前端判断是否展示【我的客诉】
        if (!CollUtil.isEmpty(complaintSetList)) {
            return complaintSetList.get(0).getEnableFlag();
        } else {
            ComplaintSetParam complaintSetParam = new ComplaintSetParam();
            complaintSetParam.setClassifyCode(UdcEnum.BASIC_SET_ENABLE.getValueCode());
            complaintSetParam.setClassifyName(UdcEnum.BASIC_SET_ENABLE.getValueCodeName());
            complaintSetParam.setComplaintType("3");
            complaintSetParam.setEnableFlag("1");
            this.complaintSetSave(complaintSetParam);
            return "1";
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long basicSetSave(List<ComplaintSetParam> complaintSetParamList) {
        Long id = null;
        if (!CollectionUtils.isEmpty(complaintSetParamList)) {
            for (ComplaintSetParam complaintSetParam : complaintSetParamList) {
                //分类编码，分类名称 非空校验
                Assert.notBlank(complaintSetParam.getClassifyCode(), "分类编码不能为空！");
                Assert.notBlank(complaintSetParam.getClassifyName(), "分类名称不能为空！");
                //系统进行默认评价的最长时间 仅支持整数
                if (complaintSetParam.getComplaintType().equals("3") && !this.isNumeric(complaintSetParam.getMaxHandle())) {
                    throw new BusinessException("系统进行默认评价的最长时间 仅支持整数！");
                }
                //vo转do 并保存数据
                ComplaintSetDO complaintSetDO = ComplaintSetConvert.INSTANCE.complaintSetSaveVOToDO(complaintSetParam);
                id = complaintSetRepo.save(complaintSetDO).getId();
            }
        }
        return id;
    }

    public boolean isNumeric(String str) {
        for (int i = str.length(); --i >= 0; ) {
            if (!Character.isDigit(str.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public List<ComplaintSetVO> businessComplaintSetQueryBase() {
        List<ComplaintSetVO> complaintSetList = complaintSetRepoProc.businessComplaintSetQueryBase();
        for (ComplaintSetVO rootCategory : complaintSetList) {
            List<ComplaintSetVO> subCategories = businessComplaintSetQuerySub(rootCategory.getId());
            rootCategory.setUnder(subCategories);
            rootCategory.setHasUnder(subCategories != null && !subCategories.isEmpty());
        }
        ComplaintSetVO complaintSetVO = new ComplaintSetVO();
        return complaintSetList;
    }

    @Override
    public List<ComplaintSetVO> businessComplaintSetQuerySub(Long id) {
        List<ComplaintSetVO> complaintSetList = complaintSetRepoProc.businessComplaintSetQuerySub(id);
        return complaintSetList;
    }


    private List<ComplaintSetVO> getAllSubCategories(Long pid) {
        List<ComplaintSetVO> categories = complaintSetRepoProc.businessComplaintSetQuerySub(pid);
        for (ComplaintSetVO category : categories) {
            List<ComplaintSetVO> subCategories = getAllSubCategories(category.getId());
            category.setUnder(subCategories);
            category.setHasUnder(subCategories != null && !subCategories.isEmpty());
        }
        return categories;
    }

    private List<ComplaintSetVO> flattenSubCategories(Long pid) {
        List<ComplaintSetVO> flatList = new ArrayList<>();
        List<ComplaintSetVO> categories = complaintSetRepoProc.businessComplaintSetQuerySub(pid);

        for (ComplaintSetVO category : categories) {
            flatList.add(category);
            flatList.addAll(flattenSubCategories(category.getId()));
        }

        return flatList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long businessComplaintSetCreate(BusinessComplaintSetParam businessComplaintSetParam) {
        //分类编码，分类名称 非空校验
        Assert.notBlank(businessComplaintSetParam.getClassifyName(), "分类名称不能为空！");
        Assert.notNull(businessComplaintSetParam.getPid(), "上级id不能为空");
        //校验 相同父类下名称不能重复
        if (complaintSetRepo.existsByClassifyNameAndPid(businessComplaintSetParam.getClassifyName(), businessComplaintSetParam.getPid())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分类名称已存在");
        }
        //B端客诉设置 最大响应时间，最大处理时间 仅支持整数
        if (!this.isNumeric(businessComplaintSetParam.getMaxHandle()) || !this.isNumeric(businessComplaintSetParam.getMaxResp())) {
            throw new BusinessException("最大响应时间，最大处理时间 仅支持整数！");
        }
        //vo转do 并保存数据
        ComplaintSetDO complaintSetDO = ComplaintSetConvert.INSTANCE.businessComplaintSetSaveVOToDO(businessComplaintSetParam);
        complaintSetDO.setClassifyCode(seqNumProvider.generateCode(Application.NAME, "complainSetClsCode", new ArrayList<>()));
        Long id = complaintSetRepo.save(complaintSetDO).getId();
        return id;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long businessComplaintSetUpdate(BusinessComplaintSetParam businessComplaintSetParam) {
        //分类编码，分类名称 非空校验
        Assert.notBlank(businessComplaintSetParam.getClassifyName(), "分类名称不能为空！");
        Assert.notNull(businessComplaintSetParam.getPid(), "上级id不能为空");
        val optional = complaintSetRepo.findById(businessComplaintSetParam.getId());
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据异常，未找到分类：" + businessComplaintSetParam.getClassifyName());
        }
        //不允许将自己以及自己的所有子集选为自己的上级
        //递归得出自己的所有子集标签
        if (businessComplaintSetParam.getId() != null) {
            List<ComplaintSetVO> complaintSetVOS = flattenSubCategories(businessComplaintSetParam.getId());
            List<Long> ids = complaintSetVOS.stream().map(ComplaintSetVO::getId).collect(Collectors.toList());
            ids.add(businessComplaintSetParam.getId());
            if (ids.contains(businessComplaintSetParam.getPid())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "不允许将自己或自己的子集选为自己的上级");
            }
        }
        //校验相同父类下名称不能重复
        if (complaintSetRepo.existsByClassifyNameAndPidAndIdNot(businessComplaintSetParam.getClassifyName(), businessComplaintSetParam.getPid(), businessComplaintSetParam.getId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分类名称已存在");
        }
        //B端客诉设置 最大响应时间，最大处理时间 仅支持整数
        if (!this.isNumeric(businessComplaintSetParam.getMaxHandle()) || !this.isNumeric(businessComplaintSetParam.getMaxResp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "最大响应时间，最大处理时间 仅支持整数！");
        }
        //vo转do 并保存数据
        ComplaintSetDO complaintSetDO = ComplaintSetConvert.INSTANCE.businessComplaintSetSaveVOToDO(businessComplaintSetParam);
        Long id = complaintSetRepo.save(complaintSetDO).getId();
        return id;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void businessComplaintSetDelete(List<Long> ids) {
        List<ComplaintSetDO> allById = complaintSetRepo.findAllById(ids);
        if (CollUtil.isEmpty(allById)) {
            throw new BusinessException("分类不存在，请刷新后重试");
        }
        if (complaintSetRepo.existsByPidIn(ids)) {
            throw new BusinessException("该分类下存在子分类，无法删除");
        }
        this.complaintSetRepoProc.complaintSetDelete(ids);
    }


    @Override
    public List<ComplaintSetVO> query() {
        List<ComplaintSetDO> complaintSetDOS = complaintSetRepo.findByComplaintType("2");
        if (CollUtil.isEmpty(complaintSetDOS)) {
            return Collections.emptyList();
        }
        List<ComplaintSetVO> complaintSetVOs = ComplaintSetConvert.INSTANCE.businessComplaintSetSaveDOToVOs(complaintSetDOS);
        // 将所有分类按父ID分组
        Map<Long, List<ComplaintSetVO>> groupedByParentId = complaintSetVOs.stream()
                .collect(Collectors.groupingBy(ComplaintSetVO::getPid));

        // 构建多级分类的树结构
        List<ComplaintSetVO> rootCategories = buildCategoryTree(0L, groupedByParentId);
        return rootCategories;
    }

    private List<ComplaintSetVO> buildCategoryTree(Long parentId, Map<Long, List<ComplaintSetVO>> groupedByParentId) {
        // 获取当前父ID下的子分类，并按sortNo降序排列
        List<ComplaintSetVO> categories = groupedByParentId.getOrDefault(parentId, Collections.emptyList())
                .stream()
                .sorted(Comparator.comparing(ComplaintSetVO::getSortNo).reversed())
                .collect(Collectors.toList());

        // 递归处理每个子分类
        for (ComplaintSetVO category : categories) {
            List<ComplaintSetVO> subCategories = buildCategoryTree(category.getId(), groupedByParentId);
            if (!subCategories.isEmpty()) {
                category.setUnder(subCategories);
                category.setHasUnder(true);
            } else {
                category.setHasUnder(false);
            }
        }

        return categories;
    }

}
