package com.elitesland.yst.production.aftersale.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.IdUtil;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.model.Result;
import com.elitescloud.boot.common.param.SysSendVerifyCodeVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.support.verifycode.common.VerifyCodeManager;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.authorization.sdk.cas.model.AuthUserDTO;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitesland.yst.production.aftersale.config.AuthorizationServiceConfiguration;
import com.elitesland.yst.production.aftersale.constant.Constant;
import com.elitesland.yst.production.aftersale.convert.CarOwnerInfoConvert;
import com.elitesland.yst.production.aftersale.model.bo.CarOwnerBasicBO;
import com.elitesland.yst.production.aftersale.model.entity.CarOwnerInfoDO;
import com.elitesland.yst.production.aftersale.model.param.CarOwnerInfoMngParam;
import com.elitesland.yst.production.aftersale.model.param.CarOwnerInfoParam;
import com.elitesland.yst.production.aftersale.model.param.RegisterMemberParam;
import com.elitesland.yst.production.aftersale.model.vo.CarMaintainCardVO;
import com.elitesland.yst.production.aftersale.model.vo.CarOwnerInfoMngRespVO;
import com.elitesland.yst.production.aftersale.model.vo.CarOwnerPageInfoVO;
import com.elitesland.yst.production.aftersale.model.vo.CarOwnerVehicleInfoVO;
import com.elitesland.yst.production.aftersale.provider.system.SystemRpcProvider;
import com.elitesland.yst.production.aftersale.service.CarOwnerInfoService;
import com.elitesland.yst.production.aftersale.service.CarOwnerVehicleService;
import com.elitesland.yst.production.aftersale.service.MaintainCardService;
import com.elitesland.yst.production.aftersale.service.repo.CarOwnerInfoRepo;
import com.elitesland.yst.production.aftersale.service.repo.CarOwnerInfoRepoProc;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
/**
 * @author Danny.Wang
 * @title CarOwnerInfoServiceImpl
 * @Date2023/2/8 13:29
 */
@Service
@Log4j2
public class CarOwnerInfoServiceImpl extends BaseServiceImpl implements CarOwnerInfoService {

    @Autowired
    private Constant constant;

    @Autowired
    private VerifyCodeManager verifyCodeManager;

    @Autowired
    private CarOwnerInfoRepo carOwnerInfoRepo;

    @Autowired
    private CarOwnerInfoRepoProc carOwnerInfoRepoProc;

    @Autowired
    private CarOwnerVehicleService carOwnerVehicleService;

    @Autowired
    private UserService userService;

    @Autowired
    private MaintainCardService maintainCardService;

    @Autowired
    private SystemRpcProvider systemRpcProvider;

    @Autowired
    private AuthorizationServiceConfiguration authorizationServiceConfiguration;

    @Autowired
    private RedissonClient redissonClient;

    @Value("${file.upload.url}")
    private String pictureURL;

    private final String  AFTER_KAPCHA="AFTER_KAPCHA";

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Object> save(CarOwnerInfoParam carOwnerInfoParam) {
        Long userId = userService.getUserId();
        carOwnerInfoParam.setCarOwnerId(userId);

        CarOwnerInfoDO ownerInfoDO = carOwnerInfoRepo.findByCarOwnerId(carOwnerInfoParam.getCarOwnerId());
        if (Optional.ofNullable(ownerInfoDO).isEmpty()) {
            throw new BusinessException("未查询到相应信息，请稍后重试");
        }
        CarOwnerInfoDO carOwnerInfoDO = CarOwnerInfoConvert.INSTANCE.carOwnerInfoVOTODO(carOwnerInfoParam);
        carOwnerInfoDO.setId(ownerInfoDO.getId());
        carOwnerInfoDO.setCasUserId(ownerInfoDO.getCasUserId());
        if (StringUtils.isNotBlank(carOwnerInfoParam.getBirthdayTime())) {
            carOwnerInfoDO.setBirthdayTime(constant.swapTimeMin(carOwnerInfoParam.getBirthdayTime()));
        }

        if (StringUtils.isBlank(carOwnerInfoDO.getUserName())) {
//            throw new BusinessException("姓名不能为空");
        }
        //名字为空则随机生成
        if (StringUtils.isBlank(carOwnerInfoDO.getUserName())) {
            carOwnerInfoDO.setUserName("立马车主" + constant.getRandomNum(8));
        }
        carOwnerInfoRepo.save(carOwnerInfoDO);

        // 0710新增逻辑车主表更新同步更新绑车用户信息
        carOwnerVehicleService.updateCarowner(carOwnerInfoDO);

        return ApiResult.ok();
    }

    @Override
    public ApiResult<Object> query() {
        Long userId = userService.getUserId();
        CarOwnerInfoDO carOwnerInfoDO = carOwnerInfoRepo.findByCarOwnerId(userId);
        if(!StringUtils.isEmpty(carOwnerInfoDO.getUrl())){
            if(!carOwnerInfoDO.getUrl().startsWith("http")){
                carOwnerInfoDO.setUrl(pictureURL + carOwnerInfoDO.getUrl());
            }
        }
        return ApiResult.ok(carOwnerInfoDO);
    }

    @Override
    public ApiResult<Object> pageInfoQuery() {
        Long userId = userService.getUserId();
        CarOwnerPageInfoVO pageInfo = carOwnerInfoRepoProc.findPageInfo(userId);
        if (Optional.ofNullable(pageInfo).isEmpty()) {
            throw new BusinessException("未查询到信息，请稍后重试");
        }
        if(!StringUtils.isEmpty(pageInfo.getUrl())){
            if(!pageInfo.getUrl().startsWith("http")){
                pageInfo.setUrl(pictureURL + pageInfo.getUrl());
            }
        }
        ApiResult<Object> bindVehicleQuery = carOwnerVehicleService.bindVehicleQuery();
        List<CarOwnerVehicleInfoVO> bindVehicleQueryData = (List<CarOwnerVehicleInfoVO>) bindVehicleQuery.getData();
        if (CollUtil.isEmpty(bindVehicleQueryData)) {
            ApiResult.ok(pageInfo);
        }
        pageInfo.setCarOwnerVehicleInfoVOs(bindVehicleQueryData);
        return ApiResult.ok(pageInfo);

    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> registerCarOwnerOnNotExists(String phone, Long casUserId, String userName) {
        String lockKey = "11" + phone + "-" + casUserId;
        RLock rLock = redissonClient.getLock("CarOwner:" + lockKey);

        try{
            if (!rLock.tryLock(5,300, TimeUnit.SECONDS)){
                throw new BusinessException("获取锁失败，请稍后重试");
            }

            if (redissonClient.getBucket(lockKey).get() == null){
                redissonClient.getBucket(lockKey).set("k" ,3 ,TimeUnit.SECONDS);
            } else{
                throw new BusinessException("相同时间内存在疑似重复提交");
            }

        } catch (BusinessException | InterruptedException e) {
            throw new BusinessException("注册存在异常");
        } finally {
            if(rLock.isHeldByCurrentThread()){
                rLock.unlock();
            }
        }


        // 获取小程序所属租户
        Long tenantId = this.obtainTenantIdFromWechat();
        Assert.notNull(tenantId, "未知小程序所属租户");

        if (StringUtils.isBlank(userName)) {
//            throw new BusinessException("姓名不能为空");
        }
        if (StringUtils.isBlank(userName)) {
            userName = "立马车主" + constant.getRandomNum(8);
        }
        if (userName.contains("auto")) {
            userName = "立马车主" + constant.getRandomNum(8);
        }
        String finalUserName = userName;
        Supplier<Long> supplierCarOwnerQuery = () -> {
            // 先查询车主信息
            Long carOwnerId = carOwnerInfoRepoProc.findByCasUserIdAndUserPhone(phone, casUserId);
            if (carOwnerId != null) {
                return carOwnerId;
            }
            //根据手机号查询  不存在则注册一个新数据
            Long aLong = carOwnerInfoRepoProc.findByUserPhone(phone);
            if(aLong==null){
                // 不存在则注册车主
                CarOwnerInfoDO carOwnerInfoDO = new CarOwnerInfoDO();
                carOwnerInfoDO.setCarOwnerId(IdUtil.getSnowflakeNextId());
                log.info("注册车主{}, {}", tenantId, carOwnerInfoDO.getCarOwnerId());
                carOwnerInfoDO.setUserPhone(phone);
                carOwnerInfoDO.setCasUserId(casUserId);
                carOwnerInfoDO.setUserName(finalUserName);
                carOwnerInfoDO.setTenantId(tenantId);
                return carOwnerInfoRepo.save(carOwnerInfoDO).getCarOwnerId();
            }else{
               throw new BusinessException("该手机号已被绑定,请检查!");
            }
        };
        var tenant = tenantClientProvider.getTenant(tenantId);
        var curOwnerId = tenant == null ? supplierCarOwnerQuery.get() : tenantDataIsolateProvider.byTenantDirectly(supplierCarOwnerQuery::get, tenant);

        return ApiResult.ok(curOwnerId);
    }

    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> registerMemberPc(String phone, String userName, Long casUserId) {
        CarOwnerInfoDO carOwnerInfoDO = new CarOwnerInfoDO();
        carOwnerInfoDO.setCarOwnerId(IdUtil.getSnowflakeNextId());
        log.info("生成的车主id {}", carOwnerInfoDO.getCarOwnerId());
        carOwnerInfoDO.setUserPhone(phone);
        carOwnerInfoDO.setCasUserId(casUserId);

        if (StringUtils.isBlank(userName)) {
//            throw new BusinessException("姓名不能为空");
        }
        carOwnerInfoDO.setUserName(StringUtils.isNotEmpty(userName) ? userName : ("立马车主" + constant.getRandomNum(8)));
        CarOwnerInfoDO save = carOwnerInfoRepo.save(carOwnerInfoDO);
        return ApiResult.ok(save.getCarOwnerId());
    }

    @Override
    public ApiResult<CarOwnerBasicBO> getCarOwnerByCarOwnerId(Long carOwnerId) {
        // 获取小程序所属租户
        Long tenantId = this.obtainTenantIdFromWechat();
        log.error("车主认证：{}, {}", tenantId, carOwnerId);
        var tenant = tenantClientProvider.getTenant(tenantId);
        return tenantDataIsolateProvider.byTenantDirectly(() -> {
            CarOwnerInfoDO carOwnerInfoDO = carOwnerInfoRepo.findByCarOwnerId(carOwnerId);
            if (Optional.ofNullable(carOwnerInfoDO).isPresent()) {
                var userByMobile = UserTransferHelper.getInstance(authorizationServiceConfiguration.getAuthServer()).getUserByMobile(carOwnerInfoDO.getUserPhone());
                if (userByMobile.getSuccess().equals(false)) {
                    throw new BusinessException("getCarOwnerByCarOwnerId调用认证域异常，" + userByMobile.getMsg());
                }
                if(Objects.isNull(userByMobile.getData())){
                    throw new BusinessException("认证域-根据手机号查询账号信息为空，" + userByMobile.getMsg());
                }
                CarOwnerBasicBO carOwnerBasicBO = new CarOwnerBasicBO();
                carOwnerBasicBO.setCarOwnerId(carOwnerInfoDO.getCarOwnerId());
                carOwnerBasicBO.setUserName(userByMobile.getData().getUsername());
                carOwnerBasicBO.setUserPhone(carOwnerInfoDO.getUserPhone());
                carOwnerBasicBO.setTenantDTO(tenant);
                return ApiResult.ok(carOwnerBasicBO);
            }
            return ApiResult.ok();
        }, tenant);
    }

    @Override
    @SysCodeProc
    public PagingVO<CarOwnerInfoMngRespVO> ownerInfoMngQuery(CarOwnerInfoMngParam param) {
        PagingVO<CarOwnerInfoMngRespVO> ownerInfoMngQuery = carOwnerInfoRepoProc.ownerInfoMngQuery(param);
        ownerInfoMngQuery.getRecords().forEach(vo->{
            if(!StringUtils.isEmpty(vo.getUrl())){
                if(!vo.getUrl().startsWith("http")){
                    vo.setUrl(pictureURL + vo.getUrl());
                }
            }
        });
        return ownerInfoMngQuery;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void ownerInfoMngUpdateStatus(List<Long> casUserIds, boolean enabled, int enabledFlag) {
        if (!CollUtil.isEmpty(casUserIds)) {
            casUserIds.stream().forEach(casUserId -> {
                log.info("启用/禁用会员【{}, {}, {}】", casUserId, enabled, enabledFlag);
                //调用统一身份认证服务 启用/禁用会员
                Result<Long> userData = UserTransferHelper.getInstance(authorizationServiceConfiguration.getAuthServer()).updateEnabled(casUserId, enabled);
                if (userData.getSuccess().equals(false)) {
                    throw new BusinessException("启用/禁用会员失败，" + userData.getMsg());
                }
                //车主信息表 启用/禁用会员
                carOwnerInfoRepoProc.updateEnabledFlag(casUserId, enabledFlag);
            });
        }
    }

    @Override
    @SysCodeProc
    public CarOwnerInfoMngRespVO ownerInfoMngDetail(Long carOwnerId) {
        CarOwnerInfoDO ownerInfoDO = carOwnerInfoRepo.findByCarOwnerId(carOwnerId);
        if (Optional.ofNullable(ownerInfoDO).isEmpty()) {
            throw new BusinessException("未查询到相应信息，请稍后重试");
        }
        //车主信息
        CarOwnerInfoMngRespVO carOwnerInfoMngRespVO = CarOwnerInfoConvert.INSTANCE.carOwnerInfoDOTOVO(ownerInfoDO);
        if(!StringUtils.isEmpty(carOwnerInfoMngRespVO.getUrl())){
            if(!carOwnerInfoMngRespVO.getUrl().startsWith("http")){
                carOwnerInfoMngRespVO.setUrl(pictureURL + carOwnerInfoMngRespVO.getUrl());
            }
        }
        //绑定车辆信息
        List<CarOwnerVehicleInfoVO> carOwnerVehicleInfoVOS = carOwnerVehicleService.bindVehicleMngQuery(carOwnerId);
        if (!CollUtil.isEmpty(carOwnerVehicleInfoVOS)) {
            carOwnerVehicleInfoVOS.stream().forEach(s -> {
                //车辆三包信息
                ApiResult<CarMaintainCardVO> maintainCard = maintainCardService.queryById(s.getVehicleNo());
                if (maintainCard.getData() != null) {
                    CarMaintainCardVO carMaintainCardVO = (CarMaintainCardVO) maintainCard.getData();
                    s.setCarMaintainCardVO(carMaintainCardVO);
                } else {
                    s.setCarMaintainCardVO(null);
                }
            });
        }
        carOwnerInfoMngRespVO.setCarOwnerVehicleInfoVOS(carOwnerVehicleInfoVOS);
        return carOwnerInfoMngRespVO;
    }

    @Override
    @SysCodeProc
    public CarOwnerInfoMngRespVO ownerInfoByUserPhone(String userPhone) {

        if (StringUtils.isBlank(userPhone)) {
            throw new BusinessException("手机号不能为空");
        }
        List<CarOwnerInfoDO> carOwnerInfoDOS = carOwnerInfoRepoProc.queryByUserPhone(userPhone);
        if (CollectionUtils.isEmpty(carOwnerInfoDOS)) {
            //前端根据错误码500判断是否跳转注册页面
            throw new BusinessException(ApiCode.FAIL,"根据手机号未查询到相关客户信息");
        }else if(!CollectionUtils.isEmpty(carOwnerInfoDOS)&&carOwnerInfoDOS.size()>1){
            throw new BusinessException("根据手机号查询到多个车主信息");
        }
        CarOwnerInfoMngRespVO carOwnerInfoMngRespVO = CarOwnerInfoConvert.INSTANCE.carOwnerInfoDOTOVO(carOwnerInfoDOS.get(0));
        // 新增绑车信息
        List<CarOwnerVehicleInfoVO> res = carOwnerVehicleService.findCarByUserPhone(userPhone);
        carOwnerInfoMngRespVO.setCarOwnerVehicleInfoVOS(res);
        return carOwnerInfoMngRespVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long registerMember(RegisterMemberParam registerMemberParam) {
        log.info("注册会员【{}, {}, {}】", registerMemberParam.getMobile(), registerMemberParam.getUserName(), registerMemberParam.getId());

        Long casUserId = null;

        CarOwnerInfoDO carOwnerInfoDO = carOwnerInfoRepo.findByUserPhone(registerMemberParam.getMobile());
        if (Optional.ofNullable(carOwnerInfoDO).isPresent()) {
            throw new BusinessException("当前手机号已注册");
        }
        String verify = verifyCodeManager.verify(AFTER_KAPCHA, registerMemberParam.getMobile(), registerMemberParam.getKaptcha());
        if (!org.springframework.util.ObjectUtils.isEmpty(verify)){
            throw new BusinessException(verify);
        }
//        if (StringUtils.isBlank(registerMemberParam.getUserName())) {
//            throw new BusinessException("姓名不能为空");
//        }

        //查询统一认证服务中此手机号是否有效
        var userByMobile = UserTransferHelper.getInstance(authorizationServiceConfiguration.getAuthServer()).getUserByMobile(registerMemberParam.getMobile());

        if (userByMobile.getSuccess().equals(false)) {
            throw new BusinessException("查询统一认证服务手机号失败，" + userByMobile.getMsg());
        }
        if (Objects.isNull(userByMobile.getData())) {
            AuthUserDTO userInfoDTO = new AuthUserDTO();
            userInfoDTO.setMobile(registerMemberParam.getMobile());
            userInfoDTO.setUsername(StringUtils.isNotEmpty(registerMemberParam.getUserName()) ? registerMemberParam.getUserName() : ("立马车主" + constant.getRandomNum(8)));
            //调用统一身份认证服务 保存账号
            Result<Long> userData = UserTransferHelper.getInstance(authorizationServiceConfiguration.getAuthServer()).upsertUser(userInfoDTO);
            if (userData.getSuccess().equals(false)) {
                throw new BusinessException("注册失败，" + userData.getMsg());
            }
            casUserId = userData.getData();
        } else {
            casUserId = userByMobile.getData().getId();
        }


        ApiResult<Long> result = null;
        try {
            //保存至车主信息表
            result = this.registerMemberPc(registerMemberParam.getMobile(), registerMemberParam.getUserName(), casUserId);
        } catch (Exception e) {
            log.error("注册会员异常：{}", e);
            throw new BusinessException("注册失败，" + e.getMessage());
        }

        if (result != null && result.getData() != null) {
            return result.getData();
        }

        throw new BusinessException("注册会员失败，" + (result == null ? "系统异常" : result.getMsg()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long cancelMember(RegisterMemberParam registerMemberParam) {
        log.info("注销会员【{}, {}, {}】", registerMemberParam.getUserName(), registerMemberParam.getCasUserId(), registerMemberParam.getCarOwnerId());

        //调用统一身份认证服务 注销账号
        Result<Long> userData = UserTransferHelper.getInstance(authorizationServiceConfiguration.getAuthServer()).updateEnabled(registerMemberParam.getCasUserId(), registerMemberParam.getEnabled());
        if (userData.getSuccess().equals(false)) {
            throw new BusinessException("注销失败，" + userData.getMsg());
        }

        //车主信息表 禁用账号
        carOwnerInfoRepoProc.updateEnabledFlag(registerMemberParam.getCasUserId(), 0);

        return null;
    }

    @Override
    public void getKaptcha(String phone) {
        SysSendVerifyCodeVO sysSendVerifyCodeVO = new SysSendVerifyCodeVO();
        sysSendVerifyCodeVO.setAccount(phone);
        sysSendVerifyCodeVO.setAccountType("mobile");
        sysSendVerifyCodeVO.setPeriod(600L);
        verifyCodeManager.send(AFTER_KAPCHA,sysSendVerifyCodeVO);
    }

    private Long obtainTenantIdFromWechat() {
        // 获取小程序所属租户
        var request = HttpServletUtil.currentRequest();
        Assert.notNull(request, "未获取到有效HttpServletRequest");
        String wechatAppId = request.getParameter("wx_appid");
        Assert.hasText(wechatAppId, "wx_appid为空");
        Long tenantIdByAccount = systemRpcProvider.getTenantIdByAccount(wechatAppId);
        return tenantIdByAccount;
    }
}
