package com.elitesland.yst.production.aftersale.service.impl;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;

import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.yst.production.aftersale.convert.OrderEvaluationConvert;
import com.elitesland.yst.production.aftersale.convert.PictureOrderConvert;
import com.elitesland.yst.production.aftersale.model.entity.OrderEvaluationDO;
import com.elitesland.yst.production.aftersale.model.entity.picture.PictureOrderDO;
import com.elitesland.yst.production.aftersale.model.param.*;
import com.elitesland.yst.production.aftersale.model.vo.OrderEvaluationVO;
import com.elitesland.yst.production.aftersale.model.vo.PictureOrderVO;
import com.elitesland.yst.production.aftersale.out.sys.SystemService;
import com.elitesland.yst.production.aftersale.service.OrderEvaluationService;
import com.elitesland.yst.production.aftersale.service.OrderLogService;
import com.elitesland.yst.production.aftersale.service.PictureOrderService;
import com.elitesland.yst.production.aftersale.service.repo.OrderEvaluationRepo;
import com.elitesland.yst.production.aftersale.service.repo.OrderEvaluationRepoProc;
import com.elitesland.yst.production.aftersale.service.repo.PictureOrderRepo;
import com.elitesland.yst.production.aftersale.util.UdcEnum;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author RYH
 * @date 2023/2/14
 */
@Service
@Log4j2
public class OrderEvaluationServiceImpl implements OrderEvaluationService {

    @Autowired
    private OrderEvaluationRepo orderEvaluationRepo;

    @Autowired
    private PictureOrderRepo pictureOrderRepo;

    @Autowired
    private OrderEvaluationRepoProc orderEvaluationRepoProc;

    @Autowired
    private SystemService systemService;

    @Autowired
    private PictureOrderService pictureOrderService;

    @Autowired
    private OrderLogService orderLogService;

    @Autowired
    private UserService userService;

    @Value("${file.upload.url}")
    private String pictureURL;

    private static final String EVALUATION = "工单评论";
    private static final String RETURN_APPLY = "工单回访";


    @Override
    @Transactional
    public Long save(com.elitesland.yst.production.aftersale.model.param.OrderEvaluationParam saveVO) {
        log.info("工单评价保存入参", JSON.toJSONString(saveVO));
        //必填字段校验
        this.check(saveVO);

        OrderEvaluationVO oldEvaluation = orderEvaluationRepoProc.query(saveVO);
        if (null != oldEvaluation) {
            throw new BusinessException(ApiCode.FAIL, "该工单已评论，请勿重复评论！");
        }

        //保存工单评论
        OrderEvaluationDO orderEvaluationDO = OrderEvaluationConvert.INSTANCE.saveVoTODO(saveVO);
        Long id = orderEvaluationRepo.save(orderEvaluationDO).getId();

        //图片集合调用图片工单service进行保存
        List<com.elitesland.yst.production.aftersale.model.param.PictureOrderParam> pictureOrderParamList = saveVO.getPictureOrderParamList();
        if (null != pictureOrderParamList) {
            List<PictureOrderDO> collect = pictureOrderParamList.stream().map(t -> {
                PictureOrderDO pictureOrderDO = PictureOrderConvert.INSTANCE.saveVoTODO(t);
                pictureOrderDO.setOrderId(orderEvaluationDO.getOrderId());
                pictureOrderDO.setOrderType(UdcEnum.PICTURE_ORDER_TYPE_3.getValueCode());
                return pictureOrderDO;
            }).collect(Collectors.toList());
            pictureOrderRepo.saveAll(collect);
        }

        //日志添加
        SysUserDTO user = userService.getUserDetail();
        String remark = "";
        if (!StringUtils.isBlank(saveVO.getEvaluationLevel())) {
            Map<String, String> ajStatusMap = systemService.sysUdcGetCodeMap(UdcEnum.EVALUATION_LEVEL_1.getModel(), UdcEnum.EVALUATION_LEVEL_1.getCode());
            if (!CollectionUtils.isEmpty(ajStatusMap) && ajStatusMap.containsKey(saveVO.getEvaluationLevel())) {
                remark = remark + "评论级别：" + ajStatusMap.get(saveVO.getEvaluationLevel());
            }
        }
        if (!StringUtils.isBlank(saveVO.getEvaluationContent())) {
            remark = remark + " 评论内容：" + saveVO.getEvaluationContent();
        }
        OrderLogSaveParam orderLogSaveParam = new OrderLogSaveParam(saveVO.getOrderId(), saveVO.getOrderType(), EVALUATION, LocalDateTime.now(), user.getUsername(), user.getId());
        orderLogSaveParam.setRemark(remark);
        orderLogService.orderLogSave(orderLogSaveParam);
        return id;
    }

    @Override
    public OrderEvaluationVO query(OrderEvaluationParam param) {
        OrderEvaluationVO orderEvaluationVO = orderEvaluationRepoProc.query(param);

        if (null == orderEvaluationVO) {
            return null;
        }

        //获取评论图片
        PictureOrderParam pictureOrderParam = new PictureOrderParam();
        List<Long> idList = new ArrayList<>();
        idList.add(orderEvaluationVO.getOrderId());
        pictureOrderParam.setOrderList(idList);
        pictureOrderParam.setOrderType(UdcEnum.PICTURE_ORDER_TYPE_3.getValueCode());
        List<PictureOrderVO> data = pictureOrderService.query(pictureOrderParam).getData();

        for (PictureOrderVO pictureOrderVO : data) {
            pictureOrderVO.setFullUrl(pictureURL + pictureOrderVO.getUrl());
        }
        orderEvaluationVO.setPictureOrderVOS(data);

        //udc翻译
        this.udcTranslate(orderEvaluationVO);
        return orderEvaluationVO;
    }

    @Override
    public PagingVO<OrderEvaluationVO> pageQuery(OrderEvaluationPageParam param) {
        PagingVO<OrderEvaluationVO> pagingVO = null;
        if (param.getOrderType().equals(UdcEnum.EVALUATION_ORDER_TYPE_1.getValueCode())) {
            pagingVO = orderEvaluationRepoProc.repairPageQuery(param);
        } else if (param.getOrderType().equals(UdcEnum.EVALUATION_ORDER_TYPE_2.getValueCode())) {
            pagingVO = orderEvaluationRepoProc.complaintPageQuery(param);
        }

        //udc翻译
        if (null != pagingVO)
            pagingVO.getRecords().forEach(this::udcTranslate);

        return pagingVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void orderEvaluationDelete(List<Long> ids) {
        log.info("工单评论删除入参", ids);
        if (ids == null || ids.isEmpty()) {
            throw new BusinessException(ApiCode.FAIL, "id为空，请检查");
        }

        orderEvaluationRepoProc.deleteOrderEvaluation(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void orderEvaluationSolve(OrderEvaluationParam saveVO) {
        if (null == saveVO.getId()) {
            throw new BusinessException(ApiCode.FAIL, "id为空，请检查！");
        }

        orderEvaluationRepoProc.solve(saveVO);

        //日志添加
        SysUserDTO user = userService.getUserDetail();
        String remark = "";
        if (!StringUtils.isBlank(saveVO.getReturnLevel())) {
            Map<String, String> ajStatusMap = systemService.sysUdcGetCodeMap(UdcEnum.RETURN_LEVEL_1.getModel(), UdcEnum.RETURN_LEVEL_1.getCode());
            if (!CollectionUtils.isEmpty(ajStatusMap) && ajStatusMap.containsKey(saveVO.getReturnLevel())) {
                remark = remark + "回访满意度：" + ajStatusMap.get(saveVO.getReturnLevel());
            }
        }
        if (!StringUtils.isBlank(saveVO.getReturnReply())) {
            remark = remark + " 处理回复:" + saveVO.getReturnReply();
        }
        if (!StringUtils.isBlank(saveVO.getReturnAdvice())) {
            remark = remark + " 改善意见:" + saveVO.getReturnAdvice();
        }
        OrderLogSaveParam orderLogSaveParam = new OrderLogSaveParam(saveVO.getOrderId(), saveVO.getOrderType(), RETURN_APPLY, LocalDateTime.now(), user.getUsername(), user.getId());
        orderLogSaveParam.setRemark(remark);
        orderLogService.orderLogSave(orderLogSaveParam);
    }

    @Override
    public List<OrderEvaluationVO> queryList(OrderEvaluationParam orderEvaluationParam) {
        List<OrderEvaluationVO> orderEvaluationVOS = orderEvaluationRepoProc.queryList(orderEvaluationParam);
        orderEvaluationVOS.forEach(this::udcTranslate);
        return orderEvaluationVOS;
    }

    @Override
    public void batchSave(List<OrderEvaluationDO> orderEvaluationDOS) {
        orderEvaluationRepo.saveAll(orderEvaluationDOS);
    }

    private void check(com.elitesland.yst.production.aftersale.model.param.OrderEvaluationParam saveVO) {
        log.info("工单评论提交入参", JSON.toJSONString(saveVO));
        if (saveVO.getOrderType() == null) {
            throw new BusinessException(ApiCode.FAIL, "工单类型为空，请检查！");
        }
        if (saveVO.getOrderId() == null) {
            throw new BusinessException(ApiCode.FAIL, "工单id为空，请检查！");
        }
        if (saveVO.getEvaluationLevel() == null) {
            throw new BusinessException(ApiCode.FAIL, "评论级别为空，请检查！");
        }
    }

    /**
     * UDC翻译
     *
     * @param orderEvaluationVO
     */
    public void udcTranslate(OrderEvaluationVO orderEvaluationVO) {
        Map<String, String> ajStatusMap = systemService.sysUdcGetCodeMap(UdcEnum.EVALUATION_LEVEL_1.getModel(), UdcEnum.EVALUATION_LEVEL_1.getCode());
        if (!CollectionUtils.isEmpty(ajStatusMap) && ajStatusMap.containsKey(orderEvaluationVO.getEvaluationLevel())) {
            orderEvaluationVO.setEvaluationLevelName(ajStatusMap.get(orderEvaluationVO.getEvaluationLevel()));
        }

        Map<String, String> ajStatusMap2 = systemService.sysUdcGetCodeMap(UdcEnum.EVALUATION_ORDER_TYPE_1.getModel(), UdcEnum.EVALUATION_ORDER_TYPE_1.getCode());
        if (!CollectionUtils.isEmpty(ajStatusMap2) && ajStatusMap2.containsKey(orderEvaluationVO.getOrderType())) {
            orderEvaluationVO.setOrderTypeName(ajStatusMap2.get(orderEvaluationVO.getOrderType()));
        }

        Map<String, String> ajStatusMap3 = systemService.sysUdcGetCodeMap(UdcEnum.RETURN_LEVEL_1.getModel(), UdcEnum.RETURN_LEVEL_1.getCode());
        if (!CollectionUtils.isEmpty(ajStatusMap3) && ajStatusMap3.containsKey(orderEvaluationVO.getReturnLevel())) {
            orderEvaluationVO.setReturnLevelName(ajStatusMap3.get(orderEvaluationVO.getReturnLevel()));
        }
    }
}
