package com.elitesland.yst.production.aftersale.util;

import cn.hutool.core.util.ReflectUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.annotation.ExcelProperty;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.yst.production.aftersale.util.excel.CommonCellStyleStrategy;
import com.elitesland.yst.production.aftersale.util.excel.CustomCellWriteHandler;
import com.elitesland.yst.production.aftersale.util.excel.ExcelExportUtil;
import com.elitesland.yst.production.aftersale.util.excel.support.ExportColumnParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;

import javax.servlet.http.HttpServletResponse;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * @author RYH
 * @date 2023/8/9
 */
@Slf4j
@Validated
public class AsyncExcelUtils {

    /**
     * 构建导出列
     * @param clazz 导出对象class
     */
    public static List<ExportColumnParam> buildExportColumn(Class<?> clazz) {
        List<ExportColumnParam> exportColumn = new ArrayList<>();
        Field[] fields = ReflectUtil.getFields(clazz);
        Arrays.stream(fields).filter(f -> f.isAnnotationPresent(ExcelProperty.class)).forEach(f -> {
            var exportColumnParam = new ExportColumnParam();
            exportColumnParam.setField(f.getName());
            exportColumnParam.setTitle(f.getAnnotation(ExcelProperty.class).value()[0]);
            exportColumn.add(exportColumnParam);
        });
        return exportColumn;
    }

    /**
     * 导出数据
     *
     * @param response        response
     * @param fileName        导出文件名，不含后缀
     * @param columnParamList 导出的数据列
     * @param dataProducer    数据生产者
     * @param queryParam      查询参数
     * @param <T>             参数类型
     * @throws Exception 导出异常
     */
    public static <T extends AbstractOrderQueryParam> void export(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList,
                                                                  Function<T, PagingVO<?>> dataProducer, T queryParam) throws Exception {
        getExportUtilInstance(response, fileName, columnParamList)
                .write("导出数据", (page, pageSize) -> {
                    queryParam.setCurrent(page);
                    queryParam.setSize(pageSize);

                    return dataProducer.apply(queryParam).getRecords();
                });
    }

    /**
     * 导出数据
     *
     * @param response        response
     * @param fileName        导出文件名，不含后缀
     * @param columnParamList 导出的数据列
     * @param dataProducer    数据生产者
     * @param queryParam      查询参数
     * @param otherParam      其它参数
     * @param <T>             参数类型
     * @throws Exception 导出异常
     */
    protected <T extends AbstractOrderQueryParam> void export(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList,
                                                              BiFunction<T, Object[], PagingVO<?>> dataProducer, T queryParam, Object[] otherParam) throws Exception {
        getExportUtilInstance(response, fileName, columnParamList)
                .write("导出数据", (page, pageSize) -> {
                    queryParam.setCurrent(page);
                    queryParam.setSize(pageSize);

                    return dataProducer.apply(queryParam, otherParam).getRecords();
                });
    }

    private static ExcelExportUtil getExportUtilInstance(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList) throws Exception {
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8).replace(" ", "%20");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");

        return ExcelExportUtil.instance(response.getOutputStream())
                .batchSize(100)
                .fields(columnParamList);
    }


    /**
     * 多sheet页批量导出Excel:
     * <p>
     * 重点：存入集合顺序要一一对应(sheetNameList、clazzs、queryParams、dataProducers)
     *
     * @param response      HttpServletResponse
     * @param fileName      导出文件名
     * @param sheetNameList 多Sheet页名称集合
     * @param clazzs        sheet页表头对象/带注解的导出对象一致集合(一个sheet页对象对应一个clazz对象)
     * @param queryParams   分页查询的入参对象集合(一个sheet页对象对应一个分页查询的入参对象)
     * @param dataProducers 数据生产者集合
     */
    public static <T extends AbstractOrderQueryParam> void exportSheetPage(HttpServletResponse response, String fileName,
                                                                           List<String> sheetNameList, List<Class<?>> clazzs,
                                                                           List<T> queryParams, List<Function<T, PagingVO<?>>> dataProducers) {

        if (sheetNameList.size() != clazzs.size() || queryParams.size() != clazzs.size() || queryParams.size() != dataProducers.size()) {
            //throw new ArrayIndexOutOfBoundsException();
            throw new IllegalArgumentException("size参数不合法");
        }

        long startTime = System.currentTimeMillis();
        //log.info("导出开始时间:" + startTime);
        OutputStream outputStream = null;

        try {
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");
            fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8).replace("\\+", "%20");
            response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
            outputStream = response.getOutputStream();

            ExcelWriter writer = EasyExcel.write(outputStream)
                    // 注册通用格式策略
                    .registerWriteHandler(CommonCellStyleStrategy.getHorizontalCellStyleStrategy())
                    // 设置自定义格式策略
                    .registerWriteHandler(new CustomCellWriteHandler())
                    .build();

            //总sheet页数量初始值
            AtomicInteger sheetCount = new AtomicInteger(-1);
            //记录总数:实际中需要根据查询条件进行统计即可
            //Integer totalCount = 3000000;
            //每一个Sheet存放数据量
            Integer sheetDataRows = ConstantsAfterSale.PER_SHEET_ROW_COUNT;
            //每次写入的数据量
            Integer writeDataRows = ConstantsAfterSale.PER_WRITE_ROW_COUNT;

            int i = 0;
            for (String sheetName : sheetNameList) {
                T queryParam = queryParams.get(i);
                Function<T, PagingVO<?>> dataProducer = dataProducers.get(i);
                //记录总数
                Integer totalCount = Long.valueOf(dataProducer.apply(queryParam).getTotal()).intValue();
                if (totalCount != 0){
                    writeSheetPage(sheetCount, totalCount, sheetDataRows, writeDataRows,
                            sheetName, clazzs.get(i), writer, (page, pageSize) -> {
                                queryParam.setCurrent(page);
                                queryParam.setSize(pageSize);
                                return dataProducer.apply(queryParam).getRecords();
                            });
                }else {
                    sheetCount.addAndGet(1);
                    WriteSheet sheet = EasyExcel.writerSheet(sheetCount.intValue(), sheetName).head(clazzs.get(i)).build();
                    writer.write(Collections.emptyList(), sheet);
                }
                ++i;
            }

            writer.finish();

            long endTime = System.currentTimeMillis();
            //log.info("导出结束时间:" + endTime + "ms");
            log.info("导出所用时间:" + (endTime - startTime) / 1000 + "秒");
        } catch (Exception e) {
            throw new BusinessException(e.getMessage());
        }

    }


    /**
     * 分sheet页批量写入Excel
     *
     * @param sheetCount    总sheet页数量初始值
     * @param totalCount    数据量总数
     * @param sheetDataRows 每一个Sheet页存放的条数量
     * @param writeDataRows 每次写入的数据量
     * @param sheetName     sheet页名称
     * @param cls           表头对象/带注解的导出对象一致
     * @param writer        ExcelWriter
     * @param dataProducer  数据生产者
     */
    public static void writeSheetPage(AtomicInteger sheetCount, Integer totalCount, Integer sheetDataRows, Integer writeDataRows,
                                      String sheetName, Class<?> cls, ExcelWriter writer, BiFunction<Integer, Integer, List<?>> dataProducer) {
        //计算需要的Sheet数量
        Integer sheetNum = totalCount % sheetDataRows == 0 ? (totalCount / sheetDataRows) : (totalCount / sheetDataRows + 1);
        //计算一般情况下每一个Sheet需要写入的次数(一般情况不包含最后一个sheet,因为最后一个sheet不确定会写入多少条数据)
        Integer oneSheetWriteCount = sheetDataRows / writeDataRows;
        //计算最后一个sheet需要写入的次数
        //Integer lastSheetWriteCount = totalCount % sheetDataRows == 0 ? oneSheetWriteCount : (totalCount % sheetDataRows % writeDataRows == 0 ? (totalCount / sheetDataRows / writeDataRows) : (totalCount / sheetDataRows / writeDataRows + 1));
        //去掉最后一个sheet页的sheet页数量之和
        Integer beforeSheetNum = sheetNum - 1;
        //去掉最后一个sheet页的数量总和=每页sheet数量*去掉最后一个sheet页的sheet页数量
        Integer beforeTotalCount = beforeSheetNum * sheetDataRows;
        //剩余的数量
        Integer surplusTotalCount = totalCount - beforeTotalCount;
        //最后一个sheet需要写入的次数
        Integer surplusSheetWriteCount = surplusTotalCount % writeDataRows == 0 ? (surplusTotalCount / writeDataRows) : (surplusTotalCount / writeDataRows + 1);
        //计算最后一个sheet需要写入的次数
        Integer lastSheetWriteCount = totalCount % sheetDataRows == 0 ? oneSheetWriteCount : surplusSheetWriteCount;
        //开始分批查询分次写入
        //注意这次的循环就需要进行嵌套循环了,外层循环是Sheet数目,内层循环是写入次数
        List<?> dataList = new ArrayList<>();
        List<?> transitionList = new ArrayList<>();
        for (int i = 0; i < sheetNum; i++) {
            //累加sheet页的数量，比如A1\A2\B1\B2样式
            sheetCount.addAndGet(1);
            //创建Sheet
            WriteSheet sheet = EasyExcel.writerSheet(sheetCount.intValue(), sheetName + (i + 1)).head(cls).build();
            //循环写入次数: j的自增条件是当不是最后一个Sheet的时候写入次数为正常的每个Sheet写入的次数,如果是最后一个就需要使用计算的次数lastSheetWriteCount
            for (int j = 0; j < (i != sheetNum - 1 ? oneSheetWriteCount : lastSheetWriteCount); j++) {
                //集合复用,便于GC清理
                dataList.clear();
                transitionList.clear();
                //分页查询
                dataList = dataProducer.apply(j + 1 + oneSheetWriteCount * i, writeDataRows);
                //把分页查询数据转换成带导出注解的对象
                transitionList = JSON.parseArray(JSON.toJSONString(dataList), cls);
                //写数据
                writer.write(transitionList, sheet);
            }
        }
    }
}
