package com.elitesland.yst.dataturbo.threadpool;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

@Component
@Slf4j
public class CloseableHttpClientResourcePool {

    private final ArrayBlockingQueue<CloseableHttpClient> resourcePool;
    private final int poolSize;
    private final int connectionRequestTimeout;
    private final int connectionTimeout;
    private final int socketTimeout;

    public CloseableHttpClientResourcePool() {
        /**
         * 初始数量
         */
        this.poolSize = 10;
        this.resourcePool = new ArrayBlockingQueue<>(poolSize);
        /**
         * 请求超时时长
         */
        this.connectionRequestTimeout = 30000;
        /**
         * 连接超时时长
         */
        this.connectionTimeout = 30000;
        /**
         * socket超时时长
         */
        this.socketTimeout = 30000;
    }


    public CloseableHttpClient acquireResource(long timeout) throws InterruptedException {
        return resourcePool.poll(timeout, TimeUnit.MILLISECONDS);
    }

    public void releaseResource(CloseableHttpClient resource) {
        if (resource != null) {
            resourcePool.offer(resource);
        }
    }

    @PostConstruct
    public void initialize() {
        for (int i = 0; i < poolSize; i++) {
            PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
            connectionManager.setMaxTotal(poolSize);
            connectionManager.setDefaultMaxPerRoute(poolSize);

            CloseableHttpClient client = HttpClientBuilder.create()
                    .setConnectionManager(connectionManager)
                    .setConnectionManagerShared(true)
                    .setDefaultRequestConfig(RequestConfig.custom()
                            .setConnectionRequestTimeout(connectionRequestTimeout)
                            .setConnectTimeout(connectionTimeout)
                            .setSocketTimeout(socketTimeout)
                            .build())
                    .build();
            resourcePool.offer(client);
        }
        log.info("^_^初始化HTTP线程池成功[{}]", resourcePool.size());
    }
}
