package com.elitesland.yst.production.fin.domain.service.aporder;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitesland.yst.production.fin.application.convert.aporder.ApOrderConvert;
import com.elitesland.yst.production.fin.application.convert.aporder.ApOrderDtlConvert;
import com.elitesland.yst.production.fin.application.convert.aporder.ApOrderDtlGroupConvert;
import com.elitesland.yst.production.fin.common.FinConstant;
import com.elitesland.yst.production.fin.common.UdcEnum;
import com.elitesland.yst.production.fin.domain.entity.aporder.*;
import com.elitesland.yst.production.fin.domain.param.aporder.ApOrderPageParam;
import com.elitesland.yst.production.fin.domain.param.aporder.ApOrderParam;
import com.elitesland.yst.production.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.yst.production.fin.infr.dto.aporder.ApOrderDtlDTO;
import com.elitesland.yst.production.fin.infr.dto.aporder.ApOrderDtlGroupDTO;
import com.elitesland.yst.production.fin.infr.dto.common.ApVerDTO;
import com.elitesland.yst.production.fin.infr.factory.aporder.ApOrderFactory;
import com.elitesland.yst.production.fin.infr.repo.aporder.*;
import com.elitesland.workflow.ProcessInfo;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @package com.elitesland.yst.production.fin.domain.service.aporder
 * @date 2022/3/16 10:45
 */
@Service
@RequiredArgsConstructor
public class ApOrderDomainServiceImpl implements ApOrderDomainService {

    private final ApOrderRepo apOrderRepo;
    private final ApOrderRepoProc apOrderRepoProc;
    private final ApOrderDtlRepoProc apOrderDtlRepoProc;
    private final ApOrderDtlRepo apOrderDtlRepo;
    private final ApOrderFactory apOrderFactory;
    private final ApOrderDtlGroupRepo apOrderDtlGroupRepo;
    private final ApOrderDtlGroupRepoProc apOrderDtlGroupRepoProc;
    //发号器生成付款单号
    private final SeqNumProvider sysNumberRuleService;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ApOrder apOrder) {
        if (apOrder.getVerAmt() == null) {
            // 设置默认
            apOrder.setDef();
        }
        // 单头信息输入的校验
        apOrder.check();
        // 校验明细行
        apOrder.checkDtlList();
        // 手工来源
        if (apOrder.getCreateMode().equals(UdcEnum.FIN_AP_DOC_CLS_MANU.getValueCode())) {
            // 校验明细金额
            apOrder.checkAmt();
            // 校验明细金额/单头/汇总金额
            apOrder.checkAmtSum();
        }
        // 对账单来源
        if (apOrder.getCreateMode().equals(UdcEnum.FIN_AP_DOC_CLS_PACCK.getValueCode())) {
            // 设置默认值
            apOrder.setPacckDef();
            // 进行统计的计算
            apOrder.countByPacck();
            // 汇总列表数据添加
            List<ApOrderDtlGroup> apOrderDtlGroups = this.genDtlGroup(apOrder);
            apOrder.setApOrderDtlGroupList(apOrderDtlGroups);
        }
        apOrder.setOrderState(UdcEnum.COM_APPLY_STATUS_DRAFT.getValueCode());
        // 数据的新增
        ApOrderDO apOrderRes = saveOrUpdate(apOrder);
        return apOrderRes.getId();
    }

    /**
     * 根据明细生成汇总List
     *
     * @return
     */
    public List<ApOrderDtlGroup> genDtlGroup(ApOrder apOrder) {
        // 获取明细的List
        List<ApOrderDtl> res = apOrder.getApOrderDtlList();
        Map<String, List<ApOrderDtl>> collectMaps = res.stream().collect(Collectors.groupingBy(ApOrderDtl::getItemCode));
        List<ApOrderDtlGroup> resList = new ArrayList<>();
        for (Map.Entry<String,List<ApOrderDtl>> entry : collectMaps.entrySet()) {
            ApOrderDtlGroup apOrderDtlGroup = new ApOrderDtlGroup();
            List<ApOrderDtl> apOrderDtlList = entry.getValue();
            // 对集合中元素进行计算
            // 数量的总和
            BigDecimal qtySum = apOrderDtlList.stream().map(ApOrderDtl::getQty).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 税额的总和
            BigDecimal taxAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTaxAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 税额（本位币）taxCurAmt
            BigDecimal taxCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTaxCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 不含税金额 exclTaxAmt
            BigDecimal exclTaxAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getExclTaxAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 不含税金额（本位币）exclTaxCurAmt
            BigDecimal exclTaxCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getExclTaxCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 含税金额 totalAmt
            BigDecimal totalAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 含税金额（本位币）totalCurAmt
            BigDecimal totalCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTotalCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);

            // 拼接返回对象
            apOrderDtlGroup.setItemId(apOrderDtlList.get(0).getItemId());
            apOrderDtlGroup.setItemCode(apOrderDtlList.get(0).getItemCode());
            apOrderDtlGroup.setItemName(apOrderDtlList.get(0).getItemName());
            apOrderDtlGroup.setSmallCateCode(apOrderDtlList.get(0).getSmallCateCode());
            apOrderDtlGroup.setSmallCateName(apOrderDtlList.get(0).getSmallCateName());
            apOrderDtlGroup.setQty(qtySum);
            apOrderDtlGroup.setTotalAmt(totalAmtSum);
            apOrderDtlGroup.setExclTaxAmt(exclTaxAmtSum);
            apOrderDtlGroup.setTaxAmt(taxAmtSum);
            apOrderDtlGroup.setTotalCurAmt(totalCurAmtSum);
            apOrderDtlGroup.setExclTaxCurAmt(exclTaxCurAmtSum);
            apOrderDtlGroup.setTaxCurAmt(taxCurAmtSum);
            resList.add(apOrderDtlGroup);
            // 页面选择参数 (设置费用部门/费用类型)
        }
        return resList;
    }


    /**
     * 对 应付单主表/明细表/汇总明细表
     * 进行数据库新增
     *
     * @param apOrder
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public ApOrderDO saveOrUpdate(ApOrder apOrder) {
        if (CharSequenceUtil.isBlank(apOrder.getApOrderNo())) {
            // 发号器-生成应付单号
            String apOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN,FinConstant.YFD, null);
            apOrder.setApOrderNo(apOrderNo);
        }
        ApOrderDO apOrderDO = ApOrderConvert.INSTANCE.convert(apOrder);
        //新增时锁版本默认值0
        if (apOrderDO.getId() == null) {
            apOrderDO.setAuditDataVersion(0);
        }

        ApOrderDO apOrderRes = apOrderRepo.save(apOrderDO);
        // 应付单明细单的新增
        // 根据应付单id删除明细行
        apOrderDtlRepoProc.delByMasId(Lists.newArrayList(apOrderRes.getId()));
        List<ApOrderDtl> apOrderDtlList = apOrder.getApOrderDtlList();
        List<ApOrderDtlDO> apOrderDtlDOS = ApOrderDtlConvert.INSTANCE.convert(apOrderDtlList);
        apOrderDtlDOS.stream().forEach(x -> {
            x.setMasId(apOrderRes.getId());
            apOrderDtlRepo.save(x);
        });
        // 应付单汇总数据新增
        // 根据应付单id删除汇总行
        apOrderDtlGroupRepoProc.delByMasId(Lists.newArrayList(apOrderRes.getId()));
        List<ApOrderDtlGroup> apOrderDtlGroups = apOrder.getApOrderDtlGroupList();
        if (null != apOrderDtlGroups) {
            List<ApOrderDtlGroupDO> apOrderDtlGroupDOS = ApOrderDtlGroupConvert.INSTANCE.convert(apOrderDtlGroups);
            apOrderDtlGroupDOS.stream().forEach(x -> {
                x.setMasId(apOrderRes.getId());
                apOrderDtlGroupRepo.save(x);
            });
        }
        return apOrderRes;
    }


    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public Long commit(ApOrder apOrder) {
        // 校验单头与明细
        apOrder.check();
        apOrder.checkDtlList();
        if (apOrder.getVerAmt() == null) {
            apOrder.setDef();
        }
        // 校验明细金额
        apOrder.checkAmt();
        // 金额数据校验(单头，明细，汇总)金额的校验
        apOrder.checkAmtSum();
        // 状态为待审批
        apOrder.setOrderState(UdcEnum.COM_APPLY_STATUS_DOING.getValueCode());
        // 进行数据的新增或保存
        ApOrderDO apOrderRes = saveOrUpdate(apOrder);
        return apOrderRes.getId();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        apOrderRepoProc.updateWorkInfo(processInfo, id);
    }


    @Override
    public ApOrderDTO getApOrderAndDtl(Long id) {
        ApOrderDTO apOrderDTO = get(id);
        // 新增明细列表
        List<ApOrderDtlDTO> apOrderDtlDOS = apOrderDtlRepoProc.listByMisId(id);
        // 新增汇总列表
        apOrderDTO.setApOrderDtlDTOList(apOrderDtlDOS);
        List<ApOrderDtlGroupDTO> apOrderDtlGroupDTOS = apOrderDtlGroupRepoProc.listByMisId(id);
        apOrderDTO.setApOrderDtlGroupDTOList(apOrderDtlGroupDTOS);
        return apOrderDTO;
    }


    @Override
    public List<ApOrderDTO> queryByIds(List<Long> ids) {
        return apOrderRepoProc.queryByIds(ids);
    }

    @Override
    public List<ApOrderDTO> getApOrderList(ApOrderParam apOrderParam) {
        List<ApOrderDTO> res = apOrderRepoProc.getApOrderList(apOrderParam);
        return res;
    }

    @Override
    public Boolean queryByApTypeId(Long apTypeId) {
        return apOrderRepoProc.queryByApTypeId(apTypeId);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    @Retryable(value = {BusinessException.class}, maxAttempts = 5, backoff = @Backoff(value = 500))
    public void updateVerAmt(Long id, BigDecimal amt) {
        ApVerDTO apVerDTO = apOrderRepoProc.queryVerAmtById(id);
        if (Objects.isNull(apVerDTO)){
            throw new BusinessException("未查询到应付单信息");
        }
        BigDecimal totalAmt = Objects.isNull(apVerDTO.getTotalAmt()) ? BigDecimal.ZERO : apVerDTO.getTotalAmt();
        if (apVerDTO.getVerAmt().add(amt).compareTo(totalAmt) > 0) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前核销金额大于应付单金额，请刷新页面重新输入");
        }
        apVerDTO.setAmtAndVerState(amt);
        long rows = apOrderRepoProc.updateVerAmt(apVerDTO);
        if (rows == 0) {
            throw new BusinessException("应付单更新已核销金额失败");
        }
    }

    @Override
    public PagingVO<ApOrderDTO> page(ApOrderPageParam apOrderPageParam) {
        PagingVO<ApOrderDTO> res = apOrderFactory.page(apOrderPageParam);
        return res;
    }

    @Override
    public ApOrderDTO get(Long id) {
        ApOrderDTO apOrderDTO = apOrderRepoProc.get(id);
        return apOrderDTO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long audit(List<Long> ids, String content, SysUserDTO user) {
        Long res = apOrderRepoProc.audit(ids, content, user);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long del(List<Long> ids) {

        List<ApOrderDTO> apOrderDTOList = apOrderRepoProc.queryByIds(ids);
        apOrderDTOList.stream().forEach(x -> {
            if (!x.getOrderState().equals(UdcEnum.COM_APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException("非草稿状态不能删除");
            }
        });

        Long res = apOrderRepoProc.del(ids);
        // 删除明细信息
        apOrderDtlRepoProc.delByMasId(ids);
        // 删除汇总信息
        apOrderDtlGroupRepoProc.delByMasId(ids);
        return res;
    }

}
