package com.elitesland.yst.production.fin.application.service.rectype;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.yst.production.fin.application.convert.rectype.RecTypeConvert;
import com.elitesland.yst.production.fin.application.facade.param.rectype.RecTypeSaveParam;
import com.elitesland.yst.production.fin.application.facade.vo.rectype.RecTypeVO;
import com.elitesland.yst.production.fin.domain.entity.rectype.RecType;
import com.elitesland.yst.production.fin.domain.param.rectype.RecTypePageParam;
import com.elitesland.yst.production.fin.domain.service.recorder.RecOrderDomainService;
import com.elitesland.yst.production.fin.domain.service.rectype.RecTypeDomainService;
import com.elitesland.yst.production.fin.domain.service.rectype.RecTypeOuDomainService;
import com.elitesland.yst.production.fin.infr.dto.recorder.RecOrderDTO;
import com.elitesland.yst.production.fin.infr.dto.rectype.RecTypeDTO;
import com.elitesland.yst.production.fin.infr.dto.rectype.RecTypeOuDTO;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/4/11 13:41
 */
@Service
@RequiredArgsConstructor
public class RecTypeServiceImpl implements RecTypeService {

    private final RecTypeDomainService recTypeDomainService;

    private final RecTypeOuDomainService recTypeOuDomainService;

    private final RecOrderDomainService recOrderDomainService;

    @Override
    public ApiResult<PagingVO<RecTypeVO>> page(RecTypePageParam param) {
        PagingVO<RecTypeDTO> page = recTypeDomainService.page(param);
        PagingVO<RecTypeVO> pagingVO = RecTypeConvert.INSTANCE.convertPage(page);
        return ApiResult.ok(pagingVO);
    }

    @Override
    public ApiResult<RecTypeVO> queryById(Long id) {
        RecTypeDTO recTypeDTO = recTypeDomainService.queryById(id);
        RecTypeVO recTypeVO = RecTypeConvert.INSTANCE.dtoToVo(recTypeDTO);
        return ApiResult.ok(recTypeVO);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> makeDefault(Long id) {
        Long aLong = recTypeDomainService.makeDefault(id);
        return ApiResult.ok(aLong);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> delete(List<Long> ids) {
        List<RecTypeOuDTO> recTypeOuDTOS = recTypeOuDomainService.queryByRecTypeIds(ids);
        Map<Long, List<RecTypeOuDTO>> typeOuDTOMap = recTypeOuDTOS.stream().collect(Collectors.groupingBy(RecTypeOuDTO::getRecTypeId));
        Map<Long, List<RecOrderDTO>> orderMap = recOrderDomainService.queryByRecTypeId(ids).stream().collect(Collectors.groupingBy(RecOrderDTO::getRecTypeId));
        for (Map.Entry<Long, List<RecTypeOuDTO>> entry : typeOuDTOMap.entrySet()) {
            if (entry.getValue() != null && CollUtil.isNotEmpty(entry.getValue())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已分配公司!");
            }
        }
        for (Map.Entry<Long, List<RecOrderDTO>> entry : orderMap.entrySet()) {
            if (entry.getValue() != null && CollUtil.isNotEmpty(entry.getValue())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已关联收款单!");
            }
        }
        return ApiResult.ok(recTypeDomainService.delete(ids));
    }

    @Override
    public ApiResult<List<RecTypeVO>> findAll() {
        List<RecTypeDTO> all = recTypeDomainService.findAll();
        List<RecTypeVO> recTypeVOS = RecTypeConvert.INSTANCE.dtoToVo(all);
        return ApiResult.ok(recTypeVOS);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> enable(List<Long> ids) {
        List<RecTypeOuDTO> recTypeOuDTOS = recTypeOuDomainService.queryByRecTypeIds(ids);
        Map<Long, RecTypeDTO> dtoMap = recTypeDomainService.queryByIds(ids).stream().collect(Collectors.toMap(RecTypeDTO::getId, t -> t));
        Map<Long, List<RecTypeOuDTO>> ouMap = recTypeOuDTOS.stream().collect(Collectors.groupingBy(RecTypeOuDTO::getRecTypeId));
        if (CollUtil.isEmpty(ouMap)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未分配公司信息,请分配后再启用");
        } else {
            check(dtoMap, ouMap);
        }
        return ApiResult.ok(recTypeDomainService.enable(ids));
    }

    private static void check(Map<Long, RecTypeDTO> dtoMap, Map<Long, List<RecTypeOuDTO>> ouMap) {
        for (Map.Entry<Long, List<RecTypeOuDTO>> entry : ouMap.entrySet()) {
            if (entry.getValue() != null && dtoMap.get(entry.getKey()) != null && CollUtil.isEmpty(entry.getValue())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "类型代码" + dtoMap.get(entry.getKey()).getRecTypeCode() + "启用前需要分配公司");
            }
            if (dtoMap.get(entry.getKey()) != null && dtoMap.get(entry.getKey()).getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已启用状态不允许再次启用");
            }
        }
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> stopEnable(List<Long> ids) {
        return ApiResult.ok(recTypeDomainService.stopEnable(ids));
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> save(RecTypeSaveParam param) {
        RecType recType = RecTypeConvert.INSTANCE.convert(param);
        Long id = recTypeDomainService.save(recType);
        return ApiResult.ok(id);
    }
}
