package com.elitesland.yst.production.fin.application.service.arorder;

import com.elitesland.yst.production.fin.application.convert.arorder.ArOrderConvert;
import com.elitesland.yst.production.fin.application.facade.param.arorder.ArOrderSaveParam;
import com.elitesland.yst.production.fin.application.facade.vo.arorder.ArOrderVO;
import com.elitesland.yst.production.fin.common.UdcEnum;
import com.elitesland.yst.production.fin.domain.entity.arorder.ArOrder;
import com.elitesland.yst.production.fin.domain.param.arorder.ArOrderPageParam;
import com.elitesland.yst.production.fin.domain.service.arorder.ArOrderDomainService;
import com.elitesland.yst.production.fin.domain.service.arorder.ArOrderDtlDomainService;
import com.elitesland.yst.production.fin.domain.service.artype.ArTypeDomainService;
import com.elitesland.yst.production.fin.rpc.system.SystemRpcService;
import com.elitesland.yst.production.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.yst.production.fin.infr.dto.arorder.ArOrderDTO;
import com.elitesland.yst.production.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.security.util.SecurityUtil;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @package com.elitesland.yst.production.fin.application.service.arorder
 * @date 2022/4/12 15:46
 */
@Service
@RequiredArgsConstructor
public class ArOrderServiceImpl implements ArOrderService {

    private final ArOrderDomainService arOrderDomainService;
    private final ArOrderDtlDomainService arOrderDtlDomainService;
    private final ArTypeDomainService arTypeDomainService;
    private final SystemRpcService systemRpcService;
    private final WorkflowRpcService workflowRpcService;

    @SysCodeProc
    @Override
    public PagingVO<ArOrderVO> page(ArOrderPageParam arOrderPageParam) {
        PagingVO<ArOrderDTO> page = arOrderDomainService.page(arOrderPageParam);
        return ArOrderConvert.INSTANCE.convertPage(page);
    }

    @SysCodeProc
    @Override
    public ArOrderVO get(Long id) {
        ArOrderDTO arOrderDTO = arOrderDomainService.get(id);
        ArOrderVO res = ArOrderConvert.INSTANCE.convert(arOrderDTO);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> audit(Long id) {

        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> refush(List<Long> ids, String content) {

        SysUserDTO user = SecurityUtil.getUser().getUser();
        Long res = arOrderDomainService.audit(ids, content, user);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> del(List<Long> ids) {
        Long res = arOrderDomainService.del(ids);
        return ApiResult.ok(res);
    }

    @SysCodeProc
    @Override
    public ArOrderVO getArOrderAndDtl(Long id) {
        ArOrderDTO arOrderAndDtl = arOrderDomainService.getArOrderAndDtl(id);
        ArOrderVO res = ArOrderConvert.INSTANCE.convert(arOrderAndDtl);
        return res;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> save(ArOrderSaveParam apOrderSaveParam) {
        // 新增只用于单据来源为手工的
        if (!apOrderSaveParam.getCreateMode().equals(UdcEnum.FIN_AR_DOC_CLS_MANU.getValueCode())) {
            throw new BusinessException("新增的单据来源需为手工");
        }
        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Long res = arOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }


    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public ApiResult<Long> commit(ArOrderSaveParam apOrderSaveParam) {
        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Boolean isAuto = arTypeDomainService.queryIsAuto(apOrderSaveParam.getArTypeId());
        if (isAuto) {
            apOrder.setAuditDate(LocalDateTime.now());
            apOrder.setAuditUser("admin");
            SysUserDTO user = systemRpcService.getUserByName("admin");
            apOrder.setAuditUserId(user.getId());
        }
        Long resId = arOrderDomainService.commit(apOrder, isAuto);
        if (!isAuto) {
            // 启动工作流
            startWorkFlow(apOrder, resId);
        }
        return ApiResult.ok(resId);
    }

    private void startWorkFlow(ArOrder apOrder, Long resId) {
        if (apOrder.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(apOrder.getProcInstStatus())) {
            // 启动流程
            String procInstName = "应收单审核-" + apOrder.getArOrderNo();
            String procKey = WorkFlowDefKey.FIN_AR_ORDER.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, resId.toString(), new HashMap<>());
            arOrderDomainService.updateWorkInfo(processInfo, resId);
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> update(ArOrderSaveParam apOrderSaveParam) {

        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        apOrder.checkUpdate();
        Long res = arOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }
}
