package com.elitesland.yst.production.fin.application.service.flow;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.yst.production.fin.application.facade.param.flow.AccountFlowDealerParam;
import com.elitesland.yst.production.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.yst.production.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.yst.production.fin.application.facade.vo.flow.AccountFlowDealerVO;
import com.elitesland.yst.production.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.yst.production.fin.common.FinConstant;
import com.elitesland.yst.production.fin.common.SysNumberGenerator;
import com.elitesland.yst.production.fin.common.UdcEnum;
import com.elitesland.yst.production.fin.repo.flow.AccountFlowRepo;
import com.elitesland.yst.production.fin.repo.flow.AccountFlowRepoProc;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;


/**
 * <p>
 * 功能说明:账户流水
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
public class AccountFlowServiceImpl implements AccountFlowService {
    @Autowired
    private AccountFlowRepo accountFlowRepo;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;

    @Autowired
    private SysNumberGenerator sysNumberGenerator;

    @Autowired
    private AccountFlowZcService accountFlowZcService;
    @Autowired
    private AccountFlowPjService accountFlowPjService;
    @Autowired
    private AccountFlowFjbService accountFlowFjbService;
    @Autowired
    private AccountFlowTcService accountFlowTcService;
    @Autowired
    private AccountFlowCommonService accountFlowCommonService;


    /**
     * 账户流水分页查询：中台端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> page(AccountFlowPageParam pageParam) {
        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 根据数据来源+来源单号查询流水的信息
     * <p>
     * 使用场景1：校验是否有重复数据已经存在
     *
     * @param dataSource 数据来源
     * @param sourceNo   来源单号
     * @return 账户流水信息
     */
    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectBySourceAndNo(String dataSource, String sourceNo) {
        if (StringUtils.isEmpty(dataSource)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据来源为空!");
        }
        if (StringUtils.isEmpty(sourceNo)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "来源单号为空!");
        }
        AccountFlowPageParam accountFlowPageParam = new AccountFlowPageParam();
        accountFlowPageParam.setDataSource(dataSource);
        accountFlowPageParam.setSourceNoEq(sourceNo);
        List<AccountFlowVO> accountFlowVOList = accountFlowRepoProc.selectListByParam(accountFlowPageParam);
        if (CollectionUtils.isEmpty(accountFlowVOList)) {
            return Collections.emptyList();
        }
        return accountFlowVOList;
    }

    /**
     * 账户流水分页查询：经销商端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> dealerPage(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        if (StringUtils.isNotBlank(pageParam.getYearMonthStr())) {
            //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
            LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
            //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
            accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);
        }
        //PC端只展示实际扣减和增加的流水
        pageParam.setAccIoTypeList(Lists.newArrayList(FinConstant.ACC_IO_TYPE_ADD, FinConstant.ACC_IO_TYPE_SUB));
        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.dealerPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 经销商端查询相关统计金额：入账金额、使用金额
     *
     * @param pageParam 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAmt(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加
        //那么标产里就是对应账户金额+-的发生金额，通过交易类型来分区发生金额是扣减还是增加，然后求的发生金额之和

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
        LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
        //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
        accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);

        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setUseAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setRecordAmt(BigDecimal.ZERO);
        //等产品确定根据新的维度进行判断扣减或增加统计
        //获取发生金额为增加的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.ADD_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_ADD);
        List<AccountFlowVO> addAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(addAmtList)) {
            //入账金额
            BigDecimal recordAmt = addAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setRecordAmt(recordAmt);
        }
        //获取发生金额为扣减的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.SUB_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_SUB);
        List<AccountFlowVO> subAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(subAmtList)) {
            //使用金额
            BigDecimal useAmt = subAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setUseAmt(useAmt);
        }

        return accountFlowDealerVO;
    }


    /**
     * 账户流水保存
     *
     * @param paramList 入参
     * @return 账户流水返回对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public AccountFlowVO save(List<AccountFlowParam> paramList) {
        log.info("获取流水入参{}", JSON.toJSONString(paramList));
        //校验必填项
        this.checkMandatoryFieldList(paramList);
        //以防万一，正常不需要这种校验处理
        this.checkRepeatability(paramList);
        //默认赋值，放到校验逻辑之后,目前下放到每个业务里了
        //this.defaultAssignment(paramList);

        this.distinguishAccType(paramList);

        //TODO 占用前都会有这个   校验的接口逻辑
        // 则朋 就是这块不是先释放  再占用吗     就是释放后   还有一个校验的接口检查完了之后看看是否满足  才能占用
        // 校验逻辑：整车下单、整车订单修改需要进行校验
        // 订单的时候需要返回结果和值给他们
        // 金额扣减为负的情况，在校验接口里处理
        // 流水第二条是占用的情况，需要再次反向调用校验接口，只有外部接口里需要


        return new AccountFlowVO();
    }

    /**
     * 按照账户类型区分
     *
     * @param paramList 入参
     */
    private void distinguishAccType(List<AccountFlowParam> paramList) {

        //避免批量穿过来会交叉多种账户类型模块的数据，故每个大模块账户类型都做了过滤区分
        List<AccountFlowParam> accountFlowZcList = paramList.stream().filter(param -> accountFlowCommonService.judgeZcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowZcList)) {
            accountFlowZcService.saveZc(accountFlowZcList);
        }
        List<AccountFlowParam> accountFlowPjList = paramList.stream().filter(param -> accountFlowCommonService.judgePjAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowPjList)) {
            accountFlowPjService.savePj(accountFlowPjList);
        }
        List<AccountFlowParam> accountFlowFjbList = paramList.stream().filter(param -> accountFlowCommonService.judgeFjbAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowFjbList)) {
            accountFlowFjbService.saveFjb(accountFlowFjbList);
        }
        List<AccountFlowParam> accountFlowTcList = paramList.stream().filter(param -> accountFlowCommonService.judgeTcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowTcList)) {
            accountFlowTcService.saveTc(accountFlowTcList);
        }


        //1.按照账户类型拆分成4个大模块
        //2.在每个模块里根据数据来源区分不同的方法，方便后续变动或扩展
        //3.把设计到账户金额、账户占用金额(子账户金额、子账户占用金额、经销商账户金额、经销商占用金额)
        // 查询上一条记录的能共用的抽离出单独的方法块，即使立马项目里无法引用，但是代码可以复用

    }


    /**
     * 默认赋值
     *
     * @param paramList 入参
     */
    /*private void defaultAssignment(List<AccountFlowParam> paramList) {
        paramList.forEach(accountFlowParam -> {
            if (StringUtils.isEmpty(accountFlowParam.getSourcePlatform())) {
                accountFlowParam.setSourcePlatform(UdcEnum.FIN_SOURCE_PLATFORM_TYPE_OTHER.getValueCode());
            }
            String flowNo = sysNumberGenerator.generate(SysNumEnum.FIN_FLOW_NO.getCode());
            accountFlowParam.setFlowNo(flowNo);
        });
    }*/

    /**
     * 校验必填项
     *
     * @param paramList 入参
     */
    @Override
    public void checkMandatoryFieldList(List<AccountFlowParam> paramList) {
        if (CollectionUtils.isEmpty(paramList)) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "数据为空!");
        }
        paramList.forEach(accountFlowParam -> checkMandatoryField(accountFlowParam));
    }

    /**
     * 校验必填项
     *
     * @param param 入参
     */
    @Override
    public void checkMandatoryField(AccountFlowParam param) {
        if (StringUtils.isBlank(param.getAccType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户类型为空!");
        }
        if (StringUtils.isBlank(param.getDataSource())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "数据来源为空!");
        }
        if (StringUtils.isBlank(param.getTransactionType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易类型为空!");
        }
        if (Objects.isNull(param.getAmount())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "发生金额为空!");
        }
        if (Objects.isNull(param.getTransactionTime())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易日期为空!");
        }
        if (StringUtils.isBlank(param.getAccCode())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户编码为空!");
        }
        if (StringUtils.isBlank(param.getAccName())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户名称为空!");
        }
        if (StringUtils.isBlank(param.getSourceNo())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单号为空!");
        }
        if (StringUtils.isBlank(param.getSourceDoc())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单据为空!");
        }
    }

    /**
     * 校验是否有重复数据或脏数据、数据结构是否正确：
     * 必须每次只有一种数据来源，要不数据融合在一起无法判断谁先谁后，会涉及到上一条数据问题。
     * 同种数据来源，也应该是单次的（一个单号），就是不是批量的，会涉及到上一条数据问题。
     * 可以先判断数据来源是否是多种，再判断账户类型+数据来源+交易类型是否是唯一
     *
     * @param paramList 入参
     */
    @Override
    public void checkRepeatability(List<AccountFlowParam> paramList) {
        val dataSourceList = paramList.stream().map(AccountFlowParam::getDataSource).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(dataSourceList) && dataSourceList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "存在多种数据来源!");
        }
        Map<String, List<AccountFlowParam>> accountFlowListMap = paramList.stream().collect(Collectors.groupingBy(
                i -> i.getAccType() + FinConstant.LINE_SPLIT + i.getDataSource() + FinConstant.LINE_SPLIT + i.getTransactionType()));
        for (String accountFlowKey : accountFlowListMap.keySet()) {
            val accountFlowList = accountFlowListMap.get(accountFlowKey);

            String[] keyParams = accountFlowKey.split(FinConstant.LINE_SPLIT);
            String dataSource = keyParams[FinConstant.ARRAY_INDEX_1];
            //数据来源是调剂单审核通过时，交易类型可能是相同的，所以使用拆分类型来区分
            if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TJD03.getValueCode())) {
                List<String> splitTypeList = accountFlowList.stream().map(AccountFlowParam::getSplitType).distinct().collect(Collectors.toList());
                if (!Objects.equals(accountFlowList.size(), splitTypeList.size())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "调剂单数据不唯一，无法处理!");
                }
            } else {
                if (!CollectionUtils.isEmpty(accountFlowList) && accountFlowList.size() > 1) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据不唯一，无法处理!");
                }
            }

        }
    }


    /**
     * 校验必填项02:年月日期、账户编码、账户类型
     *
     * @param pageParam 查询入参
     */
    @Override
    public void checkMandatoryField02(AccountFlowPageParam pageParam) {
        if (StringUtils.isBlank(pageParam.getYearMonthStr())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月日期为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户类型为空!");
        }
    }


    /**
     * 经销商端查询相关统计金额：储值账户金额、返利账户金额
     *
     * @param param 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAccAmt(AccountFlowDealerParam param) {
        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setCzAccAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setFlAccAmt(BigDecimal.ZERO);

        //整车/配件储值账户金额
        Optional<AccountFlowVO> czOptional = accountFlowCommonService.selectByAccCode(param.getCzAccCode());
        czOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setCzAccAmt(accountFlowVO.getAccAmt());
        });

        //整车/配件返利账户金额
        Optional<AccountFlowVO> flOptional = accountFlowCommonService.selectByAccCode(param.getFlAccCode());
        flOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setFlAccAmt(accountFlowVO.getAccAmt());
        });

        return accountFlowDealerVO;
    }

}
