package com.elitesland.yst.production.fin.domain.service.rectype;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.yst.production.fin.application.convert.rectype.RecTypeConvert;
import com.elitesland.yst.production.fin.domain.entity.rectype.RecType;
import com.elitesland.yst.production.fin.domain.entity.rectype.RecTypeDO;
import com.elitesland.yst.production.fin.domain.param.rectype.RecTypePageParam;
import com.elitesland.yst.production.fin.infr.dto.rectype.RecTypeDTO;
import com.elitesland.yst.production.fin.infr.factory.rectype.RecTypeFactory;
import com.elitesland.yst.production.fin.infr.repo.rectype.RecTypeRepo;
import com.elitesland.yst.production.fin.infr.repo.rectype.RecTypeRepoProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;

/**
 * @author zhiyu.he
 * @date 2022/4/11 13:27
 */
@Service
@RequiredArgsConstructor
public class RecTypeDomainServiceImpl implements RecTypeDomainService {

    private final RecTypeRepo reTypeRepo;

    private final RecTypeRepoProc recTypeRepoProc;

    private final RecTypeFactory recTypeFactory;

    @Override
    public PagingVO<RecTypeDTO> page(RecTypePageParam param) {
        return recTypeFactory.payTypePage(param);

    }

    @Override
    public RecTypeDTO queryById(Long id) {
        return recTypeRepoProc.queryById(id);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long makeDefault(Long id) {
        List<RecTypeDTO> dtos = recTypeRepoProc.findByIds(List.of(id));
        dtos.forEach(dto -> {
            if (dto.getDefaultFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已默认状态不能在默认");
            }
        });
        //更新之前默认为false
        recTypeRepoProc.updateDefault();
        //更新选中数据为默认
        return recTypeRepoProc.updateDefaultById(id, true);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> delete(List<Long> id) {
        List<RecTypeDTO> dtos = recTypeRepoProc.findByIds(id);
        dtos.forEach(dto -> {
            if (dto.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除未启用的数据");
            }
            if (dto.getDefaultFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "默认类型不能删除");
            }
        });
        recTypeRepoProc.delete(id);
        return id;
    }

    @Override
    public List<RecTypeDTO> findAll() {
        return recTypeRepoProc.findAll();
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> stopEnable(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选中数据!");
        }
        List<RecTypeDTO> dtos = recTypeRepoProc.findByIds(ids);
        dtos.forEach(dto -> {
            if (!dto.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已停用状态不能在停用");
            }
        });
        recTypeRepoProc.updateEnable(ids, false);
        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> enable(List<Long> ids) {
        recTypeRepoProc.updateEnable(ids, true);
        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(RecType recType) {
        recType.checkNotNull();
        if (recType.getId() == null) {
            Boolean existCode = recTypeRepoProc.existRecCode(recType.getRecTypeCode());
            if (existCode) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "收款单类型编码已存在");
            }
            Boolean existName = recTypeRepoProc.existRecName(recType.getRecTypeName());
            if (existName) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "收款单类型名称已存在");
            }
            //数据第一条永远保持是默认
            if (recTypeRepoProc.isFirst()) {
                recType.setDefault(false);
            } else {
                recType.setDefault(true);
            }
        }
        RecTypeDO typeDO = RecTypeConvert.INSTANCE.convert(recType);
        RecTypeDO typeDO1 = reTypeRepo.save(typeDO);
        return typeDO1.getId();
    }

    @Override
    public RecTypeDTO defaultValue() {
        return recTypeRepoProc.defaultValue();
    }

    @Override
    public Boolean getAuto(Long id) {
        return recTypeRepoProc.queryIsAutoAudit(id);
    }

    @Override
    public List<RecTypeDTO> queryByIds(Collection<Long> ids) {
        return  recTypeRepoProc.findByIds(ids);
    }
}
