package com.elitesland.yst.production.fin.repo.account;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.yst.production.fin.application.facade.param.account.AccountPageParam;
import com.elitesland.yst.production.fin.application.facade.param.account.AccountParam;
import com.elitesland.yst.production.fin.application.facade.vo.account.AccountVO;
import com.elitesland.yst.production.fin.common.UdcEnum;
import com.elitesland.yst.production.fin.entity.account.QAccountDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @package com.elitesland.yst.production.fin.repo
 * @date 2023/2/16 15:00
 */
@Component
@RequiredArgsConstructor
public class AccountRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private final QAccountDO qAccountDO = QAccountDO.accountDO;


    public PagingVO<AccountVO> page(AccountPageParam accountPageParam) {

        List<Predicate> predicates = new ArrayList<>();
        if (StringUtils.isNotBlank(accountPageParam.getAccType())) {
            predicates.add(qAccountDO.accType.eq(accountPageParam.getAccType()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getAccName())) {
            predicates.add(qAccountDO.accName.like('%' + accountPageParam.getAccName() + '%'));
        }
        if (StringUtils.isNotBlank(accountPageParam.getState())) {
            predicates.add(qAccountDO.state.eq(accountPageParam.getState()));
        }

        JPAQuery<AccountVO> query = select(AccountVO.class).where(ExpressionUtils.allOf(predicates));
        accountPageParam.setPaging(query);
        accountPageParam.fillOrders(query, qAccountDO);
        List<AccountVO> fetch = query.fetch();
        fetch.forEach(x -> {
            x.setAccAvaAmt(x.getAccAmt().subtract(x.getAccOccAmt()));
        });
        return PagingVO.<AccountVO>builder()
                .total(query.fetchCount())
                .records(fetch)
                .build();
    }

    public List<AccountVO> queryByAccounts(List<String> accounts, String type, String state) {
        List<Predicate> predicates = new ArrayList<>();
        if (!StringUtils.isEmpty(type)) {
            predicates.add(qAccountDO.accType.eq(type));
        }
        if (!CollectionUtils.isEmpty(accounts)) {
            predicates.add(qAccountDO.accCode.in(accounts));
        }
        if (!StringUtils.isEmpty(state)) {
            predicates.add(qAccountDO.state.eq(state));
        }
        return select(AccountVO.class)
                .where(ExpressionUtils.allOf(predicates))
                .fetch();
    }

    public Boolean updateAmtByCode(AccountParam accountParam) {
        Long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accAmt, accountParam.getAccAmt())
                .set(qAccountDO.accOccAmt, accountParam.getAccOccAmt())
                .where(qAccountDO.accCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qAccountDO.id,
                qAccountDO.accName,
                qAccountDO.accCode,
                qAccountDO.accNameCode,
                qAccountDO.accType,
                qAccountDO.state,
                qAccountDO.createTime,
                qAccountDO.creator,
                qAccountDO.ouName,
                qAccountDO.accOccAmt,
                qAccountDO.accAmt.subtract(qAccountDO.accOccAmt).as("accAvaAmt"),
                qAccountDO.accAmt,
                qAccountDO.ouId,
                qAccountDO.ouCode,
                qAccountDO.area,
                qAccountDO.areaName,
                qAccountDO.areaId

                )
        ).from(qAccountDO);
    }

    public Long updateState(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.state, accountParam.getState())
                .where(qAccountDO.id.in(accountParam.getIds()))
                .execute();
        return res;
    }

    public long existAccName(String accName) {
        long res = jpaQueryFactory.select(qAccountDO.id)
                .from(qAccountDO)
                .where(qAccountDO.accName.eq(accName))
                .fetchCount();
        return res;
    }

    public AccountVO get(Long id) {
        AccountVO accountVO = select(AccountVO.class)
                .where(qAccountDO.id.eq(id))
                .fetchOne();
        return accountVO;
    }

    public AccountVO getByCode(String accCode) {
        AccountVO res = select(AccountVO.class)
                .where(qAccountDO.accCode.eq(accCode))
                .where(qAccountDO.state.eq(UdcEnum.FIN_ENABLE_STATE_ENABLE.getValueCode()))
                .fetchOne();
        return res;
    }

    public Boolean updateOccAmtByCode(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accOccAmt, accountParam.getAccOccAmt())
                .where(qAccountDO.accCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;

    }

    public Boolean updateAccAmtByCode(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accAmt, accountParam.getAccAmt())
                .where(qAccountDO.accCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;

    }

    public Long existAccNameCode(String accNameCode) {
        long res = jpaQueryFactory.select(qAccountDO.id)
                .from(qAccountDO)
                .where(qAccountDO.accNameCode.eq(accNameCode))
                .fetchCount();
        return res;
    }
}
