package com.elitesland.yst.production.inv.utils;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.groups.Default;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

/**
 * <p>
 * 校验工具类
 * </p >
 *
 * @author ssy
 * @date 2020/8/5
 */
@Slf4j
public abstract class ValidationUtil {

    public static Validator validator = Validation.buildDefaultValidatorFactory().getValidator();

    /**
     * 校验实体(所有带有校验注解的属性，exp: @NoNull,@Email...)
     *
     * @param obj 校验实体
     * @param <T> 实体类型
     * @return obj 校验结果
     */
    public static <T> ValidResult validate(T obj) {
        Set<ConstraintViolation<T>> validates = validator.validate(obj, Default.class);
        if (!CollectionUtils.isEmpty(validates)) {
            Map<String, String> errorMsgs = new HashMap<>(validates.size());
            validates.forEach(cv -> errorMsgs.put(cv.getPropertyPath().toString(), cv.getMessage()));
            return new ValidResult(Boolean.TRUE, errorMsgs);
        }
        return ValidResult.noError();
    }

    /**
     * 自定义处理函数
     *
     * @param obj      校验实体
     * @param consumer 自定义接口执行
     * @param <T>      实体类型
     */
    public static <T> void validate(T obj, Consumer<ValidResult> consumer) {
        ValidResult validate = validate(obj);
        if (null != consumer) {
            consumer.accept(validate);
        }
    }

    /**
     * 校验实体某些属性
     *
     * @param obj        校验实体
     * @param properties 需要校验的实体属性
     * @param <T>        实体类型
     * @return obj       校验结果
     */
    public static <T> ValidResult validateProperty(T obj, String... properties) {
        Map<String, String> errorMsgs = new HashMap<>();
        if (null != properties) {
            Arrays.asList(properties).forEach(propertyName -> {
                Set<ConstraintViolation<T>> validates = validator.validateProperty(obj, propertyName, Default.class);
                if (!CollectionUtils.isEmpty(validates)) {
                    validates.forEach(cv -> errorMsgs.put(cv.getPropertyPath().toString(), cv.getMessage()));
                }
            });
        }
        if (errorMsgs.size() != 0) {
            return new ValidResult(Boolean.TRUE, errorMsgs);
        }
        return ValidResult.noError();
    }

    @Getter
    public static class ValidResult {
        private boolean error;
        private Map<String, String> errorMsgs;

        ValidResult(boolean error, Map<String, String> errorMsgs) {
            this.error = error;
            this.errorMsgs = errorMsgs;
        }

        static ValidResult noError() {
            return new ValidResult(Boolean.FALSE, null);
        }
    }

}
