package com.elitesland.yst.production.inv.application.service.impl;

import com.elitesland.yst.production.inv.application.facade.vo.SelectOptionParam;
import com.elitesland.yst.production.inv.application.facade.vo.SelectOptionVO;
import com.elitesland.yst.production.inv.application.facade.vo.whAreaSetting.InvWhAreaSettingParamVO;
import com.elitesland.yst.production.inv.application.facade.vo.whAreaSetting.InvWhAreaSettingRespVO;
import com.elitesland.yst.production.inv.application.facade.vo.whAreaSetting.InvWhAreaSettingSaveVO;
import com.elitesland.yst.production.inv.application.out.SystemService;
import com.elitesland.yst.production.inv.application.service.InvWhAreaSettingService;
import com.elitesland.yst.production.inv.domain.convert.InvWhAreaSettingConvert;
import com.elitesland.yst.production.inv.domain.entity.whAreaSetting.InvWhAreaSetting;
import com.elitesland.yst.production.inv.domain.service.InvWhAreaSettingDomainService;
import com.elitesland.yst.production.inv.infr.dto.InvWhAreaSettingDTO;
import com.elitesland.yst.production.inv.utils.InvUdcEnum;
import com.elitesland.yst.production.inv.utils.UdcEnum;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.MapUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:功能区配置
 * </p>
 *
 * @Author Darren
 * @Date 2022/04/19
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service("InvWhAreaSettingService")
@AllArgsConstructor
public class InvWhAreaSettingServiceImpl implements InvWhAreaSettingService {


    private final InvWhAreaSettingDomainService invWhAreaSettingDomainService;

    private final SystemService systemService;


    /**
     * 新增功能区配置数据
     *
     * @param saveVo 新增功能区配置数据
     * @return 新增功能区配置ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> createOne(InvWhAreaSettingSaveVO saveVo) {

        //处理仓库编码、可多选、逗号分隔存储。
        saveVo.setWhType(transitionWhTypeCode(saveVo.getWhTypes()));
        InvWhAreaSetting invWhAreaSetting = InvWhAreaSettingConvert.INSTANCE.saveVoToEn(saveVo);
        //新增时默认启用
        invWhAreaSetting.setStatus(InvUdcEnum.INV_ACTIVE_STATUS_ACTIVE.getValueCode());
        //检查必填项
        checkMandatory(invWhAreaSetting);
        //校验编码和名称是否已存在
        checkDeter2AndDeter2Name(invWhAreaSetting);
        //保存
        Long id = invWhAreaSettingDomainService.saveInvWhAreaSetting(invWhAreaSetting, Optional.empty());

        return ApiResult.ok(id);
    }

    /**
     * 修改功能区配置数据
     *
     * @param saveVo 修改功能区配置对象
     * @return 功能区配置ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> update(InvWhAreaSettingSaveVO saveVo) {

        val whAreaSetOptional = invWhAreaSettingDomainService.findById(saveVo.getId());
        if (whAreaSetOptional.isEmpty()) {
            return ApiResult.fail("未查询到功能区配置信息!");
        }
        //处理仓库编码、可多选、逗号分隔存储。
        saveVo.setWhType(transitionWhTypeCode(saveVo.getWhTypes()));
        InvWhAreaSetting invWhAreaSetting = InvWhAreaSettingConvert.INSTANCE.saveVoToEn(saveVo);
        //检查必填项
        checkMandatory(invWhAreaSetting);
        //校验编码和名称是否已存在
        checkDeter2AndDeter2Name(invWhAreaSetting);
        //保存
        Long id = invWhAreaSettingDomainService.saveInvWhAreaSetting(invWhAreaSetting, whAreaSetOptional);

        return ApiResult.ok(id);
    }


    /**
     * 检查必填项
     *
     * @param invWhAreaSetting 功能区配置对象
     */
    private void checkMandatory(InvWhAreaSetting invWhAreaSetting) {
        if (! StringUtils.hasLength(invWhAreaSetting.getDeter2Type())) {
            throw new BusinessException(ApiCode.FAIL, "功能区类型编码不能为空!");
        }
        if (! StringUtils.hasLength(invWhAreaSetting.getDeter2TypeName())) {
            throw new BusinessException(ApiCode.FAIL, "功能区类型名称不能为空!");
        }
        if (Objects.isNull(invWhAreaSetting.getBusinessChoose())) {
            throw new BusinessException(ApiCode.FAIL, "建仓前端是否可选不能为空!");
        }
        if (Objects.isNull(invWhAreaSetting.getIsZeroWh())) {
            throw new BusinessException(ApiCode.FAIL, "是否零价库不能为空!");
        }
        if (! StringUtils.hasLength(invWhAreaSetting.getStatus())) {
            throw new BusinessException(ApiCode.FAIL, "是否启用不能为空!");
        }
        if (! StringUtils.hasLength(invWhAreaSetting.getWhType())) {
            throw new BusinessException(ApiCode.FAIL, "可使用仓库类型编码不能为空!");
        }

    }

    /**
     * 校验编码和名称是否已存在
     *
     * @param whAreaSet 功能区配置对象
     */
    private void checkDeter2AndDeter2Name(InvWhAreaSetting whAreaSet) {
        InvWhAreaSettingParamVO param = new InvWhAreaSettingParamVO();
        param.setDeter2Type(whAreaSet.getDeter2Type());
        List<InvWhAreaSettingDTO> whAreaSetList = invWhAreaSettingDomainService.selectByParam(param);
        val whAreaSetOptional = whAreaSetList.stream().filter(dto ->
                Objects.equals(whAreaSet.getDeter2Type(), dto.getDeter2Type()) && !Objects.equals(whAreaSet.getId(), dto.getId())
        ).findFirst();
        whAreaSetOptional.ifPresent(whAreaSetDTO -> {
            throw new BusinessException(ApiCode.FAIL, "功能区类型编码已存在-(" + whAreaSetDTO.getDeter2Type() + ")!");
        });
        param.setDeter2Type(null);
        param.setDeter2TypeName(whAreaSet.getDeter2TypeName());
        List<InvWhAreaSettingDTO> whAreaSetNameList = invWhAreaSettingDomainService.selectByParam(param);
        val whAreaSetNameOptional = whAreaSetNameList.stream().filter(dto ->
                Objects.equals(whAreaSet.getDeter2TypeName(), dto.getDeter2TypeName()) && !Objects.equals(whAreaSet.getId(), dto.getId())
        ).findFirst();
        whAreaSetNameOptional.ifPresent(whAreaSetDTO -> {
            throw new BusinessException(ApiCode.FAIL, "功能区类型名称已存在-(" + whAreaSetDTO.getDeter2TypeName() + ")!");
        });
    }


    /**
     * 通过功能区配置ID查询单条数据
     *
     * @param id 功能区配置ID
     * @return 单条功能区配置数据
     */
    @Override
    @SysCodeProc
    public ApiResult<InvWhAreaSettingRespVO> findIdOne(Long id) {
        val whAreaSetOptional = invWhAreaSettingDomainService.findById(id);
        if (whAreaSetOptional.isEmpty()) {
            return ApiResult.fail("未查询到功能区配置信息!");
        }
        InvWhAreaSettingDTO invWhAreaSettingDTO = whAreaSetOptional.get();
        //转换可使用仓库类型编码并取名称
        invWhAreaSettingDTO.setWhTypes(transitionWhTypeCodeList(invWhAreaSettingDTO.getWhType()));
        transitionWhTypeCodeName(Collections.singletonList(invWhAreaSettingDTO));
        //转换RespVO
        InvWhAreaSettingRespVO invWhAreaSettingRespVO = InvWhAreaSettingConvert.INSTANCE.dtoToRespVo(invWhAreaSettingDTO);

        return ApiResult.ok(invWhAreaSettingRespVO);
    }

    /**
     * 通过功能区编码查询单条数据
     *
     * @param deter2 功能区编码
     * @return 单条功能区配置数据
     */
    @Override
    @SysCodeProc
    public Optional<InvWhAreaSettingRespVO> findByDeter2(String deter2) {
        val whAreaSetOptional = invWhAreaSettingDomainService.findByDeter2(deter2);
        if (whAreaSetOptional.isEmpty()) {
            return Optional.empty();
        }
        InvWhAreaSettingDTO invWhAreaSettingDTO = whAreaSetOptional.get();
        //转换可使用仓库类型编码并取名称
        invWhAreaSettingDTO.setWhTypes(transitionWhTypeCodeList(invWhAreaSettingDTO.getWhType()));
        transitionWhTypeCodeName(Collections.singletonList(invWhAreaSettingDTO));
        //转换RespVO
        InvWhAreaSettingRespVO invWhAreaSettingRespVO = InvWhAreaSettingConvert.INSTANCE.dtoToRespVo(invWhAreaSettingDTO);

        return Optional.of(invWhAreaSettingRespVO);
    }

    @Override
    public List<InvWhAreaSettingRespVO> findByDeter2s(List<String> deter2s) {
        List<InvWhAreaSettingDTO> invWhAreaSettingDTOS = invWhAreaSettingDomainService.findByDeter2Batch(deter2s);
        if(CollectionUtils.isEmpty(invWhAreaSettingDTOS)){
            return Collections.EMPTY_LIST;
        }
        List<InvWhAreaSettingRespVO> whAreaSettingRespVOS = invWhAreaSettingDTOS.stream().map(InvWhAreaSettingConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());
        return whAreaSettingRespVOS;
    }

    @Override
    public List<InvWhAreaSettingRespVO> findByDeter2Types(List<String> deter2Types) {
        return invWhAreaSettingDomainService.findByDeter2Types(deter2Types);
    }

    /**
     * 通过功能区编码集合查询数据
     *
     * @param deter2List 功能区编码集合
     * @return 功能区配置数据集合
     */
    @Override
    @SysCodeProc
    public List<InvWhAreaSettingRespVO> findByDeter2Batch(List<String> deter2List) {
        List<InvWhAreaSettingDTO> invWhAreaSettingDTOList = invWhAreaSettingDomainService.findByDeter2Batch(deter2List);
        if (CollectionUtils.isEmpty(invWhAreaSettingDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换可使用仓库类型编码并取名称
        invWhAreaSettingDTOList.forEach(invWhAreaSettingDTO -> {
            invWhAreaSettingDTO.setWhTypes(transitionWhTypeCodeList(invWhAreaSettingDTO.getWhType()));
        });
        transitionWhTypeCodeName(invWhAreaSettingDTOList);
        //转换RespVO
        List<InvWhAreaSettingRespVO> whAreaSettingRespVOList = invWhAreaSettingDTOList.stream().map(
                InvWhAreaSettingConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return whAreaSettingRespVOList;
    }

    /**
     * 分页查询功能区配置
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条功能区配置数据
     */
    @Override
    @SysCodeProc
    public PagingVO<InvWhAreaSettingRespVO> search(InvWhAreaSettingParamVO param) {

        //可使用仓库类型编码转换成字符串
        val pagingVo = invWhAreaSettingDomainService.search(param);
        if (CollectionUtils.isEmpty(pagingVo.getRecords())) {
            return PagingVO.<InvWhAreaSettingRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        val vos = pagingVo.getRecords();

        //转换可使用仓库类型编码并取名称
        vos.forEach(invWhAreaSettingDTO -> {
            invWhAreaSettingDTO.setWhTypes(transitionWhTypeCodeList(invWhAreaSettingDTO.getWhType()));
        });
        transitionWhTypeCodeName(vos);
        //转换RespVO
        List<InvWhAreaSettingRespVO> whAreaSettingRespVOList = vos.stream().map(
                InvWhAreaSettingConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());


        return PagingVO.<InvWhAreaSettingRespVO>builder()
                .total(pagingVo.getTotal())
                .records(whAreaSettingRespVOList)
                .build();
    }


    /**
     * 根据功能区配置ID批量逻辑删除
     *
     * @param ids 功能区配置ID集合
     * @return 功能区配置ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> deleteBatch(List<Long> ids) {

        invWhAreaSettingDomainService.updateDeleteFlagByIds(1, ids);
        return ApiResult.ok();
    }

    /**
     * 转换成字符串：处理仓库编码、可多选、逗号分隔存储。
     *
     * @param whTypeCodes 可使用仓库类型编码
     */
    private String transitionWhTypeCode(List<String> whTypeCodes) {

        if (CollectionUtils.isEmpty(whTypeCodes)) {
            return null;
        }
        String whTypeCode = String.join(",", whTypeCodes);
        return whTypeCode;
    }


    /**
     * 转换可使用仓库类型编码并取名称
     *
     * @param whAreaSetDtoList 功能区配置
     */
    private void transitionWhTypeCodeName(List<InvWhAreaSettingDTO> whAreaSetDtoList) {
        Map<String, String> map = systemService.sysUdcGetCodeMap(InvUdcEnum.INV_WH_TYPE_1.getModel(), InvUdcEnum.INV_WH_TYPE_1.getCode());
        whAreaSetDtoList.forEach(invWhAreaSettingDTO -> {
            List<String> list = new ArrayList<>();
            if (!CollectionUtils.isEmpty(invWhAreaSettingDTO.getWhTypes()) && !MapUtils.isEmpty(map)) {
                invWhAreaSettingDTO.getWhTypes().forEach(str ->
                        list.add(map.get(str))
                );
            }
            String whTypeCodeName = list.stream().map(String::valueOf).collect(Collectors.joining(","));
            invWhAreaSettingDTO.setWhTypeName(whTypeCodeName);
            //转换是否零价库、建仓前端是否可选的名称
            transitionName(invWhAreaSettingDTO);
        });
    }

    /**
     * 转换成集合：处理仓库编码、可多选、逗号分隔存储。
     *
     * @param whTypeCode 可使用仓库类型编码
     */
    private List<String> transitionWhTypeCodeList(String whTypeCode) {

        if (StringUtils.isEmpty(whTypeCode)) {
            return Collections.EMPTY_LIST;
        }

        List<String> list = Arrays.asList(whTypeCode.split(","));

        return list;
    }

    /**
     * 功能描述  查询生效的数据
     *
     * @param
     * @return
     * @author Brian.lai
     * @date 2021/7/6
     */
    @Override
    @SysCodeProc
    public List<InvWhAreaSettingRespVO> findAllActives(InvWhAreaSettingParamVO paramVO) {

        if (StringUtils.isEmpty(paramVO.getStatus())) {
            throw new BusinessException(ApiCode.FAIL, "请传入是否启用参数");
        }
        List<InvWhAreaSettingDTO> invWhAreaSettingDTOS = invWhAreaSettingDomainService.selectByParam(paramVO);
        if (CollectionUtils.isEmpty(invWhAreaSettingDTOS)) {
            return Collections.EMPTY_LIST;
        }
        //合作伙伴类型
        Map<String, String> pTypeMap = systemService.sysUdcGetCodeMap(UdcEnum.INV_PARTNER_TYPE_CUST.getModel(), UdcEnum.INV_PARTNER_TYPE_CUST.getCode());

        //转换可使用仓库类型编码并取名称
        invWhAreaSettingDTOS.forEach(invWhAreaSettingDTO -> {
            invWhAreaSettingDTO.setWhTypes(transitionWhTypeCodeList(invWhAreaSettingDTO.getWhType()));
        });
        transitionWhTypeCodeName(invWhAreaSettingDTOS);
        //转换RespVO
        List<InvWhAreaSettingRespVO> whAreaSettingRespVOList = invWhAreaSettingDTOS.stream().map(
                InvWhAreaSettingConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());
        return whAreaSettingRespVOList;
    }

    @Override
    public List<SelectOptionVO> listDeter2Type(SelectOptionParam select) {
        return invWhAreaSettingDomainService.listDeter2Type(select);
    }


    /**
     * 功能描述  转换是否零价库、建仓前端是否可选的名称
     *
     * @param whAreaSetDTO 功能区DTO
     */
    private void transitionName(InvWhAreaSettingDTO whAreaSetDTO) {

        if (!StringUtils.isEmpty(whAreaSetDTO.getBusinessChoose())) {
            String businessChooseName = whAreaSetDTO.getBusinessChoose() ? InvUdcEnum.INV_BUSINESS_CHOOSE_Y.getValueCodeName()
                    : InvUdcEnum.INV_BUSINESS_CHOOSE_N.getValueCodeName();
            whAreaSetDTO.setBusinessChooseName(businessChooseName);
        }
        if (!StringUtils.isEmpty(whAreaSetDTO.getIsZeroWh())) {
            String isZeroWhName = whAreaSetDTO.getIsZeroWh() ? InvUdcEnum.INV_ZERO_WH_Y.getValueCodeName()
                    : InvUdcEnum.INV_ZERO_WH_N.getValueCodeName();
            whAreaSetDTO.setIsZeroWhName(isZeroWhName);
        }
    }

    /**
     * 功能描述  批量转换是否零价库、建仓前端是否可选的名称
     *
     * @param whAreaSetDtoList 功能区DTO
     */
    private void transitionNameBatch(List<InvWhAreaSettingDTO> whAreaSetDtoList) {
        whAreaSetDtoList.forEach(whAreaSetDTO -> {
            transitionName(whAreaSetDTO);
        });

    }

}
