package com.elitesland.yst.production.inv.application.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.elitesland.yst.production.inv.application.facade.vo.scene.InvSceneConfigPageVO;
import com.elitesland.yst.production.inv.application.facade.vo.scene.InvSceneConfigQueryVO;
import com.elitesland.yst.production.inv.application.facade.vo.scene.param.InvSceneConfigQueryParam;
import com.elitesland.yst.production.inv.application.facade.vo.scene.param.InvSceneConfigSaveParam;
import com.elitesland.yst.production.inv.application.facade.vo.whAreaSetting.InvWhAreaSettingRespVO;
import com.elitesland.yst.production.inv.application.out.SystemService;
import com.elitesland.yst.production.inv.application.service.InvSceneConfigBizService;
import com.elitesland.yst.production.inv.application.service.InvWhAreaSettingService;
import com.elitesland.yst.production.inv.constants.InvRedisConstant;
import com.elitesland.yst.production.inv.domain.convert.scene.InvSceneConfigConvert;
import com.elitesland.yst.production.inv.domain.convert.scene.InvSceneConfigDtlConvert;
import com.elitesland.yst.production.inv.domain.entity.scene.InvSceneConfigDO;
import com.elitesland.yst.production.inv.domain.entity.scene.InvSceneConfigDtlDO;
import com.elitesland.yst.production.inv.domain.service.InvIocDomainService;
import com.elitesland.yst.production.inv.domain.service.InvSceneConfigDomainService;
import com.elitesland.yst.production.inv.domain.service.InvSceneConfigDtlDomainService;
import com.elitesland.yst.production.inv.infr.dto.InvIocDTO;
import com.elitesland.yst.production.inv.infr.dto.InvSceneConfigDTO;
import com.elitesland.yst.production.inv.infr.dto.InvSceneConfigDtlDTO;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.yst.production.inv.infr.repo.scene.InvSceneConfigDtlRepo;
import com.elitesland.yst.production.inv.infr.repo.scene.InvSceneConfigRepo;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.data.domain.Example;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Tom.su
 * @program yst-inv
 * @description
 * @date 2022/04/20 18:01
 */
@Slf4j
@Service("invSceneConfigBizService")
@AllArgsConstructor
@Transactional(propagation = Propagation.REQUIRED,rollbackFor = Exception.class)
public class InvSceneConfigBizServiceImpl implements InvSceneConfigBizService {

    private final InvSceneConfigDomainService invSceneConfigService;

    private final InvSceneConfigDtlDomainService invSceneConfigDtlDomainService;

    private final InvIocDomainService invIocDomainService;

    private final RedissonClient redissonClient;

    private final InvSceneConfigRepo invSceneConfigRepo;

    private final InvSceneConfigDtlRepo invSceneConfigDtlRepo;

    private final SystemService systemService;

    private final InvWhAreaSettingService invWhAreaSettingService;

    @Override
    @SysCodeProc
    public PagingVO<InvSceneConfigPageVO> searchPage(InvSceneConfigQueryParam param) {
        PagingVO<InvSceneConfigPageVO> invSceneConfigPageVOPagingVO = invSceneConfigService.searchPage(param);
        if(CollectionUtil.isNotEmpty(invSceneConfigPageVOPagingVO.getRecords())){
            convertPage(invSceneConfigPageVOPagingVO.getRecords());
        }
        return invSceneConfigPageVOPagingVO;
    }

    @Override
    public Long saveOrUpdate(InvSceneConfigSaveParam param) {
        InvSceneConfigDO dbSceneConfig = validateParam(param);
        return persistence(param,dbSceneConfig);
    }

    private Long  persistence(InvSceneConfigSaveParam param, InvSceneConfigDO dbSceneConfig) {
        if(dbSceneConfig == null){
            return saveConfigAndDtl(param);
        }
        removeCache(dbSceneConfig.getSceneCode());
        return updateConfigAndDtl(param);
    }
    private Long updateConfigAndDtl(InvSceneConfigSaveParam param) {
        InvSceneConfigDO invSceneConfigDO = InvSceneConfigConvert.INSTANCE.param2DO(param);
        List<InvSceneConfigDtlDO> invSceneConfigDtlDOS = InvSceneConfigDtlConvert.INSTANCE.param2DOList(param.getInvSceneConfigDtlList());
        invSceneConfigDtlDOS.stream().forEach(dtl -> dtl.setMasId(invSceneConfigDO.getId()));
        invSceneConfigService.updateDynamically(invSceneConfigDO);
        invSceneConfigDtlDomainService.deleteByCondition(getQueryBeanByMasId(invSceneConfigDO.getId()));
        invSceneConfigDtlDomainService.saveAll(invSceneConfigDtlDOS);
        return invSceneConfigDO.getId();
    }

    private Long saveConfigAndDtl(InvSceneConfigSaveParam param) {
        InvSceneConfigDO invSceneConfigDO = InvSceneConfigConvert.INSTANCE.param2DO(param);
        List<InvSceneConfigDtlDO> invSceneConfigDtlDOS = InvSceneConfigDtlConvert.INSTANCE.param2DOList(param.getInvSceneConfigDtlList());
        invSceneConfigRepo.save(invSceneConfigDO);
        invSceneConfigDtlDOS.forEach(dtl -> dtl.setMasId(invSceneConfigDO.getId()));
        invSceneConfigDtlRepo.saveAll(invSceneConfigDtlDOS);
        return invSceneConfigDO.getId();
    }

    private InvSceneConfigDO validateParam(InvSceneConfigSaveParam param) {
        InvSceneConfigDO dbSceneConfig = null;
        if(CollectionUtil.isEmpty(param.getInvSceneConfigDtlList())){
            throw new BusinessException("库存场景明细不能为空");
        }
        param.setLine();
        if(param.getId() != null){
            Optional<InvSceneConfigDO> invSceneConfigDOOptional = invSceneConfigService.findById(param.getId());
            if (invSceneConfigDOOptional.isPresent()){
                dbSceneConfig = invSceneConfigDOOptional.get();
            }
        }
        //不存在，插入操作
        if (dbSceneConfig == null) {
            //持久化基础校验
            validateParamBySave(param);
        }else{
            validateParamByUpdate(param);
        }
        List<InvSceneConfigQueryVO> sceneCode = invSceneConfigService.findSceneConfiBySceneCode(param.getSceneCode());
        if(!CollectionUtils.isEmpty(sceneCode)){
            long count = sceneCode.stream().filter(i -> !i.getId().equals(param.getId())).count();
            if(count > 0){
                throw new BusinessException("库存场景编码已存在，不能重复");
            }
        }

        return dbSceneConfig;
    }

    private void validateParamByUpdate(InvSceneConfigSaveParam param) {
        param.validateHeadByUpdate(param);
        param.validateBodyBySave(param.getInvSceneConfigDtlList());
    }

    private void validateParamBySave(InvSceneConfigSaveParam param) {
        param.validateHeadBySave(param);
        param.validateBodyBySave(param.getInvSceneConfigDtlList());
    }
    @Override
    public void del(Long id) {
        Optional<InvSceneConfigDO> invSceneConfigDOOptional = invSceneConfigService.findById(id);

        if (invSceneConfigDOOptional.isPresent()) {
            InvSceneConfigDO dbSceneConfig = invSceneConfigDOOptional.get();
            invSceneConfigDtlDomainService.deleteByCondition(getQueryBeanByMasId(id));
            invSceneConfigService.deleteById(id);
            removeCache(dbSceneConfig.getSceneCode());
        }
    }

    private InvSceneConfigDtlDO getQueryBeanByMasId(Long id) {
        InvSceneConfigDtlDO whereBean = new InvSceneConfigDtlDO();
        whereBean.setMasId(id);
        return whereBean;
    }

    @Override
    @SysCodeProc
    public InvSceneConfigQueryVO detail(Long masId) {
        Optional<InvSceneConfigDO> dbConfig = invSceneConfigRepo.findById(masId);
        if(! dbConfig.isPresent()){
            return null;
        }
        InvSceneConfigQueryVO queryVO = InvSceneConfigConvert.INSTANCE.do2VO(dbConfig.get());
        Example<InvSceneConfigDtlDO> example = Example.of(getQueryBeanByMasId(masId));
        List<InvSceneConfigDtlDO> all = invSceneConfigDtlDomainService.findAll(example);
        List<InvSceneConfigQueryVO.InvSceneConfigDtl> invSceneConfigDtls = InvSceneConfigDtlConvert.INSTANCE.do2VOList(all);
        queryVO.setInvSceneConfigDtlList(invSceneConfigDtls);
        convertDtls(invSceneConfigDtls);
        return queryVO;
    }
    private void removeCache(String sceneCode){
        RBucket<InvSceneConfigQueryVO> bucket = redissonClient.getBucket(InvRedisConstant.SCENE_CONFIG_KEY + sceneCode);
        if(bucket.isExists()){
            bucket.delete();
        }
    }
    @Override
    public InvSceneConfigDTO getBySceneCode(String sceneCode) {
        RBucket<InvSceneConfigDTO> bucket = redissonClient.getBucket(InvRedisConstant.SCENE_CONFIG_KEY + sceneCode);
        if(bucket.isExists()){
            return bucket.get();
        }
        var dbConfig = invSceneConfigService.findBySceneCode(sceneCode);
        if(dbConfig != null){
            Example<InvSceneConfigDtlDO> dtlExample = Example.of(getQueryBeanByMasId(dbConfig.getId()));
            List<InvSceneConfigDtlDO> all = invSceneConfigDtlDomainService.findAll(dtlExample);
            List<InvSceneConfigDtlDTO> invSceneConfigDtls = InvSceneConfigDtlConvert.INSTANCE.do2DTOList(all);
            dbConfig.setInvSceneConfigDtlList(invSceneConfigDtls);
            bucket.set(dbConfig);
            return dbConfig;
        }
        return null;
    }

    private void convertDtls(List<InvSceneConfigQueryVO.InvSceneConfigDtl> invSceneConfigDtls) {
        List<InvIocDTO> invIocNameList = invIocDomainService.findByCodeBatch(invSceneConfigDtls.stream().map(InvSceneConfigQueryVO.InvSceneConfigDtl::getIoCode).collect(Collectors.toList()));
        List<String> deter2s = invSceneConfigDtls.stream().map(InvSceneConfigQueryVO.InvSceneConfigDtl::getDeter2)
                .filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<InvWhAreaSettingRespVO> whAreaSettingRespVOS = invWhAreaSettingService.findByDeter2Types(deter2s);
        Map<String, String> ioCodeMap = new HashMap<>();
        Map<String, String> ioTypeMap = new HashMap<>();
        Map<String,String> deter2TypeMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(invIocNameList)){
            ioTypeMap = invIocNameList.stream().collect(Collectors.toMap(InvIocDTO::getIoType, InvIocDTO::getIoTypeName, (key1, key2) -> key2));
            ioCodeMap = invIocNameList.stream().collect(Collectors.toMap(InvIocDTO::getIoCode, InvIocDTO::getIoName, (key1, key2) -> key2));
        }
        if(!CollectionUtils.isEmpty(whAreaSettingRespVOS)){
            deter2TypeMap = whAreaSettingRespVOS.stream().collect(Collectors.toMap(InvWhAreaSettingRespVO::getDeter2Type, InvWhAreaSettingRespVO::getDeter2TypeName, (key1, key2) -> key2));
        }
        for (InvSceneConfigQueryVO.InvSceneConfigDtl configDtl : invSceneConfigDtls) {
            if(StringUtils.isNotBlank(configDtl.getIoCode())){
                configDtl.setIoName(ioCodeMap.get(configDtl.getIoCode()));
            }
            if(StringUtils.isNotBlank(configDtl.getIoType())){
                configDtl.setIoTypeName(ioTypeMap.get(configDtl.getIoType()));
            }
            if(StringUtils.isNoneEmpty(configDtl.getDeter2())){
                configDtl.setDeter2Name(deter2TypeMap.get(configDtl.getDeter2()));
            }
        }
    }
    private void convertPage(List<InvSceneConfigPageVO> invSceneConfigDtls) {
        List<InvIocDTO> invIocNameList = invIocDomainService.findByCodeBatch(invSceneConfigDtls.stream().map(InvSceneConfigPageVO::getIoCode).collect(Collectors.toList()));
        Map<String, String> ioCodeMap = new HashMap<>();
        Map<String, String> ioTypeMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(invIocNameList)){
            ioTypeMap = invIocNameList.stream().collect(Collectors.toMap(InvIocDTO::getIoType, i -> i.getIoTypeName() == null ? "" : i.getIoTypeName(), (key1, key2) -> key2));
            ioCodeMap = invIocNameList.stream().collect(Collectors.toMap(InvIocDTO::getIoCode,i -> i.getIoName() == null ? "" : i.getIoName() , (key1, key2) -> key2));
        }
        for (InvSceneConfigPageVO configDtl : invSceneConfigDtls) {
            if(StringUtils.isNotBlank(configDtl.getIoCode())){
                configDtl.setIoName(ioCodeMap.get(configDtl.getIoCode()));
            }
            if(StringUtils.isNotBlank(configDtl.getIoType())){
                configDtl.setIoTypeName(ioTypeMap.get(configDtl.getIoType()));
            }
        }
    }

}
