package com.elitesland.yst.production.inv.domain.service.impl;

import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotCombineParam;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotParam;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotRespVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvStkDParam;
import com.elitesland.yst.production.inv.domain.convert.InvLotConvert;
import com.elitesland.yst.production.inv.domain.entity.lot.InvLot;
import com.elitesland.yst.production.inv.domain.entity.lot.InvLotDO;
import com.elitesland.yst.production.inv.domain.entity.lot.QInvLotDO;
import com.elitesland.yst.production.inv.domain.service.InvLotDomainService;
import com.elitesland.yst.production.inv.infr.dto.InvLotDTO;
import com.elitesland.yst.production.inv.infr.dto.InvLotStkDTO;
import com.elitesland.yst.production.inv.infr.repo.lot.InvLotRepo;
import com.elitesland.yst.production.inv.infr.repo.lot.InvLotRepoProc;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.AllArgsConstructor;
import lombok.val;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2022/04/15
 * @Version 1.0
 * @Content:
 */
@Service
@AllArgsConstructor
public class InvLotDomainServiceImpl implements InvLotDomainService {
    private final InvLotRepo invLotRepo;
    private final InvLotRepoProc invLotRepoProc;

    /**
     * 批次主档批量保存
     *
     * @param list
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<Long> createBatch(List<InvLot> list) {
        return invLotRepo.saveAll(list.stream().map(InvLotConvert.INSTANCE::enToDo)
                .collect(Collectors.toList())).stream()
                .map(InvLotDO::getId).collect(Collectors.toList());
    }

    /**
     * 批次主档保存
     *
     * @param invLot
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createOne(InvLot invLot) {
        return invLotRepo.save(InvLotConvert.INSTANCE.enToDo(invLot)).getId();
    }

    /**
     * 批次主档根据入参查询
     *
     * @param queryParam
     * @return
     */
    @Override
    @SysCodeProc
    public List<InvLotDTO> findByParam(InvLotParam queryParam) {
        List<InvLotDTO> invLotDTOList = invLotRepoProc.select(null).
                where(invLotRepoProc.where(queryParam)).fetch();

        if (CollectionUtils.isEmpty(invLotDTOList)) {
            return Collections.EMPTY_LIST;
        }
        return invLotDTOList;
    }

    /**
     * 批次主档分页查询
     *
     * @param param
     * @return
     */
    @Override
    @SysCodeProc
    public PagingVO<InvLotDTO> invLotSearch(InvLotParam param) {

        val pagingVo = invLotRepoProc.useSelect(param);
        return pagingVo;
    }

    @Override
    @SysCodeProc
    public PagingVO<InvLotStkDTO> useSelectD(InvStkDParam param) {

        val pagingVo = invLotRepoProc.useSelectD(param);
        return pagingVo;
    }

    @Override
    @SysCodeProc
    public PagingVO<InvLotDTO> findCombine(InvLotCombineParam param) {
        Page<InvLotDO> ret = invLotRepo.findAll(invLotRepoProc.searchWhere(param), param.getPageRequest());
        if (CollectionUtils.isEmpty(ret.getContent())) {
            return PagingVO.<InvLotRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        val vos = ret.getContent();
        List<InvLotDTO> invLotDTOList = vos.stream().map(
                InvLotConvert.INSTANCE::doToDto).collect(Collectors.toList());

        return PagingVO.<InvLotDTO>builder()
                .total(ret.getTotalElements())
                .records(invLotDTOList)
                .build();

    }


    /**
     * 根据ID查询 批次主档信息
     *
     * @param id 批次主档ID
     * @return 批次主档对象
     */
    @Override
    @SysCodeProc
    public Optional<InvLotDTO> findById(Long id) {
        if (StringUtils.isEmpty(id)) {
            return Optional.empty();
        }
        return invLotRepo.findById(id).map(InvLotConvert.INSTANCE::doToDto);
    }

    /**
     * 根据ID批量查询 批次主档信息
     *
     * @param ids 批次主档ID
     * @return 批次主档对象
     */
    @Override
    @SysCodeProc
    public List<InvLotDTO> findIdBatch(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.EMPTY_LIST;
        }
        val jpaQDo = QInvLotDO.invLotDO;
        JPAQuery<InvLotDTO> jpaQuery = invLotRepoProc.select(null).where(jpaQDo.id.in(ids));
        return jpaQuery.fetch();
    }

    /**
     * 根据商品ID、批次号查询 批次主档信息
     *
     * @param itmId 商品ID
     * @param lotNo 批次号
     * @return 批次主档对象
     */
    @Override
    @SysCodeProc
    public List<InvLotDTO> findAllByItemIdAndLotNo(Long itmId, String lotNo) {
        List<InvLotDO> invLotDOList = invLotRepo.findAllByItemIdAndLotNo(itmId, lotNo);
        if (CollectionUtils.isEmpty(invLotDOList)) {
            return Collections.EMPTY_LIST;
        }

        List<InvLotDTO> invLotDTOList = invLotDOList.stream().map(InvLotConvert.INSTANCE::doToDto).collect(Collectors.toList());

        return invLotDTOList;
    }

    @Override
    @SysCodeProc
    public List<InvLotDTO> findAllByItemIdAndLotNoAndVariId(InvLotParam param) {
        List<InvLotDTO> invLotDTOList = invLotRepoProc.findAllByItemIdAndLotNoAndVariId(param).fetch();
        if (CollectionUtils.isEmpty(invLotDTOList)) {
            return Collections.EMPTY_LIST;
        }

        return invLotDTOList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagById(Integer deleteFlag, Long id) {
        invLotRepoProc.updateDeleteFlagById(deleteFlag, id).execute();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteById(Long id) {
        invLotRepo.deleteById(id);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addSinQty(String lotNo, Long itemId, Double sinQty, String sinStatus, String sinNo){
        invLotRepo.addSinQty(lotNo, itemId, sinQty, sinStatus, sinNo);
    }

    @Override
    public List<InvLotDO> findByQueryParam(List<String> queryParam) {
        return invLotRepo.findByConcatKey(queryParam);
    }

}
